
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAccelerometer;
import org.lsst.ccs.subsystems.fcs.common.AcceleroInterface;

import java.util.logging.Logger;

/**
 * A class for carousel accelerometer.
 *
 * Each PDO includes a 16-bit trigger * counter (unsigned), followed by three 16-bit (signed) pieces of data. For the
 * first PDO, this is the angular velocity along the three axes, with a
 * conversion factor of 0.008 dps/LSB. The second PDO encodes the acceleration
 * along the three axes (gravity if there is no movement), with a conversion
 * factor of 0.2 mG/LSB.
 *
 * @author virieux
 */
public class CanOpenMG550P extends CanOpenDevice implements AcceleroInterface {
    private static final Logger FCSLOG = Logger.getLogger(CanOpenMG550P.class.getName());

    /**
     * define cobid to be able to read values from PDO data.
     */
    protected int cobid1;
    protected int cobid2;

    /**
     * PDO1 represents : 16-bit trigger counter (unsigned), three 16-bit
     * (signed) pieces of data : angular velocity along the three axes, with a
     * conversion factor of 0.008 dps/LSB.
     *
     */
    private long pdo1 = 0L;

    /**
     * PDO2 represents : 16-bit trigger counter (unsigned), three 16-bit
     * (signed) pieces of data : accelerations along the three axes (gravity if
     * there is no movement), with a conversion factor of 0.2 mG/LSB.
     */
    private long pdo2 = 0L;

    public final static double ANGULARVELOCITY_CF = 0.008;

    public final static double ACCELERATION_CF = 0.2;

    /**
     * 3 angular velocities along each axe.
     */
    private double[] angularVelocities = new double[3];

    private double[] accelerations = new double[3];

    private double gravity = 1;

    private double elevation = 1;

    private double azimut = 0;

    private double airmass = 1;

    public double getAirmass() {
        return airmass;
    }

    public double[] getAngularVelocities() {
        return angularVelocities;
    }

    public double[] getAccelerations() {
        return accelerations;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAccelerometer.class, path);
    }

    @Override
    public void init() {
        cobid1 = nodeID + 0x180;
        cobid2 = nodeID + 0x280;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        FCSLOG.finer(() -> name + " updatingFromPDO = " + pdo);
        boolean updated = false;
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
            updated = true;
            /* update from PDO1 */
            FCSLOG.finer(() -> name + " updatingFromPDO1 = " + pdo1 + " binary:" + Long.toBinaryString(pdo1));
            for (int i = 0; i < 3; i++) {
                angularVelocities[i] = extractValue(pdo1, i) * ANGULARVELOCITY_CF;
            }
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
            updated = true;
            /* update from PDO2 */
            FCSLOG.finer(() -> name + " updatingFromPDO2 = " + pdo2 + " binary:" + Long.toBinaryString(pdo2));
            for (int i = 0; i < 3; i++) {
                accelerations[i] = extractValue(pdo2, i) * ACCELERATION_CF;
            }
            computeAirmass();
        }
        if (updated) {
            this.publishDataAccelero();
        }
    }

    public double computeGravity() {
        double sq = 0;
        for (int i = 0; i < 3; i++) {
            sq = sq + Math.pow(accelerations[i], 2);
        }
        return Math.sqrt(sq) / 1000;
    }

    /**
     * if x == 0 then azimut = 90 , else azimut= atan_deg( abs(y/x) ) if x < 0
     * then azimut = 180 - az if y < 0 then azimut = 360 - az
     *
     * @param x
     * @param y
     * @return azimut compute from x and y.
     */
    private double computeAzimut(double x, double y) {
        double az;
        if (x == 0) {
            az = 90;
        } else {
            az = Math.toDegrees(Math.atan(Math.abs(y / x)));
        }
        if (x < 0) {
            az = 180 - az;
        }
        if (y < 0) {
            az = 360 - az;
        }
        return az;
    }

    public void computeAirmass() {
        gravity = computeGravity();
        double x = accelerations[0] / 1000;
        double y = accelerations[1] / 1000;
        double z = accelerations[2] / 1000;
        /* compute when g is in the correct range of values - so g != 0*/
        if (gravity >= 0.5 && gravity <= 1.5) {
            elevation = Math.toDegrees(Math.asin(-z / gravity));
            azimut = computeAzimut(x, y);
            if (Math.abs(z) < 0.1) {
                airmass = 100;
            } else {
                airmass = -gravity / z;
            }
        } else {
            FCSLOG.warning(name + " gravity = " + gravity + " => should be between 1.5 and 1.5 default values are taken.");
            gravity = 1;
            elevation = 90;
            azimut = 0;
            airmass = 1;
        }
        FCSLOG.finer(() -> name + " gravity = " + gravity);
        FCSLOG.finer(() -> name + " elevation = " + elevation);
        FCSLOG.finer(() -> name + " azimut = " + azimut);
        FCSLOG.finer(() -> name + " airmass = " + airmass);
    }

    /**
     * For accelerometer return deviceType because serial number is not
     * configured on device.
     *
     * @return
     */
    @Override
    public String printSerialNumber() {
        return Integer.toHexString((int) readSDO(0x1000, 0));
    }


    /**
     * for tests and debug
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "print values updated by PDOs to debug and test")
    public String printAngularVelocities() {
        StringBuilder sb = new StringBuilder("=> pdo1 value = ");
        sb.append(pdo1);
        sb.append(" / pdo1 = 0x").append(Long.toHexString(pdo1));
        sb.append(" / pdo1 = 0b").append(Long.toBinaryString(pdo1));
        for (int i = 0; i < 3; i++) {
            sb.append("\n angularVelocity axe i=").append(i).append(" => ");;
            sb.append(angularVelocities[i]);
        }
        return sb.toString();
    }

    /**
     * for tests and debug
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "print values updated by PDOs to debug and test")
    public String printAccelerations() {
        StringBuilder sb = new StringBuilder("=> pdo2 value = ");
        sb.append(pdo2);
        sb.append(" = 0x").append(Long.toHexString(pdo2));
        sb.append(" = 0b").append(Long.toBinaryString(pdo2));
        for (int i = 0; i < 3; i++) {
            sb.append("\n acceleration axe i=").append(i).append(" => ");
            sb.append(accelerations[i]);
        }
        return sb.toString();
    }

    /**
     * Extract angular velocity or acceleration for one axe. Value returned is
     * coded on 2 bytes and SIGNED.
     *
     * @param pdo
     * @param axe
     * @return
     */
    public static short extractValue(long pdo, int axe) {
        switch (axe) {
            case 0:
                return (short) ((pdo >> 16) & 0xFFFF);
            case 1:
                return (short) ((pdo >> 32) & 0xFFFF);
            case 2:
                return (short) ((pdo >> 48) & 0xFFFF);
            default:
                assert false : axe;
                return 0;
        }
    }

    public void publishDataAccelero() {
        KeyValueData kvd = new KeyValueData(path, createStatusDataPublishedByAccelerometer());
        subs.publishSubsystemDataOnStatusBus(kvd);
    }

    public StatusDataPublishedByAccelerometer createStatusDataPublishedByAccelerometer() {
        StatusDataPublishedByAccelerometer status = new StatusDataPublishedByAccelerometer(isBooted(), isInitialized());
        status.setAccelerationX(accelerations[0]);
        status.setAccelerationY(accelerations[1]);
        status.setAccelerationZ(accelerations[2]);
        status.setAngularVelocityX(angularVelocities[0]);
        status.setAngularVelocityY(angularVelocities[1]);
        status.setAngularVelocityZ(angularVelocities[2]);
        status.setAirmass(airmass);
        status.setAzimut(azimut);
        status.setElevation(elevation);
        status.setGravity(gravity);
        return status;
    }

}
