package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput4;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.LOCK_ONLINECLAMP;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMP_MODE_PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.REOPEN_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_ONLINECLAMP;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForOnlineClamp;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClamp extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerOnlineClamp.class.getName());
    public static final String ONLINE_CLAMP_CONFIG_CATEGORY = "onlineClampCurrent";
    public static final String CURRENT_MONITOR_TASK_NAME = "-monitorCurrent";

    // currently the Command.timeout does not take non constant values as input so
    // the use of this final value is favored
    public static final int TIMEOUT_FOR_CLOSING = 6000;
    // Timeout bigger than for closing because homing can be included in the action
    public static final int TIMEOUT_FOR_OPENING = 15000;

    @LookupField(strategy = TREE)
    private Autochanger autochanger;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private final EPOSControllerForOnlineClamp controller;

    /* couple of closeClampInCurrentMode sensors */
    private final ComplementarySensors closeSensors;

    /* couple of openClampInCurrentMode sensors */
    private final ComplementarySensors openSensors;

    @LookupField(strategy = TREE)
    private AgentPeriodicTaskService periodicTaskService;

    /**
     * currentToClamp is different for each clamp so the value is initialized from
     * configuration system.
     */
    @ConfigurationParameter(range = "-7000..7000", description = "Current to clamp ONLINE clamp", units = "mA", category = "autochanger")
    private volatile int currentToClamp = 0;

    @ConfigurationParameter(range = "-2000..2000", description = "Current to open ONLINE clamp", units = "mA", category = "autochanger")
    private volatile int currentToOpen = 0;

    /**
     * sentCurrent represents the last current value sent to controller we want to
     * save this value of current between two instances of running FCS and each time
     * we sent current to the controller.
     *
     */
    @Persist
    private volatile int sentCurrent = 0;

    /**
     * position read on controller
     */
    private int position;

    /**
     * increment of current in a current ramp, in mA
     */
    @ConfigurationParameter(range = "0..2000", description = "Increment of current to lock in a current ramp", units = "mA", category = "autochanger")
    private volatile int incrementCurrentToClamp = 250;

    /**
     * increment of current to closeClampInCurrentMode a clamp in a current ramp, in
     * mA
     */
    @ConfigurationParameter(range = "0..1000", description = "Increment of current to close clamp in a current ramp", units = "mA", category = "autochanger")
    private volatile int incrementCurrentToClose = 250;

    /**
     * increment of current to openClampInCurrentMode a clamp in a current ramp, in
     * mA
     */
    @ConfigurationParameter(range = "0..1000", description = "Increment of current to open clamp in a current ramp", units = "mA", category = "autochanger")
    private volatile int incrementCurrentToOpen = 250;

    /**
     * initial current to close a clamp in CURRENT mode in a * current ramp, in mA
     */
    @ConfigurationParameter(range = "-3000..3000", description = "Initial current to close clamp in a current ramp", units = "mA", category = "autochanger")
    private volatile int initialCurrentToClose = -currentToOpen;

    /**
     * final current to close a clamp In Current Mode a clamp in a * current ramp,
     * in mA
     */
    @ConfigurationParameter(range = "-3000..3000", description = "Final current to close clamp in a current ramp", units = "mA", category = "autochanger")
    private volatile int finalCurrentToClose = -currentToOpen * 2;

    /**
     * For the onlineClamp the lockStatus is OPENED, CLOSED, or LOCKED. Sensors say
     * if clamp is CLOSED or OPENED. To know if a clamp is LOCKED, we must know the
     * value of the current that was sent to the controller to
     * closeClampInCurrentMode the clamp. (field sentCurrent).
     */
    private volatile FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    private volatile boolean initialized;
    // During tests we don't change configuration of clamps controller.
    private boolean controllerConfigured = true;

    /* Tools for current ramp */
    private ScheduledFuture<?> currentRampHandle;

    /**
     * ******************************************** parameters for FINAL PRODUCTS
     * only *********************************************
     */
    /**
     * position when clamps is opened
     */
    @ConfigurationParameter(range = "-15000..15000", description = "Absolute target position to open clamp", units = "unitless", category = "autochanger")
    private volatile int targetPositionToOpen;

    /**
     * position when clamps is closed
     */
    @ConfigurationParameter(range = "-150000..150000", description = "Absolute target position to close clamp", units = "unitless", category = "autochanger")
    private volatile int targetPositionToClose;

    @ConfigurationParameter(range = "0..500", description = "Delta of current between target current "
            + "and current read on controller. Used to know if action is completed for homing.", units = "mA", category = "autochanger")
    private volatile int deltaCurrent = 50;

    /**
     * ******************************************** end of parameters for FINAL
     * PRODUCTS only *********************************************
     */
    /**
     * Builds an AutochangerOnlineClampModule with a controller, 4 sensors and 2
     * maps of parameters for the controller.
     *
     * @param controller
     * @param closeSensors
     * @param openSensors
     */
    public AutochangerOnlineClamp(EPOSControllerForOnlineClamp controller, ComplementarySensors closeSensors,
            ComplementarySensors openSensors) {
        this.controller = controller;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
    }

    /* for simulation and tests */
    public void setCurrentToClamp(int currentToClamp) {
        this.currentToClamp = currentToClamp;
    }

    /* for simulation & tests */
    public void setCurrentToOpen(int currentToOpen) {
        this.currentToOpen = currentToOpen;
    }

    public void setInitialCurrentToClose(int initialCurrentToClose) {
        this.initialCurrentToClose = initialCurrentToClose;
    }

    public void setFinalCurrentToClose(int finalCurrentToClose) {
        this.finalCurrentToClose = finalCurrentToClose;
    }

    /**
     *
     * @param sentCurrent
     */
    public void setSentCurrent(int sentCurrent) {
        this.sentCurrent = sentCurrent;
    }

    /* for simulation and tests */
    public void setTargetPositionToOpen(int targetPositionToOpen) {
        this.targetPositionToOpen = targetPositionToOpen;
    }

    /* for simulation and tests */
    public void setTargetPositionToClose(int targetPositionToClose) {
        this.targetPositionToClose = targetPositionToClose;
    }

    /**
     *
     * @param stepHeightAbsValue
     */
    @ConfigurationParameterChanger
    public void setIncrementCurrentToClamp(int stepHeightAbsValue) {
        FcsUtils.checkPositive(stepHeightAbsValue);
        this.incrementCurrentToClamp = stepHeightAbsValue;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToOpen(int incrementCurrentToOpen) {
        FcsUtils.checkPositive(incrementCurrentToOpen);
        this.incrementCurrentToOpen = incrementCurrentToOpen;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToClose(int incrementCurrentToClose) {
        FcsUtils.checkPositive(incrementCurrentToClose);
        this.incrementCurrentToClose = incrementCurrentToClose;
    }

    public int getInitialCurrentToClose() {
        return initialCurrentToClose;
    }

    public int getFinalCurrentToClose() {
        return finalCurrentToClose;
    }

    public int getIncrementCurrentToOpen() {
        return incrementCurrentToOpen;
    }

    public int getIncrementCurrentToClamp() {
        return incrementCurrentToClamp;
    }

    public int getIncrementCurrentToClose() {
        return incrementCurrentToClose;
    }

    /**
     * returns last value of current sent to controller
     *
     * @return
     */
    public int getSentCurrent() {
        return sentCurrent;
    }

    /**
     * Return ONLINE clamp controller
     *
     * @return
     */
    @Override
    public EPOSControllerForOnlineClamp getController() {
        return controller;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * returns closeSensors for tests
     *
     * @return
     */
    public ComplementarySensors getCloseSensors() {
        return closeSensors;
    }

    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }

    /**
     * returns openSensors for tests
     *
     * @return
     */
    public ComplementarySensors getOpenSensors() {
        return openSensors;
    }

    /**
     * Returns true if closeSensor and closeSensorC return the same value. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if lockSensor and lockSensorC return the same value. "
            + "Doesn't read again sensors.")
    public boolean isCloseSensorsInError() {
        return closeSensors.isInError();
    }

    /**
     * Returns true if openSensor and openSensorC return the same value. Doesn't read
     * again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if unlockSensor and unlockSensorC return the same value. "
            + "Doesn't read again sensors.")
    public boolean isOpenSensorsInError() {
        return openSensors.isInError();
    }

    /**
     * Returns true if LockStatus=ERROR, this means closeSensor or openSensor is in
     * ERROR or openSensor and closeSensor return non consistant values. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if in ERROR which means that "
            + "closeSensor or openSensor is in ERROR or that"
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return lockStatus == ERROR;
    }

    /**
     * Returns true if LockStatus=CLOSED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if CLOSED. Doesn't read again sensors.")
    public boolean isClosed() {
        return lockStatus == CLOSED;
    }

    /**
     *
     * @return true if homing of controller has been done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if homing of controller has been done. Does not read the controller.")
    public boolean isHomingDone() {
        return controller.isHomingDone();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Reads the homing status on the controller, True means that the homing as been done and is still in memory.")
    public boolean checkHomingStatusOnController() {
        return controller.checkHomingStatusOnController();
    }

    public void updateHomingFromControllerStatusWord() {
        controller.updateHomingFromControllerStatusWord();
    }

    /**
     * Returns true if LockStatus=LOCKED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if LOCKED. Doesn't read again sensors.")
    public boolean isLocked() {
        return lockStatus == LOCKED;
    }

    /**
     * Returns true if LockStatus=OPENED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if OPENED. Doesn't read again sensors.")
    public boolean isOpened() {
        return lockStatus == OPENED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == INTRAVEL;
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least booted)
     * and clamp controller is initialized (parameters in the controller CPU have
     * been opened and controller has been configured).
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if clamp is initialized : controller is booted and parameters in the controller "
            + "CPU have been opened and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(HARDWARE_ERROR.getAlert(controller.getName()), alwaysClear);
        alertService.registerAlert(SDO_ERROR.getAlert(controller.getName()), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);

    }

    /**
     *
     */
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerOnlineClamp.class, path);
        for (MobileItemAction action : new MobileItemAction[]{
            CLOSE_ONLINECLAMP_MODE_CURRENT,
            CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION,
            LOCK_ONLINECLAMP,
            OPEN_ONLINECLAMP_MODE_CURRENT,
            OPEN_ONLINECLAMP_MODE_PROFILE_POSITION,
            REOPEN_ONLINECLAMP_MODE_CURRENT,
            UNLOCK_ONLINECLAMP}) {
            registerAction(action);
        }

        /**
         * Monitor current read on controller.
         */
        // periodicTaskService
        // .scheduleAgentPeriodicTask(new AgentPeriodicTask(name +
        // CURRENT_MONITOR_TASK_NAME, this::monitorCurrent)
        // .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofSeconds(60)));

    }

    /**
     * *** end of lifecycle methods
     * *************************************************
     */
    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller is
     * initialized and configured.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if controller is initialized and configured.")
    @Override
    public boolean myDevicesReady() {
        return controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to openClampInCurrentMode the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the onlineClamp can be opened.")
    public void checkConditionsForOpening() {
        autochanger.checkConditionsForActioningOnlineClamps();

        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(name + " can't be OPENED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to closeClampInCurrentMode the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the onlineClamp can be closed.")
    public void checkConditionsForClosing() {
        autochanger.checkConditionsForActioningOnlineClamps();
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT mode
     * if it's not in this mode and it writes in the CPU of the controller the
     * values of configuration for CURRENT mode parameters.
     */
    @Override
    public void postStart() {
        FCSLOG.fine(() -> name + " BEGIN postStart.");
        if (controller.isBooted()) {
            initializeController();
            controller.updateEposState();

            updatePosition();
            checkHomingAtStartup();
        } else {
            FCSLOG.severe(name + " could not initialize controller in postStart. Do it by hand.");
        }
        FCSLOG.fine(() -> name + " END postStart.");
    }

    /**
     * check that controllers are correctly configured.
     */
    private void initializeController() {
        try {
            /*
             * check that parameters on CPU are those on configuration
             */
            controller.initializeAndCheckHardware();
            controller.updateHomingFromControllerStatusWord();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", controller.getName(), ex);
        }
    }

    /**
     * Enables controller, change mode depending on action and checkFault.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Deprecated // get rid of this method to be more readable
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Change mode depending of action to be done, then enable and release brake.")
    public void enableAndCheckControllerBeforeAction(MobileItemAction action) {
        // changeMode has to be done BEFORE enableAndReleaseBrake see EPOS doc p 103
        // §8.4 "Profile Position Mode"
        // https://filterchangerdoc.pages.in2p3.fr/fcs-doc/troubleshooting/EPOS-documentation/EPOS2_Application_Notes_Collection_En.pdf
        if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION || action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION) {
            controller.changeMode(PROFILE_POSITION);
        } else {
            controller.changeMode(CURRENT);
        }
        // enable and release brake before motion
        controller.enableAndReleaseBrake();
        controller.checkFault();
    }

    public void checkControllerWhenGoingFromEngineeringModeToNormal() {
        controller.checkParameters(CURRENT);

        if (!controller.isParametersOK()) {
            String msg = name + " Some parameter values are not" + " the same in CPU and configuration system.";
            FCSLOG.severe(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Configure ONLINE clamp controller.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Configure controller.")
    public void configureController() {
        try {
            controllerConfigured = false;
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (SDORequestException ex) {
            FCSLOG.log(Level.WARNING, name, ex);
        }
    }

    /**
     * Update position. Do not publish here.
     */
    public void updatePosition() {
        /* as online clamp position is read by PDO in class CanOpenEPOSWithPDO no need to read by SDO*/
        this.position = controller.getPosition();
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in
     * error.
     *
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Update state from sensors values.")
    public synchronized void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-acClamp")) {
            boolean closed = this.closeSensors.isOn();
            boolean opened = this.openSensors.isOn();
            boolean someSensorsInError = closeSensors.isInError() || openSensors.isInError();
            boolean inError = someSensorsInError || (closed && opened);

            if (inError) {
                lockStatus = ERROR;
            } else if (closed) {
                lockStatus = CLOSED;
            } else if (opened) {
                lockStatus = OPENED;
            } else {
                lockStatus = INTRAVEL;
            }
            /*
             * if the clamp is closed and a currentToClamp has been previously sent to
             * controller then clamp is LOCKED
             */
            if (closed && this.sentCurrent == this.currentToClamp) {
                lockStatus = LOCKED;
            }
            updatePosition();
            this.publishData();
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
        /* when reading from PDO*/
        /* The following lines will work if the PDO update was successful */
        controller.publishDataDuringMotion();
        position = controller.getPosition();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == UNLOCK_ONLINECLAMP) {
            return sentCurrent == finalCurrentToClose;

        } else if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
            // TODO REVIEW AC risque de ne jamais arriver à la position demandée
            // delta de isAtPosition à 500 alors que la position à atteindre est ~ 100 000
            // delta à augmenter ou contrainte à revoir
            return isClosed() && isAtPosition(targetPositionToClose);

        } else if (action == REOPEN_ONLINECLAMP_MODE_CURRENT || action == OPEN_ONLINECLAMP_MODE_CURRENT) {
            /* for homing */
            return isOpened();

        } else if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION) {
            return isOpened() && isAtPosition(targetPositionToOpen);

        } else if (action == CLOSE_ONLINECLAMP_MODE_CURRENT) {
            return isClosed();

        } else if (action == LOCK_ONLINECLAMP) {
            return isLocked();

        } else {
            throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
        }
    }

    /**
     * return true if clamp position is around targetPositionToClose
     *
     * @return
     */
    public boolean isAtPositionClose() {
        return isAtPosition(targetPositionToClose);
    }

    /**
     * return true if clamp position is around targetPositionToOpen
     *
     * @return
     */
    public boolean isAtPositionOpen() {
        return isAtPosition(targetPositionToOpen);
    }

    /**
     * return true if position of online clamp is around pos given as argument
     *
     * @param pos
     * @return
     */
    public boolean isAtPosition(int pos) {
        return Math.abs(position - pos) < 500;
    }

    /**
     * check with clamp controller if position is loosely around
     * the closed or the locked position
     * needed for determining if homing is done at startup
     *
     * @return true if clamp is closed or locked
     */
    public boolean isAtPositionClosedOrLocked() {
        return Math.abs(position - targetPositionToClose) < 15000;
    }

    /**
     * Closes the ONLINE clamp. For Proto only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Closes the ONLINE clamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void closeClampInCurrentMode() {
        updateStateAndCheckSensors();
        if (this.isOpened() || isInTravel()) {
            checkConditionsForClosing();
            this.executeAction(CLOSE_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_CLOSING);
            checkCloseSensors();
            updateStateAndCheckSensors();

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a close action.");
        }
    }

    /**
     * Closes the ONLINE clamp in mode PROFILE_POSITION.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Closes the ONLINE clamp in mode PROFILE_POSITION.", timeout = TIMEOUT_FOR_CLOSING)
    public void close() {
        try (AutoTimed at = new AutoTimed("clamp-close")) {
            if (!controller.isHomingDone()) {
                throw new RejectedCommandException(name + " homing not done yet. Can't close.");
            }
            updateStateAndCheckSensors();
            if (this.isOpened() || isInTravel()) {
                checkConditionsForClosing();
                this.executeAction(CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION, TIMEOUT_FOR_CLOSING);
                checkCloseSensors();
                updateStateAndCheckSensors();

            } else if (this.isClosed()) {
                FCSLOG.info(name + " is already CLOSED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be OPENED before a close action.");
            }
        }
    }

    private void checkCloseSensors() {
        if (isClosed()) {
            FCSLOG.info(name + " is CLOSED");
        } else {
            throw new FailedCommandException(name + " is not CLOSED after close or unlock command.");
        }
    }

    private void checkOpenSensors() {
        if (isOpened()) {
            FCSLOG.info(name + " is OPEN");
        } else {
            throw new FailedCommandException(name + " is not OPEN after open or openAndHoming command."
                    + " current action : " + currentAction.toString());
        }
    }

    /**
     * Opens the ONLINE clamp.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Opens the ONLINE clamp.", timeout = TIMEOUT_FOR_OPENING)
    public void openClampInCurrentMode() {
        try (AutoTimed at = new AutoTimed("openClampInCurrentMode")) {
            updateStateAndCheckSensors();
            if (this.isClosed()) {
                checkConditionsForOpening();
                this.executeAction(OPEN_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_OPENING);

            } else if (this.isOpened() || this.isInTravel()) {
                FCSLOG.info(name + " is already OPENED. Send again currentToOpen.");
                this.executeAction(REOPEN_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_OPENING);

            } else {
                throw new RejectedCommandException(name + " has to be CLOSED before an open action.");
            }
            controller.checkFault();
            checkOpenSensors();
            updateStateAndCheckSensors();
        }
    }

    /**
     * openClampInCurrentMode clamp in mode PROFILE_POSITION condition : homing must
     * have been done before, and latches must be closed. for FINAL products AC1 and
     * AC2.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Opens the ONLINE clamp in PROFILE_POSITION.", timeout = TIMEOUT_FOR_OPENING)
    public void open() {
        try (AutoTimed at = new AutoTimed("clamp-open")) {
            if (!controller.isHomingDone()) {
                throw new RejectedCommandException(name + " homing not done yet. Use openInCurrentModeAndHoming.");
            }
            updateStateAndCheckSensors();
            if (this.isClosed() || this.isInTravel()) {
                checkConditionsForOpening();
                this.executeAction(OPEN_ONLINECLAMP_MODE_PROFILE_POSITION, TIMEOUT_FOR_OPENING);
                checkOpenSensors();

                updateStateAndCheckSensors();

            } else if (this.isOpened()) {
                FCSLOG.info(name + " is already OPENED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be CLOSED before an open action.");
            }
        }
    }

    /**
     * openClampInCurrentMode clamp in CURRENT mode and do homing.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "open clamp in CURRENT mode and do homing", timeout = 5000)
    public void openInCurrentModeAndHoming() {
        try (AutoTimed at = new AutoTimed("openInCurrentModeAndHoming")) {
            openClampInCurrentMode();
            controller.homing();
            updatePosition();
            /* Request from Aurelien to release the stress on the clamps after homing
            * by sending them to the targetPositionToOpen
            */
            this.executeAction(OPEN_ONLINECLAMP_MODE_PROFILE_POSITION, TIMEOUT_FOR_OPENING);
            updatePosition();
            publishData();
        }
    }


    /**
	 * Check if clamp homing is done at startup.
	 * If controller position is 0, the homing was indeed lost and needs to be redone.
     * But if the position is around that of CLOSED or LOCKED position and the CLOSED
     * sensors indicate the filter presence, then set homing as done.
	 */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Check whether homing is already done at FCS startup")
    public void checkHomingAtStartup() {
        if (closeSensors.isOn() && isAtPositionClosedOrLocked()) {
            controller.setHomingDone();
            publishData();
        }
    }

    /**
     * Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp
     * of current from currentToClose to currentToRamp. The clamp has to be CLOSED *
     * before this action. At the end of this action, the clamp is LOCKED.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp of current"
            + "from currentToClose to currentToRamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void lock() {
        try (AutoTimed at = new AutoTimed("clamp-lock")) {
            updateStateAndCheckSensors();
            if (this.isClosed()) {
                this.executeAction(LOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

                updateStateAndCheckSensors();

            } else if (this.isLocked()) {
                FCSLOG.info(name + " is already LOCKED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be CLOSED before lock action.");
            }
        }
    }

    /**
     * Unlocks the ONLINE clamp : sends currentToClamp to the controller with a ramp
     * of current from currentToClamp to currentToClose. The clamp has to be LOCKED
     * * before this action. At the end of this action, the clamp is CLOSED. The
     * goal if this action is to slow down the pressure exerted by the clamp on the
     * filter. If we decrease too quickly the pressure the clamp could be ejected
     * out by elastic return.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Unlocks the ONLINE clamp : sends currentToClamp to the controller "
            + " with a ramp of current from currentToClamp to currentToClose.", timeout = TIMEOUT_FOR_OPENING)
    public void unlock() {
        try (AutoTimed at = new AutoTimed("clamp-unlock")) {
            updateStateAndCheckSensors();
            // TODO REVIEW AC check conditions for unlocking
            if (this.isLocked()) {
                this.executeAction(UNLOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);
                checkCloseSensors();
                updateStateAndCheckSensors();

            } else if (this.isClosed()) {
                FCSLOG.info(name + " is already CLOSED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be LOCKED before unlock action.");
            }
        }
    }

    /**
     * Start action of openClampInCurrentMode or closeClampInCurrentMode or lock or
     * unlock.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("AutoOnlineClamp-action-" + action.toString())) {

            autochanger.checkConditionsForActioningOnlineClamps();

            if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
                controller.checkFault();
                controller.changeMode(PROFILE_POSITION);
                controller.enableAndReleaseBrake();
                controller.writeTargetPosition(targetPositionToClose);
                controller.writeControlWord(ControlWord.ABSOLUTE_POSITION_AND_MOVE);

            } else if (action == CLOSE_ONLINECLAMP_MODE_CURRENT) {
                /*
                 * Only used for the autochanger PROTO at LPNHE
                 */
                controller.checkFault();
                controller.changeMode(CURRENT);
                controller.goToOperationEnable();
                controller.doReleaseBrake();
                this.writeCurrentToControllerAndSaveValue(initialCurrentToClose);
                executeCurrentRamp(initialCurrentToClose, finalCurrentToClose, incrementCurrentToClose);

            } else if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION) {
                controller.checkFault();
                controller.changeMode(PROFILE_POSITION);
                controller.enableAndReleaseBrake();
                controller.writeTargetPosition(targetPositionToOpen);
                controller.writeControlWord(ControlWord.ABSOLUTE_POSITION_AND_MOVE);

                /* for homing when isClosed */
            } else if (action == OPEN_ONLINECLAMP_MODE_CURRENT) {
                controller.checkFault();
                controller.changeMode(CURRENT);
                controller.goToOperationEnable();
                writeCurrentToControllerAndSaveValue(finalCurrentToClose);
                FcsUtils.sleep(250, name);
                controller.doReleaseBrake();
                executeCurrentRamp(finalCurrentToClose, currentToOpen, incrementCurrentToOpen);

                /* for homing when opened or inTravel */
            } else if (action == REOPEN_ONLINECLAMP_MODE_CURRENT) {
                controller.checkFault();
                controller.changeMode(CURRENT);
                controller.goToOperationEnable();
                controller.doReleaseBrake();
                writeCurrentToControllerAndSaveValue(currentToOpen);

            } else if (action == LOCK_ONLINECLAMP) {
                controller.checkFault();
                controller.changeMode(CURRENT);
                controller.goToOperationEnable();
                /*
                 * to avoid an elastic reaction we send current, then wait before releasing
                 * brake
                 */
                writeCurrentToControllerAndSaveValue(finalCurrentToClose);
                FcsUtils.sleep(250, name);
                controller.doReleaseBrake();
                /*
                 * write a ramp from finalCurrentToClose to currentToClamp within maxTimeToLock
                 */
                executeCurrentRamp(this.finalCurrentToClose, this.currentToClamp, incrementCurrentToClamp);

            } else if (action == UNLOCK_ONLINECLAMP) {
                /*
                 * the clamp is LOCKED so we send currentToClamp before releasing brake to avoid
                 * an elastic return of the clamp
                 */
                controller.checkFault();
                controller.changeMode(CURRENT);
                controller.goToOperationEnable();
                /*
                 * to avoid an elastic reaction we send current, then wait before releasing
                 * brake
                 */
                writeCurrentToControllerAndSaveValue(this.currentToClamp);
                FcsUtils.sleep(250, name);
                controller.doReleaseBrake();
                /*
                 * write a ramp from currentToClamp to finalCurrentToClose within
                 * maxTimeToUnlock
                 */
                executeCurrentRamp(this.currentToClamp, this.finalCurrentToClose, incrementCurrentToClamp);

            } else {
                throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
            }
        }
    }

    /**
     * Writes current to controller and saves current value in the field
     * sentCurrent. Field sentCurrent is persisted.
     *
     * The goal of saving the value is to be able to restart next time and know
     * which value of current has been sent to controller. To know this value of
     * current is important because the pressure exerted on the filter by the online
     * clamps depends on the value of this current. Higher is the current value,
     * higher is the pressure. And the process of opening or closing of a online
     * clamp depends on this previous sent current value.
     *
     * @param current to send to controller
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "sent current to control and save this value")
    public void writeCurrentToControllerAndSaveValue(int current) {
        FCSLOG.finest(() -> name + " current to write to controller " + getControllerName() + ":" + current);
        controller.writeCurrent((short) current);
        this.sentCurrent = current;
        subs.getAgentPersistenceService().persistNow();
        this.publishData();
    }

    /**
     * What to do to abort an action.
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is ABORTING action " + action.toString() + " within delay " + delay);
        if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION || action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
            controller.stopAction();
            controller.activateBrake();
        } else {
            // autochanger.decreaseCurrentMonitoring();
            controller.activateBrake();
            controller.writeCurrent((short) 0);
            controller.goToSwitchOnDisabled();
        }
    }

    /**
     * Ending action OPEN / CLOSE / CLAMP / UNCLAMP
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " is ENDING action " + action.toString());

        // autochanger.decreaseCurrentMonitoring();

        if (action == OPEN_ONLINECLAMP_MODE_CURRENT || action == REOPEN_ONLINECLAMP_MODE_CURRENT) {
            // in case of homing we have time to wait clamp is completely open
            FCSLOG.info(name + " waiting for clamp to be completely open.");
            FcsUtils.sleep(3000, name);
            controller.activateBrake();
            FcsUtils.sleep(autochanger.getWaitTimeForBrakeOC(), name);
            controller.goToSwitchOnDisabled();

        } else if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION || action == LOCK_ONLINECLAMP || action == CLOSE_ONLINECLAMP_MODE_CURRENT) {
            controller.activateBrake();
            FcsUtils.sleep(autochanger.getWaitTimeForBrakeOC(), name);
            controller.goToSwitchOnDisabled();
        }
        // TODO add else case here
        // mmmm or should we add a case for lock?
        // we need to disable
        // - after open (end of action)
        // - after lock (end of action)
        // - between mode changes
        // it should be done either in lock/open or here...
        FCSLOG.finer(() -> name + " END of action " + action.toString());
        this.publishData();
        controller.checkFault();
    }

    /**
     * This method is used to configure the controllers of the autochanger online
     * clamps. ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital Inputs
     * activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01 Mask=Enabled
     * Polarity=HighActive HE_Sensor General_A Input=02 Mask=Enabled
     * Polarity=HighActive HE_Sensor General_B Input=03 Mask=Enabled
     * Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital Input
     * Functionality index 0x2070 subindex input_ID Functionality tab Value = 15
     * General Purpose A Value = 14 General Purpose B Value = 13 General Purpose C
     * Value = 12 General Purpose D Value = 11 General Purpose E Value = 10 General
     * Purpose F Value = 9 General Purpose G Value = 8 General Purpose H Value = 7
     * General Purpose I Value = 6 General Purpose J Value = 5 QuickStop Value = 4
     * Drive enable Value = 3 Position marker Value = 2 Home switch Value = 1
     * Positive Limit switch Value = 0 Negative Limit switch
     * *************************************************************** 2 - Configure
     * the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0 High Active
     * Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionality state will not be displayed Value = 1 Functionality state
     * will be displayed Digital Input Functionality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose I 0
     * Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0 Bit_3
     * Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0 Bit_0
     * Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04 Digital
     * Input Execution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5 QuickStop 1
     * Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1
     * Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps() {
        // 1-Configure Digital Input Fonctionnality
        // QuickStop (2070,01)
        controller.writeParameter(ConfigurationOfDigitalInput1, 0x5);

        // 2-Configure the Digital Input Polarity (2071,03)
        controller.writeParameter(DigitalInputFonctionnalityPolarity, 0);

        // 3-Configure the Digital Input Mask 8C20:1000110000100000 (2071,02)
        controller.writeParameter(DigitalInputFonctionnalityMask, 0x8C20);

        // 4 - Configure the Digital Input Execution Mask 30:0011 0000 (2071,04)
        controller.writeParameter(DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    /**
     * msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_2
     * General_A,wsdo,%s,2070,02,02,8000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_3
     * General_B,wsdo,%s,2070,02,02,4000\n' % (str(cobID))
     */
    /**
     * This method is used to configure the autochanger onlineClamps EPOS
     * controllers. Digital Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault
     * Output=01 Mask=Enabled Polarity=HighActive General_A Output=0 Mask=Enabled
     * Polarity=HighActive ******************************************* 1 - Configure
     * the Digital Output Functionality index 0x2079 subindex input_ID
     * Functionality tab : Value = 15 General Purpose Out_A Value = 14 General
     * Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General Purpose
     * Out_D Value = 11 General Purpose Out_E Value = 10..8 not used Value = 7..3
     * reserved Value = 2 Holding Brake Value = 1 Position compare Value = 0
     * Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionality Mask index 0x2078 subindex
     * 0x02 Value = 0 functionality not activated Value = 1 functionality activated
     * ********************************************************************************
     * Digital Output Functionality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General
     * Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7..3 reserved
     * 0 Bit_2 Holding Brake 0 (read only) Bit_1 Position Compare (read only) Bit_0
     * Ready/Fault (read only)
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value = 1
     * associated output inverted => LowActive
     * ********************************************************************************
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps() {
        // 1 - Configure the Digital Output Functionality
        // Ready/Fault Active on Device Ready / Inactive on Fault
        controller.writeParameter(ConfigurationOfDigitalOutput1, 0);
        // General_A
        controller.writeParameter(ConfigurationOfDigitalOutput4, 0xF); // pas bon

        // 2 - Configure the Digital Output Functionality Mask 8001=1000 0000 0000 0001
        controller.writeParameter(DigitalOutputFonctionnalityMask, 0x8001);

        // 3 - Configure the Digital Output Functionality Polarity 8001=1000 0000 0000
        // 0001
        controller.writeParameter(DigitalOutputFonctionnalityPolarity, 0);
    }

    /**
     * Creates and returns the object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setLockSensorOn(closeSensors.isOn());
        status.setUnlockSensorOn(openSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(closeSensors.isInError());
        status.setUnlockSensorInError(openSensors.isInError());
        status.setInError(lockStatus == ERROR);
        status.setSentCurrent(sentCurrent);
        status.setHomingDone(controller.isHomingDone());
        return status;
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByOnlineClamp()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.quickStop();
    }

    /**
     * ********************************************************** Methods to write
     * current to the controller with a ramp
     * /************************************************************
     */
    /**
     * Cancel currentReadOnController ramp
     */
    private void cancelCurrentRamp() {
        // since this is called from within the runnable, a mayInterrupt=true seems a
        // bit risky.
        // no need to interrupt a new current application (during a sleep).
        this.currentRampHandle.cancel(false);
        FCSLOG.finer(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    private void waitForEndOfCurrentRamp(int timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF CURRENT RAMP");
        try {
            currentRampHandle.get(timeout, TimeUnit.SECONDS); // will never return, only throw exception
        } catch (CancellationException ex) {
            if (currentRampHandle.isDone()) {
                FCSLOG.info(name + " CURRENT RAMP ENDED");
            } else {
                FCSLOG.log(Level.SEVERE, name + " during action " + this.currentAction
                        + " waiting was cancelled before END of CURRENT RAMP\n", ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.log(Level.SEVERE, name + " CURRENT RAMP IS DURING TOO LONG - timeout=" + timeout + " ", ex);
            cancelCurrentRamp();

        } catch (InterruptedException ex) {
            FCSLOG.log(Level.SEVERE, name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);
            throw new FcsHardwareException(
                    name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);

        } catch (ExecutionException ex) {
            FCSLOG.log(Level.SEVERE, name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction, ex);

        } finally {
            FCSLOG.info(name + " STOP WAITING FOR END OF CURRENT RAMP");
        }
    }

    /**
     * Sends currentReadOnController value to controller with a
     * currentReadOnController ramp from initialValue to finalValue. Uses a
     * scheduler to schedule the task.
     *
     * @param initialValue first value to send to controller
     * @param finalValue   last value to send to controller
     * @param increment    current increment.
     */
    private void writeCurrentRamp(final int initialValue, final int finalValue, final int increment) {
        FCSLOG.finer(() -> {
            StringBuilder message = new StringBuilder("############################\n");
            message.append(name).append("writeCurrentRamp\n");
            message.append("initialValue=").append(initialValue).append("\n");
            message.append("finalValue=").append(finalValue).append("\n");
            message.append("|increment|=").append(increment).append("\n");
            message.append("period=").append(250).append("\n");
            message.append("waitTime=").append(250).append("\n");
            message.append("############################");
            return message.toString();
        });


        final Runnable currentRamp = new Runnable() {
            private final int adjustedStepHeight = FcsUtils.getSignedStepHeight(initialValue, finalValue, increment);
            private int currentValue = initialValue;

            public boolean finalValueReached() {
                boolean reachedUp = adjustedStepHeight > 0 && currentValue >= finalValue;
                boolean reachedDown = adjustedStepHeight < 0 && currentValue <= finalValue;
                return reachedUp || reachedDown;
            }

            @Override
            public void run() {
                // extra safety
                if (!finalValueReached()) {
                    currentValue += adjustedStepHeight;
                    FCSLOG.info("ramp " + name + " write current " + currentValue);
                    writeCurrentToControllerAndSaveValue(currentValue);
                }
                if (finalValueReached()) {
                    cancelCurrentRamp();
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 250, 250, TimeUnit.MILLISECONDS);
    }

    /**
     * Sends currentReadOnController to controller with a ramp : from initialCurrent
     * to finalCurrent in a given number of steps (nbStep) and waits until the final
     * value is reached. It uses method writeCurrentRamp which uses a scheduler.
     *
     * @param initialCurrent   first value to be sent to controller
     * @param finalCurrent     last value to send to controller
     * @param incrementCurrent
     */
    public void executeCurrentRamp(int initialCurrent, int finalCurrent, int incrementCurrent) {
        try (AutoTimed at = new AutoTimed("executeCurrentRamp")) {
            writeCurrentRamp(initialCurrent, finalCurrent, incrementCurrent);
            waitForEndOfCurrentRamp(TIMEOUT_FOR_OPENING);
        }
    }

    /**
     * ************************************************************** end of Methods
     * to write current to the controller with a ramp
     * /****************************************************************
     */


    /**
     * BELOW ARE METHODS TO FAKE CURRENT SENT TO THE CONTROLLER
     * SO THE LOCKED STATUS CAN BE DEFINED MANUALLY BACK AND FORTH
     * THIS WAS USED TO FIX A BUG WHEN THE LOCKED STATUS WAS NOT RECOVERED THROUGH PERSISTANCE
     * AFTER A FCS RESTART DECEMBER 2023
     */

    // TODO : remove ASAP these methods once we find a fix for the loss of this information
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Fake the current sent to the controller for this clamp - MUST BE USED BY FES ENGINEERS ONLY")
    public void setSentCurrentDANGEROUS(int value) {
        FCSLOG.severe("setSentCurrentValueDANGEROUS : fake the current sent to the controller for this clamp to " + value + " mA");
        this.sentCurrent = value;
    }

    // TODO : remove ASAP these methods once we find a fix for the loss of this information
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Fake the LOCKED status of the clamp by patching the sent current - MUST BE USED BY FES ENGINEERS ONLY")
    public void setClampStatusLockedDANGEROUS() {
        FCSLOG.severe("setClampStatusLockedDANGEROUS : fake the LOCKED status of the clamp by patching the sent current");
        this.sentCurrent = currentToClamp;
        updateState();
    }

    // TODO : remove ASAP these methods once we find a fix for the loss of this information
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Fake the CLOSED status of the clamp by patching the sent current - MUST BE USED BY FES ENGINEERS ONLY")
    public void setClampStatusClosedDANGEROUS() {
        FCSLOG.severe("setClampStatusClosedDANGEROUS : fake the CLOSED status of the clamp by patching the sent current");
        this.sentCurrent = finalCurrentToClose;
        updateState();
    }

}
