/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystems.fcs;

import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.Autochanger;
import org.lsst.ccs.subsystems.fcs.AutochangerThreeOnlineClamps;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterManager;
import org.lsst.ccs.subsystems.fcs.Loader;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

public class FcsMain
extends MainModule
implements HasDataProviderInfos {
    private static final long serialVersionUID = 7669526660659959402L;
    private static final Logger FCSLOG = Logger.getLogger(FcsMain.class.getName());
    private static final Duration SETFILTER_MAX_DURATION = Duration.ofSeconds(90L);
    private static final Duration LOAD_UNLOAD_FILTER_MAX_DURATION = Duration.ofSeconds(480L);
    private static final String PUBLICATION_KEY_FOR_MCM = "fcs/mcm";
    @LookupField(strategy=LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Carousel carousel;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Autochanger autochanger;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Loader loader;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private FilterManager filterManager;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AlertService alertService;
    private long setFilterDuration = 0L;
    private int percentage = 0;
    @Persist
    private volatile int previousSocketID = 0;
    private boolean firstMCMpublication = true;
    AtomicLong lastUpdateStateWithSensors = new AtomicLong(0L);

    @Override
    public void build() {
        if (this.bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (this.bridgeToLoader.equals(this.bridge)) {
            throw new RuntimeException("In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();
        KeyValueData data = this.getDataForMcm();
        this.dataProviderDictionaryService.registerData(data);
        String mainPath = data.getKey();
        KeyValueDataList dataList = (KeyValueDataList)data.getValue();
        for (KeyValueData d : dataList) {
            String path = mainPath + "/" + d.getKey();
            DataProviderInfo info = this.dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
            String units = "unitless";
            String description = "";
            switch (d.getKey()) {
                case "filter_on_autochanger_id": {
                    description = "ID of the filter currently on the autochanger";
                    break;
                }
                case "filter_on_autochanger_name": {
                    description = "Name of the filter currently on the autochanger";
                    break;
                }
                case "filter_previous_socketID": {
                    description = "ID of the socket on which the current filter used to sit";
                    break;
                }
                case "autochanger_trucks_position": {
                    description = "Absolute position of the autochanger trucks";
                    units = "micron";
                    break;
                }
                case "autochanger_trucks_location": {
                    description = "Known autochanger position (STANDBY, HANDOFF, ONLINE) or IN_TRAVEL";
                    break;
                }
                case "setFilter_percentage": {
                    description = "Progress of the setFilter command [%]";
                    break;
                }
                case "proximity": {
                    description = "Value of the autochanger proximity sensor";
                    break;
                }
                default: {
                    units = "unitless";
                    description = "";
                }
            }
            info.addAttribute(DataProviderInfo.Attribute.UNITS, units);
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, description);
        }
        this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.UNKNOWN});
    }

    @Override
    public void init() {
        super.init();
        ((AgentPropertiesService)this.subs.getAgentService(AgentPropertiesService.class)).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs", "fcs");
        DataProviderInfo data = new DataProviderInfo("duration/SETFILTER", DataProviderInfo.Type.TRENDING, "duration/SETFILTER");
        data.addAttribute(DataProviderInfo.Attribute.UNITS, "millisecond");
        data.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "duration of command setFilter");
        data.addAttribute(DataProviderInfo.Attribute.TYPE, "long");
        this.dataProviderDictionaryService.addDataProviderInfoToDictionary(data);
        ClearAlertHandler alwaysClear = new ClearAlertHandler(){

            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        this.alertService.registerAlert(FcsEnumerations.FcsAlert.HARDWARE_ERROR.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsEnumerations.FcsAlert.CA_LOCKING_ISSUE.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsEnumerations.FcsAlert.HARDWARE_ERROR.getAlert("carousel"), alwaysClear);
    }

    @Override
    public void updateFCSStateToReady() {
        if (this.carousel.isInitialized() && this.autochanger.isInitialized() && this.loader.isInitialized()) {
            this.updateAgentState(FcsEnumerations.FilterState.READY);
            this.updateAgentState(FcsEnumerations.FilterReadinessState.READY);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        this.updateAgentState(FcsEnumerations.FilterState.READY);
        this.updateAgentState(FcsEnumerations.FilterReadinessState.READY);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Returns the name of the filter which is at ONLINE or NONE if no filter is at ONLINE.")
    public String getMCMOnlineFilterName() {
        if (this.autochanger.isHoldingFilter() && this.autochanger.isAtOnline()) {
            return this.autochanger.getFilterOnTrucksName();
        }
        return "NONE";
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="returns the filter which has for name the String given as argument.")
    public Filter getFilterByName(String filterName) {
        return this.filterManager.getFilterByName(filterName);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return this.autochanger.isLoaderConnected();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        this.loader.disconnectLoaderCANbus();
        this.updateStateWithSensors();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Connect the loader hardware.")
    public void connectLoaderCANbus() {
        this.loader.connectLoaderCANbus();
        this.updateStateWithSensors();
    }

    public void checkFilterName(String aName) {
        this.filterManager.checkFilterName(aName);
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="Return the list of LOADER CANopen hardware that this subsystem manages.")
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="Return the list of names of sensors plugged on Loader pluto gateway.")
    public List<String> listLoSensorsNames() {
        return this.bridgeToLoader.listLoSensorsNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return this.carousel.isFilterOnCarousel(filterID) || this.autochanger.isFilterOnAC(filterID);
    }

    @Command(type=Command.CommandType.ACTION, level=1, description="Store the filter on a given socket of the carousel and move the empty autochanger to HANDOFF position.", autoAck=false, timeout=60000)
    public void storeFilterOnSocket(String socketName) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("storeFilterOnSocket");){
            this.updateStateWithSensors();
            FCSLOG.info(this.name + " === About to store filter on carousel ===");
            if (!FcsEnumerations.FilterReadinessState.READY.equals((Object)this.getFilterReadinessState())) {
                throw new RejectedCommandException(this.name + " FilterReadinessState must be READY");
            }
            CarouselSocket desiredSocket = this.carousel.getSocketByName(socketName);
            if (!this.autochanger.isAtStandby() && this.autochanger.isHoldingFilter() && !desiredSocket.isEmpty()) {
                throw new RejectedCommandException(this.name + " the target carousel socket is not empty");
            }
            if (this.autochanger.isAtOnline()) {
                AutochangerThreeOnlineClamps onlineClamps = this.autochanger.getOnlineClamps();
                if (onlineClamps.isLocked()) {
                    onlineClamps.unlockClamps();
                }
                if (onlineClamps.isClosed()) {
                    if (onlineClamps.isHomingDone()) {
                        onlineClamps.openClamps();
                    } else {
                        onlineClamps.homing();
                    }
                }
                this.updateStateWithSensors();
                if (onlineClamps.isOpened()) {
                    FCSLOG.info(this.name + " === autochanger is free to move ===");
                } else {
                    throw new FcsHardwareException(this.name + " autochanger online clamps should be opened");
                }
            }
            this.carousel.rotateSocketToStandby(socketName);
            this.carousel.checkDeltaPosition();
            FCSLOG.info(this.name + " === carousel is ready to receive a filter at standby ===");
            if (!this.autochanger.isEmpty()) {
                this.autochanger.moveFilterToStandbyPlusDelta(this.carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
                this.updateStateWithSensors();
            }
            if (this.carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                this.carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + this.carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on autochanger = " + this.autochanger.getFilterID());
                FCSLOG.info("==================================");
                this.subs.getAgentPersistenceService().persistNow();
            } else {
                this.recoveryLockingProcess();
                if (this.carousel.isHoldingFilterAtStandby()) {
                    this.carousel.getSocketAtStandby().updateFilterID();
                    this.subs.getAgentPersistenceService().persistNow();
                } else {
                    String msg = this.name + ": carousel should be CLAMPED_ON_FILTER when autochanger is at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsEnumerations.FcsAlert.CA_LOCKING_ISSUE, msg, this.name);
                    throw new FcsHardwareException(msg);
                }
            }
            if (this.autochanger.isAtStandby()) {
                this.autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(this.name + ": is going to moveEmptyFromStandbyToHandoff");
                this.autochanger.moveEmptyFromStandbyToHandoff();
            } else {
                this.autochanger.goToHandOff();
            }
        }
    }

    @Command(type=Command.CommandType.ACTION, level=1, description="Store the filter in the carousel and move the empty autochanger to HANDOFF position. Initial State for autochanger: a filter at HANDOFF or ONLINE. Final State for autochanger : empty at HANDOFF.", autoAck=false, timeout=60000)
    public void storeFilterOnCarousel() {
        block27: {
            try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("storeFilterOnCarousel");){
                this.updateStateWithSensors();
                FCSLOG.info(this.name + " === About to store filter on carousel ===");
                if (!FcsEnumerations.FilterReadinessState.READY.equals((Object)this.getFilterReadinessState())) {
                    throw new RejectedCommandException(this.name + " FilterReadinessState must be READY");
                }
                if (!this.carousel.isReadyToGrabAFilterAtStandby()) {
                    throw new RejectedCommandException(this.name + " carousel should be at standby with an empty and available socket");
                }
                if (!this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                    throw new RejectedCommandException(this.name + " autochanger must be at HANDOFF or at ONLINE");
                }
                if (!this.autochanger.isHoldingFilter()) {
                    throw new RejectedCommandException(this.name + " autochanger must hold the filter");
                }
                this.carousel.checkDeltaPosition();
                if (this.autochanger.isAtOnline()) {
                    if (this.autochanger.getOnlineClamps().isLocked()) {
                        this.autochanger.getOnlineClamps().unlockClamps();
                    }
                    if (this.autochanger.getOnlineClamps().isClosed()) {
                        if (this.autochanger.getOnlineClamps().isHomingDone()) {
                            this.autochanger.getOnlineClamps().openClamps();
                        } else {
                            this.autochanger.getOnlineClamps().homing();
                        }
                    }
                }
                this.updateStateWithSensors();
                if (!this.autochanger.getOnlineClamps().isOpened()) {
                    throw new FcsHardwareException(this.name + " autochanger online clamps should be opened");
                }
                FCSLOG.info(this.name + " === autochanger is free to move ===");
                FCSLOG.info(this.name + " === carousel is ready to receive a filter at standby ===");
                this.autochanger.moveFilterToStandbyPlusDelta(this.carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
                this.updateStateWithSensors();
                if (this.carousel.isHoldingFilterAtStandby()) {
                    FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                    this.carousel.getSocketAtStandby().updateFilterID();
                    FCSLOG.info("filter ID on CAROUSEL at STANDBY " + this.carousel.getSocketAtStandby().getFilterID());
                    FCSLOG.info("filter ID on autochanger = " + this.autochanger.getFilterID());
                    FCSLOG.info("==================================");
                    this.subs.getAgentPersistenceService().persistNow();
                } else {
                    this.recoveryLockingProcess();
                    if (this.carousel.isHoldingFilterAtStandby()) {
                        this.carousel.getSocketAtStandby().updateFilterID();
                        this.subs.getAgentPersistenceService().persistNow();
                    } else {
                        String msg = this.name + ": carousel should be CLAMPED_ON_FILTER when autochanger is at STANDBY with a filter; recovery process didn't work";
                        this.raiseAlarm(FcsEnumerations.FcsAlert.CA_LOCKING_ISSUE, msg, this.name);
                        throw new FcsHardwareException(msg);
                    }
                }
                if (this.autochanger.isAtStandby()) {
                    this.autochanger.waitForProtectionSystemUpdate();
                    FCSLOG.info(this.name + ": is going to moveEmptyFromStandbyToHandoff");
                    this.autochanger.moveEmptyFromStandbyToHandoff();
                    this.previousSocketID = 0;
                    this.publishDataForMcm();
                    break block27;
                }
                throw new FcsHardwareException(this.name + ": autochanger should be at STANDBY after moveFilterToStandby() command");
            }
        }
    }

    private void recoveryLockingProcess() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryLockingProcess");){
            if (this.carousel.getClampXplus().isLocked() && !this.carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, this.name);
                this.carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type=Command.CommandType.ACTION, level=1, autoAck=false, description="Unclamp filter from carousel, move autochanger to approachStandby position and release the carousel clamps", timeout=15000)
    public void disengageFilterFromCarousel() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("disengageFilterFromCarousel");){
            this.updateStateWithSensors();
            if (!FcsEnumerations.FilterReadinessState.READY.equals((Object)this.getFilterReadinessState())) {
                throw new RejectedCommandException(this.name + " FilterReadinessState must be READY");
            }
            if (!this.carousel.isAtStandby()) {
                throw new RejectedCommandException(this.name + "Carousel should be at Standby");
            }
            if (!this.autochanger.isAtStandby()) {
                throw new RejectedCommandException("autochanger should be at Standby Position");
            }
            this.carousel.unlockClamps();
            this.carousel.updateStateWithSensors();
            this.carousel.waitForProtectionSystemUpdate();
            if (!this.carousel.isUnclampedOnFilterAtStandby()) {
                String msg = this.name + "Carousel clamps still locked. Aborting autochanger movement because carousel is still holding the filter. bad state for socketAtStandby: " + this.carousel.getClampsStateAtStandby() + " should be UNCLAMPED_ON_FILTER";
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, msg, this.name);
                throw new FcsHardwareException(this.name + msg);
            }
            this.autochanger.waitForProtectionSystemUpdate();
            this.previousSocketID = this.carousel.getSocketAtStandbyID();
            this.autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();
            this.publishDataForMcm();
            this.updateStateWithSensors();
            if (!this.carousel.isEmptyAtStandby()) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, "Carousel is still seeing the filter", this.name);
                throw new FcsHardwareException(this.name + " aborting autochanger movement because carousel is still seeing filter.");
            }
            FcsUtils.asyncRun(() -> {
                try {
                    this.carousel.releaseClamps();
                }
                catch (Exception e) {
                    this.raiseWarning(FcsEnumerations.FcsAlert.HARDWARE_ERROR, "cannot release carousel clamps", "carousel");
                }
            });
            this.updateStateWithSensors();
        }
    }

    public boolean autochangerNotInTravel() {
        return this.autochanger.isAtHandoff() || this.autochanger.isAtOnline() || this.autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return this.autochanger.getLatches().isClosed() || this.autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return this.autochanger.getOnlineClamps().isLocked() || this.autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !this.autochanger.isAtOnline() || this.autochanger.isEmpty() || this.autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !this.autochanger.isAtStandby() || this.autochanger.isEmpty() || this.autochanger.isHoldingFilter() || this.carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return this.carousel.isHoldingFilterAtStandby() || this.carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return this.autochanger.isAtStandby() || this.autochanger.isEmpty() || this.carousel.isReadyToGrabAFilterAtStandby();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Move filter to ONLINE position.", timeout=180000, autoAck=false)
    public void setFilter(int filterID) {
        this.setFilterAtHandoffOrOnline(filterID, true, false);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Move filter to ONLINE position.", timeout=180000, autoAck=false)
    public void setFilterByName(String filterName) {
        this.subs.helper().precondition(this.filterManager.containsFilterName(filterName), "%s: Unknown filter name : %s", new Object[]{this.name, filterName}).precondition(new Enum[]{FcsEnumerations.FilterReadinessState.READY}).action(() -> this.setFilter(this.filterManager.getFilterID(filterName)));
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Select filter and move to HANDOFF position without going to ONLINE.", timeout=180000, autoAck=false)
    public void setFilterAtHandoff(int filterID) {
        this.setFilterAtHandoffOrOnline(filterID, false, false);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Select filter and move to HANDOFF position without going to ONLINE.", timeout=180000, autoAck=false)
    public void setFilterAtHandoffForLoader(int filterID) {
        this.setFilterAtHandoffOrOnline(filterID, false, true);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Store all filters on carousel and move autochanger trucks to ONLINE.", timeout=180000, autoAck=false)
    public void setNoFilter() {
        this.setNoFilterAtHandoffOrOnline(true);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Store all filters on carousel and move autochanger trucks to HANDOFF.", timeout=180000, autoAck=false)
    public void setNoFilterAtHandoff() {
        this.setNoFilterAtHandoffOrOnline(false);
    }

    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline, boolean forLoader) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter");){
            this.updateStateWithSensors();
            if (this.carousel.isPowerSaveActivated()) {
                this.carousel.powerOn();
                this.carousel.waitUntilCarouselIsAwake();
                this.autochanger.waitForProtectionSystemUpdate();
            }
            this.percentage = 0;
            this.publishDataForMcm();
            this.checkControllers();
            FCSLOG.info(this.name + ": filter to set online:" + filterID);
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", new Object[]{this.name, filterID}).precondition(this.isFilterInCamera(filterID), "%s: filter %s / filter name %s is out of camera", new Object[]{this.name, filterID, this.filterManager.getFilterNameByID(filterID)}).precondition(this.carousel.isAtStandby(), "carousel not stopped at STANDBY position", new Supplier[0]).precondition(this.autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY", new Supplier[0]).precondition(this.latchesOpenOrClosed(), "%s: bad state for autochanger latches - have to be OPENED or CLOSED", new Object[]{this.autochanger.getLatches().getLockStatus()}).precondition(this.onlineClampsOpenOrLocked(), "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtOnlineMustBeLocked(), "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by autochanger or by carousel.", new Supplier[0]).precondition(this.carouselHoldingFilterOrReadyToGrab(), "%s: bad state for carousel : should be holding a filter or ready to receive a filter", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(this.carouselReadyToClampAtStandby(), "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(new Enum[]{FcsEnumerations.FilterReadinessState.READY}).duration(SETFILTER_MAX_DURATION).enterFaultOnException(true).action(() -> {
                AutochangerTwoTrucks.ReleasedACBrakes g;
                long beginTime = System.currentTimeMillis();
                if (this.autochanger.isHoldingFilter()) {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADED});
                    FCSLOG.info(this.name + ": autochanger is holding a filter, filter name= " + this.autochanger.getFilterOnTrucksName());
                } else {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.NO_FILTER});
                    FCSLOG.info(this.name + ": autochanger is not holding filter, filter name on autochanger" + this.autochanger.getFilterOnTrucksName());
                    FCSLOG.info(this.name + ": filterID: " + filterID + " is on socket:" + this.carousel.getFilterSocket(filterID).getId());
                }
                this.carousel.setControllerPositionSensorTypeEncoderSSI();
                if (!this.autochanger.isFilterOnAC(filterID)) {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.UNLOADING});
                    FCSLOG.info(this.name + ": filterID: " + filterID + " is NOT on autochanger");
                    if (this.autochanger.isAtStandby()) {
                        FCSLOG.info(this.name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                        this.autochanger.moveEmptyFromStandbyToHandoff();
                    } else if (this.autochanger.isHoldingFilter()) {
                        FCSLOG.info(this.name + ": autochanger is holding filter: " + this.autochanger.getFilterOnTrucksName());
                        FCSLOG.info(this.name + ": is going to store a filter on carousel");
                        this.storeFilterOnCarousel();
                    }
                    this.percentage = 20;
                    this.publishDataForMcm();
                    if (!this.autochanger.isEmpty() || !this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                        throw new FcsHardwareException(this.name + ": autochanger is not empty at handoff or online");
                    }
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.ROTATING});
                    this.carousel.rotateSocketToStandby(this.carousel.getFilterSocket(filterID).getName());
                    FcsUtils.sleep(100, this.name);
                    this.percentage = 40;
                    this.publishDataForMcm();
                    this.updateStateWithSensors();
                    if (!this.carousel.isAtStandby()) {
                        throw new FcsHardwareException(this.name + ": carousel should be at standby after rotateSocketToStandby command.");
                    }
                    FCSLOG.info(this.name + ": carousel is at standby");
                    if (this.carousel.getFilterIDatStandby() != filterID) {
                        throw new FcsHardwareException(this.name + " filterID at standby is " + this.carousel.getFilterIDatStandby() + " should be: " + filterID);
                    }
                    this.carousel.checkDeltaPosition();
                    this.autochanger.waitForProtectionSystemUpdate();
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADING});
                    this.autochanger.grabFilterAtStandby();
                    this.updateStateWithSensors();
                    this.percentage = 60;
                    this.previousSocketID = this.carousel.getSocketAtStandbyID();
                    this.publishDataForMcm();
                    this.updateStateWithSensors();
                    if (!this.autochanger.isFilterOnAC(filterID)) {
                        throw new FcsHardwareException(this.name + " filter: " + filterID + " should be now on autochanger");
                    }
                }
                if (this.autochanger.isAtStandby()) {
                    g = new AutochangerTwoTrucks.ReleasedACBrakes(this.autochanger.getAutochangerTrucks());
                    Throwable throwable = null;
                    try {
                        this.disengageFilterFromCarousel();
                        this.percentage = 80;
                        this.publishDataForMcm();
                        this.updateStateWithSensors();
                        if (toOnline) {
                            this.autochanger.moveAndClampFilterOnline();
                        }
                        this.moveToHandoffWithHighVelocity(forLoader);
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (g != null) {
                            if (throwable != null) {
                                try {
                                    g.close();
                                }
                                catch (Throwable throwable3) {
                                    throwable.addSuppressed(throwable3);
                                }
                            } else {
                                g.close();
                            }
                        }
                    }
                } else if (toOnline) {
                    if (this.autochanger.isAtHandoff()) {
                        this.autochanger.moveAndClampFilterOnline();
                    } else if (this.autochanger.isAtOnline()) {
                        this.autochanger.lockFilterAtOnline();
                    }
                } else if (this.autochanger.isAtOnline() && this.autochanger.getOnlineClamps().isOpened()) {
                    g = new AutochangerTwoTrucks.ReleasedACBrakes(this.autochanger.getAutochangerTrucks());
                    Throwable throwable = null;
                    try {
                        this.autochanger.getAutochangerTrucks().moveToApproachOnlinePositionWithLowVelocity();
                        this.moveToHandoffWithHighVelocity(forLoader);
                    }
                    catch (Throwable throwable4) {
                        throwable = throwable4;
                        throw throwable4;
                    }
                    finally {
                        if (g != null) {
                            if (throwable != null) {
                                try {
                                    g.close();
                                }
                                catch (Throwable throwable5) {
                                    throwable.addSuppressed(throwable5);
                                }
                            } else {
                                g.close();
                            }
                        }
                    }
                }
                String filterName = this.filterManager.getFilterNameByID(filterID);
                this.updateAgentState(FcsEnumerations.FilterState.valueOf((String)("ONLINE_" + filterName.toUpperCase())));
                this.setFilterDuration = System.currentTimeMillis() - beginTime;
                FCSLOG.info("filter " + filterID + " is " + (toOnline ? "ONLINE" : "HANDOFF") + ". setFilterDuration = " + this.setFilterDuration);
                this.publishData();
                this.subs.publishSubsystemDataOnStatusBus(new KeyValueData("duration/SETFILTER", (Serializable)Long.valueOf(this.setFilterDuration)));
                this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADED});
                this.percentage = 100;
                this.publishDataForMcm();
                if (this.carousel.isPowerSaveAllowed()) {
                    this.carousel.powerSave();
                }
                this.percentage = 0;
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Set filter online but do not close the clamps. This is used for Winter 2023/2024 focal plane tests to avoid activating the autochanger online clamps.", timeout=180000, autoAck=false)
    public void setFilterNoClamping(int filterID) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilterNoClamping");){
            this.updateStateWithSensors();
            this.publishDataForMcm();
            this.checkControllers();
            FCSLOG.info(this.name + ": filter to set online without clamping:" + filterID);
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", new Object[]{this.name, filterID}).precondition(this.isFilterInCamera(filterID), "%s: filter %s / filter name %s is out of camera", new Object[]{this.name, filterID, this.filterManager.getFilterNameByID(filterID)}).precondition(this.carousel.isAtStandby(), "carousel not stopped at STANDBY position", new Supplier[0]).precondition(this.autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY", new Supplier[0]).precondition(this.latchesOpenOrClosed(), "%s: bad state for autochanger latches - have to be OPENED or CLOSED", new Object[]{this.autochanger.getLatches().getLockStatus()}).precondition(this.onlineClampsOpenOrLocked(), "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtOnlineMustBeLocked(), "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by AC or by carousel.", new Supplier[0]).precondition(this.carouselHoldingFilterOrReadyToGrab(), "%s: bad state for carousel : should be holding a filter or ready to receive a filter", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(this.carouselReadyToClampAtStandby(), "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(new Enum[]{FcsEnumerations.FilterReadinessState.READY}).duration(SETFILTER_MAX_DURATION).enterFaultOnException(true).action(() -> {
                long beginTime = System.currentTimeMillis();
                if (this.autochanger.isHoldingFilter()) {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADED});
                    FCSLOG.info(this.name + ": AC is holding a filter, filter name= " + this.autochanger.getFilterOnTrucksName());
                } else {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.NO_FILTER});
                    FCSLOG.info(this.name + ": AC is not holding filter, filter name on AC" + this.autochanger.getFilterOnTrucksName());
                    FCSLOG.info(this.name + ": filterID: " + filterID + " is on socket:" + this.carousel.getFilterSocket(filterID).getId());
                }
                this.carousel.setControllerPositionSensorTypeEncoderSSI();
                if (!this.autochanger.isFilterOnAC(filterID)) {
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.UNLOADING});
                    FCSLOG.info(this.name + ": filterID: " + filterID + " is NOT on AC");
                    if (this.autochanger.isAtStandby()) {
                        FCSLOG.info(this.name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                        this.autochanger.moveEmptyFromStandbyToHandoff();
                    } else if (this.autochanger.isHoldingFilter()) {
                        FCSLOG.info(this.name + ": AC is holding filter: " + this.autochanger.getFilterOnTrucksName());
                        FCSLOG.info(this.name + ": is going to store a filter on carousel");
                        this.storeFilterOnCarousel();
                    }
                    if (!this.autochanger.isEmpty() || !this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                        throw new FcsHardwareException(this.name + ": autochanger is not empty at handoff or online");
                    }
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.ROTATING});
                    this.carousel.rotateSocketToStandby(this.carousel.getFilterSocket(filterID).getName());
                    FcsUtils.sleep(100, this.name);
                    this.updateStateWithSensors();
                    if (!this.carousel.isAtStandby()) {
                        throw new FcsHardwareException(this.name + ": carousel should be at standby after rotateSocketToStandby command.");
                    }
                    FCSLOG.info(this.name + ": carousel is at standby");
                    if (this.carousel.getFilterIDatStandby() != filterID) {
                        throw new FcsHardwareException(this.name + " filterID at standby is " + this.carousel.getFilterIDatStandby() + " should be: " + filterID);
                    }
                    this.carousel.checkDeltaPosition();
                    this.autochanger.waitForProtectionSystemUpdate();
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADING});
                    this.autochanger.grabFilterAtStandby();
                    this.updateStateWithSensors();
                    if (!this.autochanger.isFilterOnAC(filterID)) {
                        throw new FcsHardwareException(this.name + " filter: " + filterID + " should be now on AC");
                    }
                }
                if (!this.autochanger.isAtOnline()) {
                    if (this.autochanger.isAtStandby()) {
                        try (AutochangerTwoTrucks.ReleasedACBrakes g = new AutochangerTwoTrucks.ReleasedACBrakes(this.autochanger.getAutochangerTrucks());){
                            this.disengageFilterFromCarousel();
                            this.updateStateWithSensors();
                            this.autochanger.moveFilterOnlineNoClamping();
                        }
                    } else if (this.autochanger.isAtHandoff()) {
                        this.autochanger.moveFilterOnlineNoClamping();
                    }
                }
                String filterName = this.filterManager.getFilterNameByID(filterID);
                this.updateAgentState(FcsEnumerations.FilterState.valueOf((String)("ONLINE_" + filterName.toUpperCase())));
                this.setFilterDuration = System.currentTimeMillis() - beginTime;
                FCSLOG.info("filter " + filterID + " is ONLINE but not clamped. setFilterDuration = " + this.setFilterDuration);
                this.publishData();
                this.subs.publishSubsystemDataOnStatusBus(new KeyValueData("duration/SETFILTER", (Serializable)Long.valueOf(this.setFilterDuration)));
                this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.LOADED});
                this.publishDataForMcm();
            });
        }
    }

    private void moveToHandoffWithHighVelocity(boolean forLoader) {
        if (forLoader) {
            this.autochanger.getAutochangerTrucks().fastProfile();
            this.autochanger.getAutochangerTrucks().goToHandOffForLoader();
        } else {
            this.autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
        }
    }

    private void setNoFilterAtHandoffOrOnline(boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setNoFilter");){
            this.updateStateWithSensors();
            if (this.carousel.isPowerSaveActivated()) {
                this.carousel.powerOn();
                this.carousel.waitUntilCarouselIsAwake();
            }
            this.publishDataForMcm();
            this.checkControllers();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.carousel.isAtStandby(), "carousel not stopped at STANDBY position", new Supplier[0]).precondition(this.autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY", new Supplier[0]).precondition(this.latchesOpenOrClosed(), "%s: bad state for autochanger latches - have to be OPENED or CLOSED", new Object[]{this.autochanger.getLatches().getLockStatus()}).precondition(this.onlineClampsOpenOrLocked(), "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtOnlineMustBeLocked(), "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by autochanger or by carousel.", new Supplier[0]).precondition(this.carouselHoldingFilterOrReadyToGrab(), "%s: bad state for carousel : should be holding a filter or ready to receive a filter", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(this.carouselReadyToClampAtStandby(), "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(new Enum[]{FcsEnumerations.FilterReadinessState.READY}).duration(SETFILTER_MAX_DURATION).enterFaultOnException(true).action(() -> {
                if (this.autochanger.isHoldingFilter()) {
                    FCSLOG.info(this.name + ": autochanger is holding filter: " + this.autochanger.getFilterOnTrucksName());
                    FCSLOG.info(this.name + ": is going to store a filter on carousel");
                    this.agentStateService.updateAgentState(new Enum[]{FcsEnumerations.McmState.UNLOADING});
                    this.storeFilterOnCarousel();
                } else {
                    FCSLOG.info(this.name + " no filter to store on carousel");
                }
                if (!this.autochanger.isEmpty() || !this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                    throw new FcsHardwareException(this.name + ": autochanger is not empty at handoff");
                }
                if (toOnline) {
                    this.autochanger.goToOnline();
                }
                this.updateAgentState(FcsEnumerations.FilterState.ONLINE_NONE);
                this.updateAgentState(FcsEnumerations.McmState.NO_FILTER);
                if (this.carousel.isPowerSaveAllowed()) {
                    this.carousel.powerSave();
                }
                this.publishDataForMcm();
                this.publishData();
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="change subsystem state.")
    public void changeState(String state) {
        this.updateAgentState(FcsEnumerations.FilterState.valueOf((String)state));
        FCSLOG.fine(() -> "SUBSYSTEM STATE=" + this.isInState((Enum)FcsEnumerations.FilterState.valueOf((String)state)));
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="list filters in Filter Changer.")
    public Map<Integer, String> listFiltersOnChanger() {
        HashMap<Integer, String> lf = new HashMap<Integer, String>();
        if (this.autochanger.getFilterID() != 0) {
            lf.put(this.autochanger.getFilterID(), this.autochanger.getFilterOnTrucksName());
        }
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty()) {
                lf.put(socket.getFilterID(), socket.getFilterName());
            }
        });
        return lf;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="list filters in Filter Changer.")
    public Map<Integer, String> listFiltersOnChanger2() {
        HashMap<Integer, String> lf = new HashMap<Integer, String>();
        if (this.autochanger.getFilterID() != 0) {
            lf.put(this.autochanger.getFilterID(), this.autochanger.getFilterOnTrucksName() + ",autochanger");
        }
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty()) {
                lf.put(socket.getFilterID(), socket.getFilterName() + "," + socket.getName());
            }
        });
        return lf;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="list all filters names.")
    public List<String> listAllFilters() {
        return this.filterManager.getFilterNames();
    }

    public Map<String, Filter> getFiltersOnChanger() {
        HashMap<String, Filter> lf = new HashMap<String, Filter>();
        if (this.autochanger.getFilterID() != 0) {
            lf.put("autochanger", this.filterManager.getFilterByID(this.autochanger.getFilterID()));
        }
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0) {
                lf.put(socket.getName(), this.filterManager.getFilterByID(socket.getFilterID()));
            }
        });
        return lf;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="list filters in Filter Changer.")
    public String printFilterOnChangerList() {
        StringBuilder sb = new StringBuilder("Filters on FilterChanger:\n");
        this.getFiltersOnChanger().values().stream().forEach(f -> {
            sb.append("filterID:");
            sb.append(f.getFilterID());
            sb.append("/filterName:");
            sb.append(f.getName());
            sb.append("/filterFamily:");
            sb.append(f.getFamily());
            sb.append("=");
            sb.append(f.getFamily().getFamilyName());
            sb.append("\n");
        });
        return sb.toString();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="list filters in Filter Changer.")
    public String printFilterLocalisation() {
        StringBuilder sb = new StringBuilder("Filters on FilterChanger: \n");
        for (Map.Entry<String, Filter> entry : this.getFiltersOnChanger().entrySet()) {
            Filter f = entry.getValue();
            sb.append("filterID:");
            sb.append(f.getFilterID());
            sb.append("/filterName:");
            sb.append(f.getName());
            sb.append("/filterFamily:");
            sb.append(f.getFamily());
            sb.append(" is on: ");
            sb.append(entry.getKey());
            sb.append("\n");
        }
        return sb.toString();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Load a filter from the loader to the camera. The loader must hold a filter at STORAGE position.The autochanger must be empty at HANDOFF, latches open. At the end of this command thefilter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.", timeout=480000, autoAck=false)
    public void loadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilter");){
            this.updateStateWithSensors();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.autochanger.isAtHandoff() && this.autochanger.isEmpty(), " autochanger is not empty at HANDOFF position; can't load a filter.", new Supplier[0]).precondition(this.autochanger.getLatches().isOpened(), "Autochanger latches must be open before loadFilter command.", new Supplier[0]).precondition(this.loader.isClampedOnFilter() && this.loader.isAtStorage(), this.name + " can't load filter because loader is not holding a filter at storage position.", new Supplier[0]).duration(LOAD_UNLOAD_FILTER_MAX_DURATION).enterFaultOnException(true).action(() -> {
                if (!this.autochanger.isPositionCorrectForExchangeWithLoader()) {
                    this.autochanger.goToHandOffForLoader();
                }
                this.loader.moveFilterToHandoff();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.autochanger.closeLatches();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.loader.openClampAndMoveEmptyToStorage();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Load a filter from the loader to the cameraand store it on socket which name is given as argument. The loader must hold a filter at STORAGE position.The autochanger must be empty at HANDOFF, latches open. At the end of this command thefilter is store on carousel socket which name is given as argument, autochanger is empty at Handoff position, and the loader carrier is empty at STORAGE.")
    public void loadFilterOnSocket(String socketName) {
        this.updateStateWithSensors();
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socket = this.carousel.getSocketsMap().get(socketName);
        if (!socket.isEmpty()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not empty. Select an empty socket.", socketName));
        }
        if (!socket.isAvailable()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not available. Select an available socket.", socketName));
        }
        this.loadFilter();
        this.carousel.rotateSocketToStandby(socketName);
        this.storeFilterOnCarousel();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Unload a filter from the camera to the loader. The camera has to be at horizontal position. The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. \nAt the end of this command, autochanger is empty at HANDOFF, latches are open, and loader carrier is holding the filter at STORAGE (hooks are clamped)", timeout=480000, autoAck=false)
    public void unloadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilter");){
            this.updateStateWithSensors();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.autochanger.isAtHandoff() && this.autochanger.isHoldingFilter(), " autochanger is not holding a filter at HANDOFF position; can't unload a filter.", new Supplier[0]).duration(LOAD_UNLOAD_FILTER_MAX_DURATION).enterFaultOnException(true).action(() -> {
                if (!this.autochanger.isPositionCorrectForExchangeWithLoader()) {
                    if (this.autochanger.isLinearRailMotionAllowed()) {
                        this.autochanger.goToHandOffForLoader();
                    } else {
                        throw new RejectedCommandException("Autochanger trucks position is not correct for exchange with loader  but command goToHandOffForLoader can't be executed because PLC doesn't allow to move trucks.");
                    }
                }
                this.loader.moveEmptyToHandoffAndClose();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.autochanger.openLatches();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.loader.moveFilterToStorage();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
            });
        }
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="check if controllers are in fault")
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    @Override
    public synchronized void updateStateWithSensors() {
        if (System.currentTimeMillis() - this.lastUpdateStateWithSensors.get() < 100L && !FcsUtils.isSimu()) {
            FCSLOG.info(this.name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
            return;
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs");){
            this.carousel.updateStateWithSensors();
            if (this.carousel.isAtStandby()) {
                this.carousel.getSocketAtStandby().updateFilterID();
            }
            this.autochanger.updateStateWithSensors();
            if (this.autochanger.getAutochangerTrucks().isAtOnline() && this.autochanger.isHoldingFilter() && this.autochanger.getOnlineClamps().isLocked()) {
                String filterName = this.filterManager.getFilterNameByID(this.autochanger.getFilterID());
                this.updateAgentState(FcsEnumerations.FilterState.valueOf((String)("ONLINE_" + filterName.toUpperCase())));
            } else {
                this.updateAgentState(FcsEnumerations.FilterState.valueOf((String)"ONLINE_NONE"));
            }
            if (this.loader.isCANbusConnected()) {
                this.loader.updateStateWithSensors();
            }
        }
        if (this.firstMCMpublication) {
            this.publishDataForMcm();
            this.firstMCMpublication = false;
        }
    }

    @Override
    public void initializeHardware() {
        this.carousel.initializeHardware();
        this.autochanger.initializeHardware();
        this.loader.initializeHardware();
        this.postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        this.bridgeToLoader.publishData();
    }

    public void publishDataForMcm() {
        if (this.autochanger.getAutochangerTrucks().getPosition() == 0) {
            FcsUtils.sleep(100, "main");
            this.autochanger.updateStateWithSensors();
        }
        this.updateAgentState(this.autochanger.getAutochangerInclinationState());
        this.subs.publishSubsystemDataOnStatusBus(this.getDataForMcm());
    }

    public void publishDataProviderCurrentData(AgentInfo ... agents) {
        for (AgentInfo agent : agents) {
            if (!AgentInfo.AgentType.MCM.equals((Object)agent.getType())) continue;
            FCSLOG.info("FCS published for MCM that just joined the buses.");
            this.publishDataForMcm();
            break;
        }
    }

    private KeyValueData getDataForMcm() {
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData("filter_on_autochanger_id", (Serializable)Integer.valueOf(this.autochanger.getFilterID()));
        kvdl.addData("filter_on_autochanger_name", (Serializable)((Object)this.autochanger.getFilterOnTrucksName()));
        kvdl.addData("filter_previous_socketID", (Serializable)Integer.valueOf(this.previousSocketID));
        kvdl.addData("autochanger_trucks_position", (Serializable)Integer.valueOf(this.autochanger.getAutochangerTrucks().getPosition()));
        kvdl.addData("autochanger_trucks_location", (Serializable)this.autochanger.getAutochangerTrucksLocation());
        kvdl.addData("setFilter_percentage", (Serializable)Integer.valueOf(this.percentage));
        kvdl.addData("proximity", (Serializable)Integer.valueOf(this.autochanger.getOnlineProximityDistance()));
        return new KeyValueData(PUBLICATION_KEY_FOR_MCM, (Serializable)kvdl);
    }

    public void postStart() {
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(this.name + " is shutting down.");
        this.bridge.doShutdown();
        if (this.loader.isCANbusConnected()) {
            this.bridgeToLoader.doShutdown();
        }
        FCSLOG.info(this.name + " is shutdown.");
    }

    public String vetoTransitionToNormalMode() {
        if (this.isChangerReady()) {
            this.publishDataForMcm();
            if (this.autochanger.isHoldingFilter() && this.autochanger.isAtOnline()) {
                this.updateAgentState(FcsEnumerations.McmState.LOADED);
            } else {
                this.updateAgentState(FcsEnumerations.McmState.NO_FILTER);
            }
            return super.vetoTransitionToNormalMode();
        }
        return "FilterChanger not ready";
    }
}

