package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *
 * @author virieux
 */
public class CanOpenEPOSWithPDO extends CanOpenEPOS {

    /**
     * define cobid to be able to read values from PDO data. initialized in init
     * method
     */
    protected int cobid1 = 0x180;
    protected int cobid2 = 0x280;

    /**
     * PDO1 represents :
     *
     * status word (0x60410010): 2 bytes
     *
     * average current (0x20270010): 2 bytes
     *
     * position (0x60640020): 4 bytes
     *
     */
    private long pdo1 = 0L;

    /**
     * PDO2 represents :
     *
     * current (0x60400010): 2 bytes
     *
     * following error (0x20F40010): 2 bytes
     *
     * average velocity (0x20280020): 4 bytes
     *
     */
    private long pdo2 = 0L;

    /**
     * set a pdo1 for simulation
     *
     * @param pdo1
     */
    public void setPdo1(long pdo1) {
        this.pdo1 = pdo1;
    }

    /**
     * set a pdo2 for simulation
     *
     * @param pdo2
     */
    public void setPdo2(long pdo2) {
        this.pdo2 = pdo2;
    }

    /**
     * for simulation
     *
     * @return pdo1
     */
    public long getPdo1() {
        return pdo1;
    }

    /**
     * for simulation
     *
     * @return pdo2
     */
    public long getPdo2() {
        return pdo2;
    }

    /**
     * for simulation only
     *
     * @return cobid1
     */
    public int getCobid1() {
        return cobid1;
    }

    /**
     * for simulation only
     *
     * @return cobid2
     */
    public int getCobid2() {
        return cobid2;
    }

    @Override
    public void init() {
        super.build();
        cobid1 = this.nodeID + 0x180;
        cobid2 = this.nodeID + 0x280;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        FCSLOG.finest(name + " updatingFromPDO = " + pdo);
        boolean updated = false;
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
            updated = true;
            /* update from PDO1 */
            FCSLOG.finer(name + " updatingFromPDO1 = " + pdo1 + " binary:" + Long.toBinaryString(pdo1));
            statusWord = extractStatusWord(pdo1);
            averageCurrent = (short) extractAverageCurrent(pdo1);
            position = (int) extractPosition(pdo1);
            updateEposState(statusWord);
            FCSLOG.finer(name + " position = " + position + " pdo1 = " + " binary:" + Long.toBinaryString(pdo1));
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
            updated = true;
            /* update from PDO2 */
            FCSLOG.finer(name + " updatingFromPDO2 = " + pdo2 + " binary:" + Long.toBinaryString(pdo2));
            current = (short) extractCurrent(pdo2);
            followingError = (short) extractFollowingError(pdo2);
            velocity = (int) extractAverageVelocity(pdo2);
        }
        if (updated) {
            this.publishData();
        }
    }

    /**
     * @param pdo
     * @return statusWord from pdo
     */
    public static int extractStatusWord(long pdo) {
        return (int) pdo & 0xFFFF;
    }

    /**
     * @param pdo
     * @return average current from pdo
     */
    public static int extractAverageCurrent(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return position from pdo
     */
    public static int extractPosition(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * @param pdo
     * @return current from pdo
     */
    public static int extractCurrent(long pdo) {
        return (int) pdo & 0xFFFF;
    }

    /**
     * @param pdo
     * @return following error from pdo
     */
    public static int extractFollowingError(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return average velocity from pdo
     */
    public static int extractAverageVelocity(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * For tests and debug
     *
     * @return values updated by PDOs
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print values updated by PDOs to debug and test")
    public String printValuesUpdatedByPDOs() {
        StringBuilder sb = new StringBuilder("=> pdo1 value = ");
        sb.append(pdo1);
        sb.append(" / pdo1 = 0x");
        sb.append(Long.toHexString(pdo1));
        sb.append(" / status word = 0x");
        sb.append(Long.toHexString(statusWord));
        sb.append("\n status word = 0b");
        sb.append(Long.toBinaryString(statusWord));
        sb.append("\n average current = ");
        sb.append(averageCurrent);
        sb.append(" / average current = 0x");
        sb.append(Long.toHexString(averageCurrent));
        sb.append("\n position = ");
        sb.append(position);
        sb.append(" / position = 0x");
        sb.append(Long.toHexString(position));
        sb.append("\n => pdo2 value = ");
        sb.append(pdo2);
        sb.append(" / pdo2 = 0x");
        sb.append(Long.toHexString(pdo2));
        sb.append("\n current = ");
        sb.append(current);
        sb.append(" / current = 0x");
        sb.append(Long.toHexString(current));
        sb.append("\n followingError = ");
        sb.append(followingError);
        sb.append(" / followingError = 0x");
        sb.append(Long.toHexString(followingError));
        sb.append(followingError);
        sb.append("\n Velocity = ");
        sb.append(velocity);
        sb.append(" / velocity = 0x");
        sb.append("\n ControlWord = ");
        sb.append(controlWord);
        sb.append(" / controlWord = 0x");
        sb.append(Long.toHexString(controlWord));
        return sb.toString();
    }

}
