
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;

import static org.lsst.ccs.subsystems.fcs.FCSCst.PLUTOGATEWAY_NB_BYTES;
import org.lsst.ccs.subsystems.fcs.ForceSensor;
import org.lsst.ccs.subsystems.fcs.Inclinometer;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * This represents the gateway to the pluto PLC involved in the camera
 * protection system. The sensors are read through this gateway. We read the
 * data from the gateway byte by byte. Usually we read 2 bytes at one time. This
 * class has to be extended to fit the real hardware control software needs and
 * the simulator.
 *
 * @author virieux
 */
public abstract class PlutoGateway extends CanOpenDevice implements PlutoGatewayInterface {

    protected int[] readValues = new int[PLUTOGATEWAY_NB_BYTES];

    /**
     * map of DigitalSensors attached to this gateway.
     */
    @LookupField(strategy = TREE)
    protected Map<String, SensorPluggedOnDevice> sensorsMap = new HashMap<>();

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByPlutoGateway.class, path);
        //Registering the KeyValueData published below
        //This registration adds some 380 quantities to the dictionary.
        KeyValueData data = (KeyValueData) getSensorsData();
        dataProviderDictionaryService.registerData(data);
        KeyValueDataList dataList = (KeyValueDataList) data.getValue();
        String mainPath = data.getKey();
        for ( KeyValueData d : dataList ) {
            String path = mainPath + "/" + d.getKey();
            DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
            info.addAttribute(DataProviderInfo.Attribute.UNITS, "unitless");
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Value of Digital Sensor " + d.getKey());
        }
    }

    @Override
    public void init() {
        FCSLOG.info(name + " =====> initialization of my sensors map");
        StringBuilder sb = new StringBuilder();
        if (sensorsMap.isEmpty()) {
            FCSLOG.severe("NO SENSORS ARE FOUND FOR " + name);
            return;
        }
        for (Iterator<SensorPluggedOnDevice> it = sensorsMap.values().iterator(); it.hasNext();) {
            SensorPluggedOnDevice sensor = it.next();
            if (name.equals(sensor.getDeviceName())) {
                sb.append(sensor.getName());
                sb.append(" ====> IS A SENSOR PLUGGED ON ");
                sb.append(name);
                sb.append(":byteNumero=");
                sb.append(sensor.getByteNumero());
                if (sensor instanceof DigitalSensor) {
                    sb.append(";inputNumero=");
                    sb.append(((DigitalSensor) sensor).getInputNumero());
                }
                sb.append("\n");
            } else {
                it.remove();
            }
        }
        FCSLOG.info(sb.toString());
    }

    /**
     *
     * For GUI
     *
     * @return list of names of sensors plugged on this device
     */
    public List<String> listMySensorsNames() {
        if (sensorsMap.isEmpty()) {
            FCSLOG.info("NO SENSORS ARE FOUND FOR " + name);
            return Collections.emptyList();
        } else {
            FCSLOG.info("listMySensorsNames for " + name);
            List<String> res = new ArrayList<>();
            res.addAll(sensorsMap.keySet());
            Collections.sort(res);
            return res;
        }
    }

    /**
     * Read new readValues on the device plutoGateway and updates field readValues.
     *
     */
    @Override
    public void updateValues() {
        this.readValues = readNewValues();
        FCSLOG.info(name + " updating my sensors values.");
        sensorsMap.values().stream().forEach((SensorPluggedOnDevice sensor) -> {
            sensor.updateValue(this.readValues[sensor.getByteNumero()]);
        });
        publishData();
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
             description = "Return the string representation of PlutoGateway")
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        for (int i = 0; i < readValues.length; i++) {
            sb.append("/byte");
            sb.append(i);
            sb.append('=');
            sb.append(String.valueOf(this.readValues[i]));
        }
        return sb.toString();
    }

    /**
     * Create an object with status data to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByPlutoGateway createStatusDataPublishedByPlutoGatewayModule() {
        return new StatusDataPublishedByPlutoGateway(isBooted(), isInitialized(), readValues);
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(
                new KeyValueData(getPath(), createStatusDataPublishedByPlutoGatewayModule()));
        subs.publishSubsystemDataOnStatusBus(getSensorsData());
    }

    private KeyValueData getSensorsData() {
        /* publication of the list of values for the sensors which are plugged on this plutoGateway */
        KeyValueDataList kvdl = new KeyValueDataList();
        sensorsMap.values().stream().forEach((SensorPluggedOnDevice sensor) -> {
            if (sensor instanceof Inclinometer) {
                kvdl.addData(sensor.getName(), ((Inclinometer) sensor).getInclinaison());
            } else if (sensor instanceof ForceSensor) {
                kvdl.addData(sensor.getName(), ((ForceSensor) sensor).getVoltage());
            } else if (sensor instanceof DigitalSensor){
                kvdl.addData(sensor.getName(), ((DigitalSensor) sensor).isOn() ? 1 : 0);
            } else {
                FCSLOG.severe(getName() + " UNKNOWN sensor instance");
            }
        });
        return new KeyValueData(getPath(), kvdl);
    }
}
