package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.OPERATION_ENABLE;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_LOCKING_ISSUE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_LOCKING_RECOVERY_FAILURE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_LOCKING_RECOVERY_SUCCESS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.CLAMPED_ON_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.READY_TO_CLAMP;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNCLAMPED_EMPTY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNCLAMPED_ON_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNDEFINED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNLOCKABLE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.LOCKABLE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.NOFILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.NOT_LOCKABLE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.RELAXED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNLOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.RELEASE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK;

import java.util.ArrayList;
import java.util.logging.Logger;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * A class to model a clamp that holds a filter on the carousel. Each clamp on
 * the carousel is coupled with 2 sensors which detect the presence of a filter,
 * or if the clamp is locked or unlocked. This class provides a method to lock,
 * unlock or release the clamps, and to compute the clampState of the clamp. The
 * clampState of the clamp is computed, updated and published on the status bus
 * each tick of the timer.
 *
 *
 * @author virieux
 */
public class CarouselClamp extends MobileItem implements MovedByEPOSController {
    private static final Logger FCSLOG = Logger.getLogger(CarouselClamp.class.getName());
    /**
     * to be able to known if autochanger hold a filter or not at STANDBY can't be
     * an instance of Autochanger because in standalone it's not an instance of
     * class Autochanger can't be strategy = TREE because in fcs__hardware
     * there are 2 FilterHolder : autochanger and loader
    */
    @LookupField(strategy = TREE, pathFilter = "autochanger")
    private FilterHolder autochanger;

    @LookupField(strategy = TREE)
    private Carousel carousel;

    @LookupField(strategy = TREE, pathFilter = CHANGER_TCPPROXY_NAME)
    protected BridgeToHardware tcpProxy;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private int id;

    private boolean available = true;

    /**
     * The motor controller which controls this clamp. This is initialized in the
     * initModule method of CarouselSocket.
     */
    protected EPOSController controller;

    /**
     * The filterPresenceSensor detects where if the filter is in the clamp.
     *
     * @See enum PresenceFilterStateOnClamp
     */
    private final SensorPluggedOnTTC580 filterPresenceSensor = new CarouselClampSensor();

    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final SensorPluggedOnTTC580 lockSensor = new CarouselClampSensor();

    private FilterClampState clampState = UNDEFINED;
    private FilterPresenceStatus filterPresenceStatus = NOT_LOCKABLE;
    private LockStatus lockStatus = LockStatus.UNKNOWN;

    @ConfigurationParameter(description = "value of current to send to controller "
            + "to unlock carousel clamp", range = "-6000..6000", units = "mA", category = "carousel")
    protected volatile int currentToUnlock = -4660;

    /**
     * If lockSensor returns a value between lockSensorMinLimit and
     * lockSensorOffset1 the clamp is unlocked, If lockSensor returns a value
     * between lockSensorOffset1 and carousel.getLockSensorMinLocked(): we don't
     * know, If the sensor returns a value between
     * carousel.getLockSensorMinLocked() and lockSensorMaxValue: the clamp is
     * locked. If the sensor returns a value greater than lockSensorMaxValue the
     * sensor is in ERROR.
     *
     * offset1 is given by the electronic card inside each clamp
     *
     * offset1 is read on hyttc580 in Carousel.initializeAndCheckClampsOffset
     * offset1 is persisted and read at each startup.
     *
     */
    @Persist
    protected volatile Integer lockSensorOffset1 = 5000;

    /**
     * if filter presence sensor returns a value between 0 and filterPresenceOffset1
     * : sensor is in error.
     *
     * filterPresenceOffset1 has same value for all clamps.
     *
     */
    protected Integer filterPresenceMinLimit = 200;

    /**
     * if filter presence sensor returns a value between filterPresenceOffset1 and
     * filterPresenceOffset2 : the filter is engaged and lockable. This value is
     * read on hyttc580 when fcs starts. It is persisted. If the new value read is
     * different from the precedent value, it means that the clamp has been
     * replaced, and an ALERT must be launched.
     */
    @Persist
    protected volatile Integer filterPresenceOffset2 = 3000;

    // TODO REVIEW CAROUSEL update description
    @ConfigurationParameter(description = "an offset for tests after carousel has been "
            + "transported", range = "0..1000", units = "mV", category = "carousel")
    protected volatile Integer filterPresenceOffset3 = 0;

    @ConfigurationParameter(description = "Maximum value for the filter presence sensor on the clamp", range = "0..12000", units = "mV", category = "carousel")
    private volatile Integer filterPresenceMaxValue = 12000;

    @ConfigurationParameter(description = "Timeout for unlocking the clamp", range = "0..10000", units = "millisecond", category = "carousel")
    protected volatile long timeoutForUnlocking = 1500;

    @ConfigurationParameter(description = "Timeout for releasing the clamp", range = "0..10000", units = "millisecond", category = "carousel")
    protected volatile long timeoutForReleasing = 1500;

    /* updated from PDO of clamp controller */
    private int velocity = 0;

    private ArrayList<String> sensorErrorCounter;

    @SuppressWarnings("rawtypes")
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarouselClamp.class, path);
        registerAction(UNLOCK);
        registerAction(RELEASE);

        //We are adding here the filterPresenceSensor and the lockSensor to the lookup field.
        //For now we are using the same naming convention as was used in groovy:
        //  "filterPresenceX[minus|plus]{i}" and "lockSensorX[minus|plus]{i}"
        //where we extract X[minus|plus]{i} from this CarouselClampName
        String suffix = name.replace("clamp", "");
        ComponentLookup lookup = subs.getComponentLookup();
        ComponentNode thisNode = lookup.getComponentNodeForObject(this);

        lookup.addComponentNodeToLookup(thisNode, new ComponentNode("filterPresence"+suffix,filterPresenceSensor));
        lookup.addComponentNodeToLookup(thisNode, new ComponentNode("lockSensor"+suffix,lockSensor));
    }

    /**
     * Returns the amount of current needed to unlock this clamp. This is defined in
     * the ConfigurationSystem.
     *
     * @return
     */
    public short getCurrentToUnlock() {
        return (short) currentToUnlock;
    }

    /**
     *
     * @return a list of error messages when updating sensors
     */
    public ArrayList<String> getSensorErrorCounter() {
        return sensorErrorCounter;
    }

    /**
     * In the initialization phase, this method is used to initialize the
     * controller. cf initModule in CarouselSocket
     *
     * @param actuator
     */
    protected void setController(EPOSController actuator) {
        this.controller = actuator;
    }

    /**
     * @return the lockSensorOffset1 for simulation
     */
    public int getLockSensorOffset1() {
        return lockSensorOffset1;
    }

    public int getLockSensorMaxLimit() {
        if (isXminus()) {
            return carousel.getLockSensorMaxLimitXminus();
        } else {
            return carousel.getLockSensorMaxLimitXplus();
        }
    }

    public int getLockSensorMinLimit() {
        if (isXminus()) {
            return carousel.getLockSensorMinLimitXminus();
        } else {
            return carousel.getLockSensorMinLimitXplus();
        }
    }

    /**
     * @return the filterPresenceOffset1
     */
    public int getFilterPresenceMinLimit() {
        return filterPresenceMinLimit;
    }

    /**
     * For simulation
     *
     * @return the filterPresenceOffset2
     */
    public int getFilterPresenceOffset2() {
        return filterPresenceOffset2;
    }

    /**
     * For simulation.
     *
     * @return the filterPresenceSensor
     */
    public SensorPluggedOnTTC580 getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * For simulation.
     *
     * @return the lockSensor
     */
    public SensorPluggedOnTTC580 getLockSensor() {
        return lockSensor;
    }

    /**
     * @return the lockStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return the lockStatus")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     *
     * @return filterPresenceStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public void setFilterPresenceOffset2(Integer value) {
        this.filterPresenceOffset2 = value;
    }

    public boolean isAvailable() {
        return available;
    }

    public void setAvailable(boolean available) {
        this.available = available;
    }

    /**
     * ***********************************************************************************************
     * * END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     *
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(CA_SENSOR_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_LOCKING_ISSUE.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_LOCKING_RECOVERY_SUCCESS.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_LOCKING_RECOVERY_FAILURE.getAlert(name), alwaysClear);
    }

    /**
     * *** end of lifecycle methods
     * *************************************************
     */


    /**
     * *
     *
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return controller.isInitialized();
    }

    /**
     * Returns the clampState of the clamp. If the clampState is being updated and
     * waiting for a response from a sensor, this methods waits until the clampState
     * is updated. If the clampState is not being updated, it returns immediately the
     * clampState.
     *
     * @return clampState
     *
     */
    public FilterClampState getClampState() {
        return clampState;
    }

    /**
     * This methods returns true if the clamp is locked. In the simulator, it is
     * overridden.
     *
     * @return true if the clamp is locked
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the clamp is locked")
    public boolean isLocked() {
        return this.lockStatus == LOCKED;
    }

    private boolean isAtStandby() {
        return carousel.getSocketAtStandbyID() == id;
    }

    /**
     *
     * @return true if filter is engaged on the clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if filter is engaged on the clamp : filter presence sensors sees it ")
    public boolean isFilterEngaged() {
        return this.filterPresenceStatus == LOCKABLE;
    }

    /**
     * compare offset1 value given as argument with persisted value.
     *
     * @param newOffset1
     */
    public void checkAndUpdateOffset1(int newOffset1) {
        if (Math.abs(newOffset1 - lockSensorOffset1) > carousel.getMaxClampsOffsetDelta()) {
            lockSensorOffset1 = newOffset1;
            this.raiseAlarm(CA_SENSOR_ERROR, " offset1 has changed; new value = " + newOffset1, name);
        }
    }

    /**
     * compare offset2 value given as argument with persisted value.
     *
     * @param newOffset2
     */
    public void checkAndUpdateOffset2(int newOffset2) {
        if (Math.abs(newOffset2 - filterPresenceOffset2) > carousel.getMaxClampsOffsetDelta()) {
            filterPresenceOffset2 = newOffset2;
            this.raiseAlarm(CA_SENSOR_ERROR, " offset2 has changed; new value = " + newOffset2, name);
        }
    }

    /**
     *
     * @return true if this clamp is clampXminus.
     */
    public boolean isXminus() {
        return name.contains("Xminus");
    }

    /**
     * update filterPresenceStatus from value returned by filterPresenceSensor
     */
    public void updateFilterPresenceStatus() {
        String errorMsg;
        int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
        if (newFilterPresenceSensorValue < filterPresenceMinLimit) {
            if (carousel.isRotating()) {
                /*
                 * during rotation, communication with TTC30 is sometimes lost, it's a transient
                 * error, just log
                 */
                FCSLOG.info(name + String.format(name + " lost communication with TTC30 during rotation at position %d",
                        carousel.getPosition()));
            } else if (available) {
                this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                errorMsg = name + " ERROR new read value FOR FILTER POSITION SENSOR  = " + newFilterPresenceSensorValue
                        + " must be >=" + filterPresenceMinLimit + " lost communication with TTC30 ?";
                sensorErrorCounter.add(errorMsg);
                this.raiseWarning(CA_SENSOR_ERROR, errorMsg, name);
            }

        } else if (newFilterPresenceSensorValue < this.filterPresenceOffset2 + filterPresenceOffset3) {
            this.filterPresenceStatus = LOCKABLE;

        } else if (newFilterPresenceSensorValue < carousel.getFilterPresenceMinNoFilter()) {
            this.filterPresenceStatus = NOT_LOCKABLE;

        } else if (newFilterPresenceSensorValue < this.filterPresenceMaxValue) {
            this.filterPresenceStatus = NOFILTER;

        } else if (available) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            errorMsg = name + " ERROR new read value FOR FILTER POSITION SENSOR  = " + newFilterPresenceSensorValue
                    + " must be <=" + this.filterPresenceMaxValue;
            sensorErrorCounter.add(errorMsg);
            this.raiseWarning(CA_SENSOR_ERROR, errorMsg, name);
        }
    }

    /**
     * Update lockStatus from value returned by lockSensor.
     *
     * @throws FcsHardwareException
     */
    public void updateLockStatus() {
        String errorMsg;
        int mechanicalValue = lockSensor.getValue();
        if (mechanicalValue < getLockSensorMinLimit()) {
            if (carousel.isRotating()) {
                /*
                 * during rotation, communication with TTC30 is sometimes lost, it's a transient
                 * error, just log
                 */
                FCSLOG.info(name + String.format(name + " lost communication with TTC30 during rotation at position %d",
                        carousel.getPosition()));
            } else if (available) {
                this.lockStatus = LockStatus.ERROR;
                errorMsg = name + " ERROR new read value FOR LOCK SENSOR  = " + mechanicalValue + " should be >= "
                        + getLockSensorMinLimit() + " lost communication with TTC30 ?";
                sensorErrorCounter.add(errorMsg);
                this.raiseWarning(CA_SENSOR_ERROR, errorMsg, name);
            }

        } else if (mechanicalValue < lockSensorOffset1) {
            this.lockStatus = UNLOCKED;

            /*
             * for clampXminus, lockSensor can sent a value > lockSensorValueA but it is
             * opened
             */
        } else if (mechanicalValue <= lockSensorOffset1 + 500 && isXminus()) {
            this.lockStatus = RELAXED;

        } else if (mechanicalValue <= carousel.getMinLockedThreshold()) {
            this.lockStatus = LockStatus.UNKNOWN;

        } else if (mechanicalValue <= getLockSensorMaxLimit()) {
            this.lockStatus = LOCKED;

        } else if (available) {
            this.lockStatus = LockStatus.ERROR;
            errorMsg = name + " ERROR new read value FOR LOCK SENSOR  = " + mechanicalValue + " should be <= "
                    + getLockSensorMaxLimit();
            sensorErrorCounter.add(errorMsg);
            this.raiseWarning(CA_SENSOR_ERROR, errorMsg, name);
        }
    }

    /**
     * Update clampState from values sent by sensors. This method updates the clamp
     * clampState regarding the value returned by the filter presence sensor and the
     * value returned by the method isLocked(). When the update is completed, it
     * sends a signal to threads waiting to get the new value of clampState.
     *
     * @throws FcsHardwareException
     */
    public void updateState() {
        synchronized (this) {
            sensorErrorCounter = new ArrayList<String>();
            this.updateFilterPresenceStatus();
            this.updateLockStatus();
            this.computeClampState();
            this.updateVelocity();
        }
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByCarouselClamp status = this.createStatusDataPublishedByClamp();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, status));
    }

    /**
     * Create an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByCarouselClamp createStatusDataPublishedByClamp() {
        StatusDataPublishedByCarouselClamp status = new StatusDataPublishedByCarouselClamp();
        status.setClampState(clampState);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterPositionSensorValue(filterPresenceSensor.getValue());
        status.setLockSensorValue(lockSensor.getValue());
        status.setLockStatus(lockStatus);
        status.setLockSensorOffset1(lockSensorOffset1);
        status.setFilterPresenceMinLimit(filterPresenceMinLimit);
        status.setFilterPresenceOffset2(filterPresenceOffset2);
        return status;
    }

    /**
     * Compute the global state of the clamp given the lock sensor and the presence
     * filter sensor state. This has to be overridden for the clamp X-.
     *
     */
    private void computeClampState() {
        // TODO REVIEW CAROUSEL: write comments on the computation with Guillaume
        if (this.filterPresenceStatus == FilterPresenceStatus.ERROR || this.lockStatus == LockStatus.ERROR) {
            clampState = FilterClampState.ERROR;

        } else if (this.filterPresenceStatus == NOT_LOCKABLE) {
            clampState = UNLOCKABLE;

        } else if (this.filterPresenceStatus == LOCKABLE) {
            // a filter is in the socket

            if (this.lockStatus == LOCKED) {
                clampState = CLAMPED_ON_FILTER;

            } else if (this.lockStatus == UNLOCKED) {
                clampState = UNCLAMPED_ON_FILTER;
                FCSLOG.info(name + " to compute duration of unlockClamps time = " + System.currentTimeMillis());

            } else {
                FCSLOG.fine(() -> name + " is " + LOCKABLE + " and " + lockStatus + " lockValue = " + lockSensor.getValue()
                        + " clampState = " + UNDEFINED);
                clampState = UNDEFINED;
            }

        } else if (this.filterPresenceStatus == NOFILTER) {
            // NOFILTER for clampXplus (when no filter is detected)
            // impossible to have no filter and clamp LOCKED for clampXminus.

            if (this.lockStatus == LOCKED) {
                clampState = READY_TO_CLAMP;

                // clampXminus can be RELAXED
            } else if (this.lockStatus == UNLOCKED || this.lockStatus == RELAXED) {
                clampState = UNCLAMPED_EMPTY;

            } else {
                FCSLOG.fine(() -> name + " is " + NOFILTER + " and " + lockStatus + " lockValue = " + lockSensor.getValue()
                        + " clampState = " + UNDEFINED);
                clampState = UNDEFINED;
            }

        } else {
            // NOT_LOCKABLE (UNDEFINED)
            FCSLOG.fine(() -> name + " is " + filterPresenceStatus + " and " + lockStatus + " lockValue = "
                    + lockSensor.getValue() + " clampState = " + UNDEFINED);
            clampState = UNDEFINED;
        }
    }

    /**
     * The clamps on the carousel are locked automatically when the filter comes at
     * the standby position. To be able to lock automatically again, it has to be
     * released after each time it has been unlocked.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING_EXPERT, type = Command.CommandType.ACTION, description = "Release clamp in order to get ready to clamp a filter again")
    public void release() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-release")) {
            FCSLOG.info("Checking conditions for release clamp " + name + " on socket at standby position.");

            carousel.updateSocketAtStandbyReadSensorsNoPublication();
            FCSLOG.info("Releasing clamp " + name + " on socket at standby position.");
            this.executeAction(RELEASE, timeoutForUnlocking);
        }
        carousel.updateSocketAtStandbyWithSensors();
    }

    /**
     *
     * Check if controller has been disabled. For end user. Not to be used in other
     * commands because it read StatusWord by SDO.
     *
     * @return true if controller is not enabled.
     */
    @Command(level = Command.ENGINEERING_EXPERT, type = Command.CommandType.ACTION, description = "Check if controller has been disabled. For end user.")
    public boolean isReleased() {
        return !controller.isEnabled();
    }

    /**
     * Unlock the clamp when a filter is locked by the clamp. This is used only in
     * engineering mode.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Unlock the clamp")
    public void unlock() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-unlock")) {
            FCSLOG.info(name + ": " + "UNLOCK State1 = " + clampState.toString());
            carousel.updateSocketAtStandbyReadSensorsNoPublication();

            if (isFilterEngaged() && autochanger.isAtStandby() && !autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(
                        name + "cannot unlock a clamp is FILTER is NOT HELD by autochanger.");
            }
            this.executeAction(UNLOCK, timeoutForUnlocking);
        }
        carousel.updateSocketAtStandbyState();
    }

    /**
     * To recover locking when clampXplus is LOCKED and not clampXminus.
     */
    public void recoveryLocking() {
        raiseWarning(CA_LOCKING_ISSUE, String.format(name + " is not LOCKED : filter presence = %d lock value = %d"
                + " about to do a recoveryLocking",
                filterPresenceSensor.getValue(), lockSensor.getValue()), name);
        this.controller.goToOperationEnable();
        this.controller.writeCurrent((short) carousel.getRecoveryLockingCurrent());
        try {
            waitForLocked();
        } finally {
            this.controller.goToSwitchOnDisabled();
        }
    }

    private void waitForLocked() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-waitForLocked")) {
            long timeoutMillis = 1500;
            final long timeStart = System.currentTimeMillis();
            long duration = 0;
            boolean state_ok = false;
            int updateRate = 50;
            while (!state_ok && duration <= timeoutMillis) {
                FcsUtils.sleep(updateRate, name);
                duration = System.currentTimeMillis() - timeStart;
                tcpProxy.updatePDOData();
                carousel.updateSocketAtStandbyState();
                updateLockStatus();
                state_ok = isLocked();
            }
            if (state_ok) {
                raiseWarning(CA_LOCKING_RECOVERY_SUCCESS, String.format(name + " filter presence = %d lock value = %d",
                        filterPresenceSensor.getValue(), lockSensor.getValue()), name);
                FCSLOG.info(name + String.format(" go to state LOCKED duration = %d", duration));
            } else {
                /* let the clamp have a chance to complete locking
                see https://jira.slac.stanford.edu/browse/LSSTCCSFCS-503 */
                controller.goToSwitchOnDisabled();
                raiseWarning(CA_LOCKING_RECOVERY_FAILURE, String.format(name + " could not be LOCKED during allocated time, stop waiting.  filter presence = %d lock value = %d",
                        filterPresenceSensor.getValue(), lockSensor.getValue()), name);
                FcsUtils.sleep(500, name);
                tcpProxy.updatePDOData();
                carousel.updateSocketAtStandbyState();
                if (isLocked()) {
                    //success : controller.goToSwitchOnDisabled was enough to lock clamp
                    raiseWarning(CA_LOCKING_RECOVERY_SUCCESS, String.format(name + " finally clamp is LOCKED after controller.goToSwitchOnDisabled. filter presence = %d lock value = %d",
                            filterPresenceSensor.getValue(), lockSensor.getValue()), name);
                    FCSLOG.info(name + String.format(" go to state LOCKED duration = %d", duration));

                } else {
                    //failure
                    String msg = name + " LOCKING RECOVERY FAILED ";
                    String cause;
                    if (velocity > carousel.getRecoveryMaxVelocity()) {
                        cause = String.format("Cause = controller velocity reached %d which is over limit %d", velocity,
                                carousel.getRecoveryMaxVelocity());
                    } else {
                        cause = String.format("Cause = couldn't go to state LOCKED during time allocated of %d ms;",
                                timeoutMillis);
                    }
                    String msg_log = msg + cause + String.format("; filter presence value = %d lock value = %d",
                            filterPresenceSensor.getValue(), lockSensor.getValue());
                    raiseAlarm(CA_LOCKING_RECOVERY_FAILURE, msg_log, name);
                    FCSLOG.severe(msg_log);
                    throw new FcsHardwareException(msg);
                }
            }
        }
    }

    @Override
    // TODO REVIEW CAROUSEL with Guillaume Daubard
    public void startAction(MobileItemAction action) {
        if (action == UNLOCK) {
            this.controller.goToOperationEnable();
            // Following a request from Guillaume Daubard, we first send a small current
            // to the clamp to prepare it and prevent an unstable state when we send the
            // full current to unlock.
            int timeToPrepareUnlock = carousel.getTimeToPrepareUnlock();
            short currentToPrepareUnlock = (short) carousel.getCurrentToPrepareUnlock();

            this.controller.writeCurrent(currentToPrepareUnlock);

            // a little sleep to let time to the hardware to prepare
            FcsUtils.sleep(timeToPrepareUnlock, name);

            this.controller.writeCurrent((short) this.currentToUnlock);

        } else if (action == RELEASE) {
            controller.goToSwitchOnDisabled();

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK " + "or RELEASE");
        }

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == UNLOCK) {
            return this.clampState == UNCLAMPED_ON_FILTER;

        } else if (action == RELEASE) {
            return !controller.isInState(OPERATION_ENABLE);

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK " + "or RELEASE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateSocketAtStandbyReadSensorsNoPublication();
        publishData();
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is ABORTING action " + action.toString() + " within delay " + delay);
        FCSLOG.finer(() -> name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " is ENDING action " + action.toString());
        carousel.updateStateWithSensors();
        FCSLOG.finer(() -> name + " NOTHING BEING DONE HERE");
    }

    public void checkVelocity() {
        FCSLOG.info(name + " checking velocity;" + " velocity = " + controller.getVelocity());
        FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
                () -> Math.abs(controller.getVelocity()) < 20,
                () -> tcpProxy.updatePDOData(),
                name + " check controller velocity is around 0 after unlock ",
                name + ": controller velocity is too high after trying every 100ms during 500 ms",
                500,
                100);
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet. Not relevant to stop action clamps.");
    }

    @Override
    public EPOSController getController() {
        return controller;
    }


    public void updateVelocity() {
        if (isAtStandby()) {
            /* velocity is updated by PDO */
            velocity = controller.getVelocity();
        } else {
            velocity = 0;
        }
    }

    @Override
    public void postStart() {
        /* id is the last char of name */
        id = Integer.parseInt(name.substring(name.length() - 1));
        FCSLOG.info(name + " ID =" + id);
    }
}
