
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

import java.util.logging.Logger;

/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. The ACTION methods : open and close of this class is used in
 * ENGINEERING mode only when an operator may need to open or close only one
 * Latch. In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. On the autochanger there are 2
 * latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatch extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerLatch.class.getName());

    @LookupField(strategy = TREE)
    private Autochanger autochanger;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private final EPOSController latchController;
    private final ComplementarySensors filterEngagedSensors;
    private final ComplementarySensors closeSensors;
    private final ComplementarySensors openSensors;

    @ConfigurationParameter(description = "current to be sent to the controller to open the latch.", units = "mA", category = "autochanger")
    private volatile int currentToOpen = 300;

    @ConfigurationParameter(description = "timeout for opening or closing latch", units = "millisecond", category = "autochanger")
    private volatile long timeoutForLatchMotion = 3000;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    /**
     * Builds a AutochangerLatchModule with a controller and 3 sensors.
     *
     * @param latchController
     * @param closeSensors
     * @param openSensors
     * @param filterEngagedSensors
     */
    public AutochangerLatch(EPOSController latchController, ComplementarySensors closeSensors,
            ComplementarySensors openSensors, ComplementarySensors filterEngagedSensors) {
        this.latchController = latchController;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
        this.filterEngagedSensors = filterEngagedSensors;
    }

    /**
     * Return latchController
     *
     * @return
     */
    @Override
    public EPOSController getController() {
        return latchController;
    }

    /**
     * Returns true if latch is initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return latchController.isInitialized();
    }

    /**
     * For simulation, return currentToClose
     *
     * @return
     */
    public int getCurrentToClose() {
        return -currentToOpen;
    }

    /**
     * For simulation, return currentToClose
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return lockStatus. Used for simulation and tests.
     *
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=LOCKED Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if LOCKED. Doesn't read again sensors.", alias = "isClosed")
    public boolean isClosed() {
        return this.closeSensors.isOn();
    }

    /**
     * Returns true if LockStatus=UNLOCKED Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if UNLOCKED. Doesn't read again sensors.", alias = "isOpen")
    public boolean isOpened() {
        return this.openSensors.isOn();
    }

    /**
     * Returns true if LockStatus=ERROR, this means that openSensor and closeSensor
     * return non consistant values. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if in ERROR, this means that "
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    /**
     * Returns true if autochanger is empty.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if autochanger is empty. Doesn't read again sensors.")
    public boolean isEmpty() {
        return !this.filterEngagedSensors.isOn();
    }

    public boolean isFilterEngagedInError() {
        return filterEngagedSensors.isInError();
    }

    /**
     * Initialize field autochanger and latchController
     */
    @Override
    public void init() {
        if (latchController == null) {
            FCSLOG.severe(() -> name + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null latchController - fix groovy description file.");
        }

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(HARDWARE_ERROR.getAlert(latchController.getName()), alwaysClear);
        alertService.registerAlert(SDO_ERROR.getAlert(latchController.getName()), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerLatch.class, path);
        registerAction(OPEN);
        registerAction(CLOSE);
    }

    /**
     * Executed during INITIALIZATION phase. Initialize and check latchController.
     */
    @Override
    public void postStart() {
        FCSLOG.finer(() -> name + " BEGIN postStart.");
        if (latchController.isBooted()) {
            initializeController();
            latchController.updateEposState();
            latchController.publishData();
        }
        FCSLOG.finer(() -> name + " END postStart.");
    }

    private void initializeController() {
        try {
            latchController.initializeAndCheckHardware();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " could not initialize controller", latchController.getName(), ex);
        }
        try {
            latchController.changeMode(CURRENT);
        } catch (FcsHardwareException ex) {
            this.raiseWarning(HARDWARE_ERROR, name + " could not change mode of controller", latchController.getName(),
                    ex);
        }
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    protected void updateState() {
        boolean closed = this.closeSensors.isOn();
        boolean opened = this.openSensors.isOn();

        boolean someSensorsInError = this.filterEngagedSensors.isInError() || this.closeSensors.isInError()
                || this.openSensors.isInError();

        boolean inError = (closed && opened) || someSensorsInError;

        if (inError) {
            lockStatus = ERROR;
        } else if (closed) {
            lockStatus = CLOSED;
        } else if (opened) {
            lockStatus = OPENED;
        } else {
            lockStatus = INTRAVEL;
        }

        this.publishData();

    }

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update latch current in reading controller.")
    public void updateCurrent() {
        try {
            latchController.readCurrent();
            this.publishData();

            /*
             * we don't want to have an ALARM when monitor-current could not read current.
             */
        } catch (Exception ex) {
            raiseWarning(SDO_ERROR, " could not updateCurrent", latchController.getName(), ex);
        }
    }

    /**
     * Returns true if autochanger CANopen hardware is connected and ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if autochanger CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return latchController.isInitialized();
    }

    /**
     * Close this latch.
     *
     * @throws FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Close latch.")
    public void close() {
        try (AutoTimed at = new AutoTimed("ACLatch-close")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                FCSLOG.info(() -> name + " is already CLOSED. No action.");
                return;
            }
            autochanger.checkConditionsForClosingLatches();
            this.executeAction(CLOSE, timeoutForLatchMotion);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Close latch even if latch sensors are in error.")
    public void recoveryClose() {
        try (AutoTimed at = new AutoTimed("ACLatch-recoveryClose")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                FCSLOG.info(() -> name + " is already CLOSED. No action.");
                return;
            }
            autochanger.checkConditionsForClosingLatches(true);
            this.executeAction(CLOSE, timeoutForLatchMotion);
        }
    }

    /**
     * Open this latch.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Open latch.")
    public void open() {
        try (AutoTimed at = new AutoTimed("ACLatch-open")) {
            updateStateAndCheckSensors();
            if (isOpened()) {
                FCSLOG.info(() -> name + " is already OPENED. No action.");
                return;
            }
            autochanger.checkConditionsForOpeningLatches();
            this.executeAction(OPEN, timeoutForLatchMotion);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Open latch even if latch sensors are in error.")
    public void recoveryOpen() {
        try (AutoTimed at = new AutoTimed("ACLatch-recoveryOpen")) {
            updateStateAndCheckSensors();
            if (isOpened()) {
                FCSLOG.info(() -> name + " is already OPENED. No action.");
                return;
            }
            autochanger.checkConditionsForOpeningLatches(true);
            this.executeAction(OPEN, timeoutForLatchMotion);
        }
    }

    /**
     * Return true if action is completed with success.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                return lockStatus == OPENED;

            case CLOSE:
                return lockStatus == CLOSED;

            default:
                assert false : action;
        }
        return false;
    }

    /**
     * Reads sensors, updates state and checks if sensors are in error. If sensors
     * are in error, raises an ALERT.
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * Read sensors and update State. Used during an action.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
        latchController.updateDataDuringMotionFromSDO();
        latchController.publishDataDuringMotion();
    }

    /**
     * Start Action OPEN or CLOSE
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                latchController.enableAndWriteCurrent((short) getCurrentToOpen());
                break;

            case CLOSE:
                latchController.enableAndWriteCurrent((short) getCurrentToClose());
                break;

            default:
                assert false : action;
        }
    }

    /**
     * Stop action OPEN or CLOSE
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " STOPPING action " + action.toString() + " within delay " + delay);
        this.latchController.stopAction();
    }

    /**
     * Ending action OPEN or CLOSE
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " ENDING action " + action.toString());
        this.latchController.stopAction();
    }

    /**
     * @throws FcsHardwareException
     * @param action
     * @param delay
     */
    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        abortAction(action, delay);
    }

    /**
     * Creates an return an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setLockSensorOn(closeSensors.isOn());
        status.setUnlockSensorOn(openSensors.isOn());
        status.setFilterPresenceSensorOn(filterEngagedSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorsInError(closeSensors.isInError());
        status.setUnlockSensorsInError(openSensors.isInError());
        status.setFilterEngagedSensorsInError(filterEngagedSensors.isInError());
        status.setInError(lockStatus == LockStatus.ERROR);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = createStatusDataPublishedByLatch();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, status));
    }

}
