package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState;

import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 *
 * @author virieux
 */
public interface EPOSControllerWithBrake extends EPOSController {

    /**
     * For GUI. Doesn't read again controller CPU. Return true if controller is
     * enabled.
     *
     * @return
     */
    boolean isBrakeActivatedPub();

    void setBrakeActivatedPub(boolean brakeActivated);

    /**
     * activateBrakeAndDisable
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "activate brake to prevent motion and disable controller.")
    default void activateBrakeAndDisable() {
        activateBrake();
        // wait for brake to be mechanically activated.
        FcsUtils.sleep(20, getName());
        goToSwitchOnDisabled();
    }

    /**
     * Release holding brake to be able to move linear rail trucks or open/close
     * ONLINE clamps. Check that brake is released within a timeout of 500 ms.
     *
     * doReleaseBrake for controller consists in forcing to ONE bit 15 of parameter
     * DigitalOutputFonctionnalityState.
     *
     * It can't be a command because if this controller is a follower we have to check
     * if driver controller is enabled. See AutoChangerTwoTrucksModule.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    default void doReleaseBrake() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("doReleaseBrake-EPOS")) {
            int val = (int) readParameter(DigitalOutputFonctionnalityState);
            writeParameter(DigitalOutputFonctionnalityState, FcsUtils.force2one(val, 15));
            checkBrakeReleased();
            publishData();
            FCSLOG.info(getName() + ": brake released.");
        }
    }

    /**
     * enableAndReleaseBrake if this controller is a follower, command is rejected
     * because we have to check if driver controller is enabled. See
     * AutoChangerTwoTrucksModule.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Enable and release brake to permit motion. Not allowed in MASTER_ENCODER mode.")
    default void enableAndReleaseBrake() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("enableAndReleaseBrake-EPOS")) {
            // OK if not master_encoder
            if (this.isInMode(MASTER_ENCODER)) {
                throw new RejectedCommandException(getName() + " brake can't be released when in mode MASTER_ENCODER");
            }
            goToOperationEnable();
            doReleaseBrake();
        }
    }

    /**
     * Activate brake to prevent motion.
     *
     * A holding brake can be associated to a Controller. This brake is activated
     * when the controller is powered on.
     *
     * A brake is associated with Autochanger Linear Rail trucks controllers and
     * Autochanger Online Clamps controllers.
     *
     * activateBrake consists in forcing to ZERO bit 15 of parameter
     * DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Activate brake to prevent motion and check that brake is activated within a timeout of 500ms.")
    default void activateBrake() {
        try (AutoTimed at = new AutoTimed("activateBrake-EPOS")) {
            int val = (int) readParameter(DigitalOutputFonctionnalityState);
            writeParameter(DigitalOutputFonctionnalityState, FcsUtils.force2zero(val, 15));
            checkBrakeActivated();
            publishData();
            FCSLOG.info(getName() + ": brake activated.");
        }
    }

    /**
     * return true if brake if activated. This is when bit15 of
     * Parameter.DigitalOutputFonctionnalityState is ZERO.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return true if brake if activated.")
    default boolean isBrakeActivated() {
        int digitalOutput = (int) readParameter(DigitalOutputFonctionnalityState);
        // TODO check the 2 other outputs
        return ((digitalOutput >> 15) & 1) == 0;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check that controller brake is activated. "
            + "If brake is not activated within a timeout of 500ms, throw an Exception.")
    default void checkBrakeActivated() {
        long timeoutMillis = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean state_ok = isBrakeActivated();
        while (!state_ok && duration <= timeoutMillis) {
            duration = System.currentTimeMillis() - timeStart;
            FcsUtils.sleep(5, getName());
            state_ok = isBrakeActivated();
        }
        if (state_ok) {
            FCSLOG.info(getName() + String.format(" brake activation duration = %d", duration));
            setBrakeActivatedPub(true);
        } else {
            String msg = getName()
                    + String.format(" couldn't activate brake during time allocated of %d ms", timeoutMillis);
            FCSLOG.error(msg);
            setBrakeActivatedPub(false);
            throw new FcsHardwareException(msg);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check that controller brake is released. "
            + "If brake is not released within a timeout of 500ms, throw an Exception.")
    default void checkBrakeReleased() {
        long timeoutMillis = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean state_ok = !isBrakeActivated();
        while (!state_ok && duration <= timeoutMillis) {
            duration = System.currentTimeMillis() - timeStart;
            FcsUtils.sleep(5, getName());
            state_ok = !isBrakeActivated();
        }
        if (state_ok) {
            FCSLOG.info(getName() + String.format(" brake release duration = %d", duration));
            setBrakeActivatedPub(false);
        } else {
            String msg = getName()
                    + String.format(" couldn't release brake during time allocated of %d ms", timeoutMillis);
            FCSLOG.error(msg);
            setBrakeActivatedPub(true);
            throw new FcsHardwareException(msg);
        }
    }

    @Override
    default StatusDataPublishedByEPOSController createStatusDataPublishedByEPOSController() {
        StatusDataPublishedByEPOSController status = new StatusDataPublishedByEPOSController(isBooted(),
                isInitialized(), isInError(), getErrorRegister(), getErrorHistoryNB(), getLastErrorCode(),
                getLastErrorName());
        status.setMode(getMode());
        status.setState(getEposState());
        status.setCurrent(getCurrent());
        status.setPosition(getPosition());
        status.setVelocity(getVelocity());
        status.setProfileAcceleration(getProfileAcceleration());
        status.setProfileDeceleration(getProfileDeceleration());
        status.setProfileVelocity(getProfileVelocity());
        status.setAverageCurrent(getAverageCurrent());
        status.setFollowingError(getFollowingError());
        status.setControllerWithBrake(true);
        /**
         * it's a bad idea to replace line below by :
         * status.setBrakeActivated(isBrakeActivated()) because this method is
         * called when subsystem starts and a read error on CANbus would prevent
         * the subsystem to complete the boot.
         */
        status.setBrakeActivated(isBrakeActivatedPub());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    default void publishData() {
        StatusDataPublishedByEPOSController status = createStatusDataPublishedByEPOSController();
        FCSLOG.info(getName() + " is publishing:" + status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), status));
    }

}
