
package org.lsst.ccs.subsystems.fcs.common;

import java.time.Instant;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;

/**
 * This interface is implemented by each class that can raises an Alert.
 * @author virieux
 */
public interface AlertRaiser {
    /**
     * to separate Alert Name or ID from device name.
     */
    char alertSeparator = '/';

    Subsystem getSubsystem();

    /**
     * return alertService
     * @return
     */
    AlertService getAlertService();

    default char getAlertSeparator() {
        return alertSeparator;
    }


    /**
     * Raises an ALARM with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause) {
        getAlertService().raiseAlert(fcsAlert.getAlert(), ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and an Exception.
     * @param fcsAlert
     * @param cause
     * @param ex
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, Exception ex) {
        FCSLOG.error(ex);
        raiseAlarm(fcsAlert, cause + " due to " + ex);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(fcsAlert.getAlert(deviceName), ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause, a deviceName and an Exception
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, String deviceName, Exception ex) {
        raiseAlarm(fcsAlert, cause + " due to " + ex, deviceName);
    }

    /**
     * Clear Alarm
     * @param fcsAlert
     * @param deviceName
     */
    default void clearAlarm(FcsAlert fcsAlert, String deviceName) {
        getAlertService().clearAlerts(fcsAlert.getAlert(deviceName).getAlertId());
    }

    /**
     * Raises an WARNING with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause) {
        getAlertService().raiseAlert(fcsAlert.getAlert(), WARNING, cause);
    }

    /**
     * Raises a WARNING with a FcsAlert, a cause and an Exception.
     * @param fcsAlert
     * @param cause
     * @param ex
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, Exception ex) {
        FCSLOG.error(ex);
        raiseWarning(fcsAlert, cause + " due to " + ex);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(fcsAlert.getAlert(deviceName), WARNING, cause);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause, a deviceName and an exception.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, String deviceName, Exception ex) {
        FCSLOG.warning(ex);
        raiseWarning(fcsAlert, cause + " due to " + ex, deviceName);
    }

    /**
     * raise an alert only if it's a new alert. this method has to be used in
     * periodic tasks to avoid having a too big volume of raised alerts.
     *
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param state
     */
    default void raiseAlertOnlyIfNew(FcsAlert fcsAlert, String cause, String deviceName, AlertState state) {
        Alert alert = fcsAlert.getAlert(deviceName);
        String alertId = alert.getAlertId();
        RaisedAlertHistory history = getAlertService().getRaisedAlertSummary().getRaisedAlert(alertId);
        if (history == null || history.getLatestAlertState() != state) {
            getAlertService().raiseAlert(alert, state, cause);
        }
    }

    default void raiseAlertOnlyEveryTenMinutes(FcsAlert fcsAlert, String cause, String deviceName, AlertState state) {
        Alert alert = fcsAlert.getAlert(deviceName);
        String alertId = alert.getAlertId();
        boolean alertToBeRaised = true;
        RaisedAlertHistory history = getAlertService().getRaisedAlertSummary().getRaisedAlert(alertId);
        if (history != null) {
            if (history.getLatestAlertState() == state) {
                Instant lastAlertInstant = history.getLatestAlertCCSTimeStamp().getUTCInstant();
                alertToBeRaised = Instant.now().minusMillis(600000).isAfter(lastAlertInstant);
            }
        }
        if (alertToBeRaised) {
            getAlertService().raiseAlert(alert, state, cause);
        }
    }

    default void raiseAlarmOnlyIfNew(FcsAlert fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyIfNew(fcsAlert, cause, deviceName, ALARM);
        FCSLOG.severe(deviceName + ": " + cause);
    }

    default void raiseWarningOnlyIfNew(FcsAlert fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyIfNew(fcsAlert, cause, deviceName, WARNING);
        FCSLOG.warning(deviceName + ": " + fcsAlert + ": " + cause);
    }

    default void raiseWarningOnlyEveryTenMinutes(FcsAlert fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyEveryTenMinutes(fcsAlert, cause, deviceName, WARNING);
        FCSLOG.warning(deviceName + ": " + fcsAlert + ": " + cause);
    }

    default void raiseAlertOnlyEveryTenMinutes(FcsAlert fcsAlert, String cause, String deviceName, Exception ex) {
        raiseAlertOnlyEveryTenMinutes(fcsAlert, cause + " due to " + ex, deviceName, WARNING);
    }

}
