package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.aiousb.USB_DIO_96;
import org.lsst.ccs.drivers.aiousb.USB_DIO_96_Interface;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is the Main Module for the Loader control software when the loader is in
 * standalone.
 *
 *
 * @author virieux
 */
public class LoaderMain extends MainModule implements USB_DIO_96_Interface {

    @LookupField(strategy = Strategy.CHILDREN, pathFilter = "loader")
    private Loader loader;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter = "autochanger")
    private FilterHolder autochanger;

    private USB_DIO_96_Interface usbdio96Card;

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make LoaderGUI listen to my
         * subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.loader", "loader");
        super.init();
    }

    /**
     * Update loader state in reading the sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update loader state in reading the sensors.")
    @Override
    public void updateStateWithSensors() {
        loader.updateStateWithSensors();
        autochanger.updateStateWithSensors();
    }


    @Override
    public void initializeHardware() {
        loader.initializeHardware();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Connect the loader hardware.")
    public void connectLoader() {
        this.bridge.connectHardware();
        loader.postStart();
    }

    /**
     * In Standalone mode, we want to connect Loader when the subsystem starts.
     */
    @Override
    public void postStart() {
        connectLoader();
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Print the list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(this.bridge.printHardwareState());
        sb.append("\n ");
        sb.append(loader.printHardwareState());
        return sb.toString();
    }

    /**
     * Connect to the card USB_DIO_96 card.
     */
    public void connectUSB_DIO_96card() {
        if (usbdio96Card == null) {
            usbdio96Card = new USB_DIO_96();
            try {
                ((USB_DIO_96) usbdio96Card).connect();
            } catch (DriverException ex) {
                throw new FcsHardwareException("Could not connect to DIO_USB_96. "
                        + "Check that a DIO_USB_96 is present on this HCU and connected throw a USB cable.", ex);
            } catch (Exception ex) {
                throw new FcsHardwareException("Could not connect to DIO_USB_96. "
                        + " because : " + ex.toString());
            }
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n").append(loader.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Print PLC loader signals read on card USB_DIO_96.")
    public String printUSBDIO96Signals() {
        connectUSB_DIO_96card();
        return usbdio96Card.toString();
    }

    /**
     * Simulate that there is no filter at HANDOFF on the test bench : AP2 & AF0
     *
     * Signals to simulate : NOPERMIT=0, ENG=0, AP2=1, AF0=1, AF1=1, AF3=0
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Simulate PLC signals that say there is no filter at HANDOFF (AP2 & AF0 & NOLOCKOUT).")
    @Override
    public void simulateAutochangerEmpty() {
        connectUSB_DIO_96card();
        usbdio96Card.simulateAutochangerEmpty();
        FcsUtils.sleep(20, name);
        updateStateWithSensors();
    }

    /**
     * Simulate that a filter is at HANDOFF on the test bench and is held : AP2
     * & AF3
     *
     * Signals to simulate : NOPERMIT=0, ENG=0, AP2=1, AF0=0, AF1=0, AF3=1
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Simulate PLC signals that say there is a filter at HANDOFF and it is held (AP2 & AF3 & NOLOCKOUT).")
    @Override
    public void simulateAutochangerHoldFilter() {
        connectUSB_DIO_96card();
        usbdio96Card.simulateAutochangerHoldFilter();
        FcsUtils.sleep(20, name);
        updateStateWithSensors();
    }

    /**
     * Load a filter into bench or storage box.
     *
     * This command simulates signals coming from autochanger.
     * To be used in standalone mode only.
     *
     * At the end of this command, the loader carrier is empty
     * at storage position.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Load a filter into bench or storage box. Simulate signals coming from autochanger.")
    public void loadFilterIntoBenchOrStorageBox() {
        simulateAutochangerEmpty();
        if (loader.isClampedOnFilter() || loader.isClosedOnFilter()) {
            loader.moveFilterToHandoff();
        }
        simulateAutochangerHoldFilter();
        loader.openClampAndMoveEmptyToStorage();
    }

    /**
     * Unload a filter from bench or storage box.
     *
     * This command simulates signals coming from autochanger.
     * To be used in standalone mode only.
     *
     * At the end of this command, hooks are CLOSED on filter,
     * and carrier is at STORAGE position.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unload a filter from bench or storage box.")
    public void unloadFilterFromBenchOrStorageBox() {
        simulateAutochangerHoldFilter();
        if (loader.isEmpty()) {
            loader.moveEmptyToHandoffAndClose();
        }
        simulateAutochangerEmpty();
        loader.moveFilterToStorage();
    }

}
