
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HasLifecycle;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * This class is a model for the fake autochanger for loader-standalone and fake
 * loader for autochanger-standalone.
 *
 * A numeric sensor is setup to simulate that the autochanger or loader holds
 * the filter at STANDBY or HANDOFF position. Those sensors can be emulated with
 * the card AIOUSB.
 *
 * In the whole FCS, carousel, autochanger and loader implements FilterHolder
 * too.
 *
 * @author virieux
 */
public class FakeFilterHolder implements FilterHolder, HasLifecycle {

    @LookupField(strategy = TOP)
    private Subsystem subs;

    private BinarySensor holdingFilterSensor;
    private BinarySensor notHoldingFilterSensor;

    private String holdingFilterSensorName;

    private String notHoldingFilterSensorName;

    @LookupName
    private String name;

    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the fake autochanger is holding the filter. "
            + "This command doesn't read again the sensor.")
    @Override
    public boolean isHoldingFilter() {
        return this.holdingFilterSensor.isOn();
    }

    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the fake autochanger is holding the filter. "
            + "This command doesn't read again the sensor.")
    @Override
    public boolean isNotHoldingFilter() {
        return this.notHoldingFilterSensor.isOn();
    }

    /**
     * This methods updates the field holdingFilter in reading the holding filter
     * sensor. Then it published status data on the STATUS bus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        KeyValueData kvd;
        kvd = new KeyValueData(name, holdingFilterSensor.isOn());
        subs.publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at HANDOFF.
     * Idem for fakeCarousel and fakeLoader.
     * @return
     */
    @Override
    public boolean isAtHandoff() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at STANDBY.
     * Idem for fakeCarousel and fakeLoader.
     *
     * @return
     */
    @Override
    public boolean isAtStandby() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position
     * has not a real meaning. The autochanger is supposed to be at STANDBY.
     *
     * @return
     */
    @Override
    public boolean isAtOnline() {
        return false;
    }

    @Override
    public void start() {
        FCSLOG.info(name + " FakeFilterHolder BEGIN start");
        ComponentLookup lookup = subs.getComponentLookup();
        this.holdingFilterSensor = (BinarySensor) lookup.getComponentByPath(holdingFilterSensorName); 
        if ( notHoldingFilterSensorName != null ) {
            this.notHoldingFilterSensor = (BinarySensor) lookup.getComponentByPath(notHoldingFilterSensorName);
        }
        FCSLOG.info(name + " FakeFilterHolder END start");
    }

    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart");
        FCSLOG.info(name + " END postStart");
    }

    @Override
    public int getFilterID() {
        return 0;
    }

}
