package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.OPERATION_ENABLE;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.CLAMPEDONFILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.READYTOCLAMP;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNCLAMPEDEMPTY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNCLAMPEDONFILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.UNDEFINED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.NOFILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.RELEASECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCKCLAMPS;

import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.IOModuleStatus;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;
import org.lsst.ccs.subsystems.fcs.utils.TTC580Utils;

/**
 * This is a socket on the carousel : there is 5 sockets on a carousel. When a
 * filter is on the carousel, it is attached at a socket. A socket has a clamp
 * on each side : - clampX- (clampXminus) - clampX+ (clampXplus) clampX+ and
 * clampX- are the same except for the method isLocked() See
 * FilterClampXminusModule and FilterClampXplusModule
 *
 * The main job of this class CarouselSocket is to synchronize the actions on
 * the 2 clamps. The state of the CarouselSocket is the state of the clamps when
 * both clamps state are identical.
 *
 * During an action on both clamps, we want to lock the CarouselSocket so no
 * other thread can try to access this object before the completion of action.
 * We do that with a Lock and Condition. (see java.util.concurrent).
 *
 * CarouselSocket extends MobileItem because we want to lock or release the
 * clamps when the carousel is at standby position and we want to wait until the
 * action is completed. We know if the action is completed in reading the clamps
 * sensors. So we use the general mechanism provided by the MobileItem.
 *
 *
 * @author virieux
 *
 */
public class CarouselSocket extends MobileItem {

    private int id;

    @LookupField(strategy = TREE)
    protected Carousel carousel;

    private CarouselClamp clampXminus;

    private CarouselClamp clampXplus;

    @LookupField(strategy = TREE, pathFilter=".*\\/clampXminusController")
    private EPOSController clampXminusController;

    @LookupField(strategy = TREE, pathFilter=".*\\/clampXplusController")
    private EPOSController clampXplusController;

    @LookupField(strategy = CHILDREN)
    protected SensorPluggedOnTTC580 ioModuleSensor;

    protected IOModuleStatus ioModuleStatus;

    /*To be able to know if the autochanger holds a filter. */
    /*This can't be a Autochanger because when the carousel is in Standalone mode
     /*there is not autochanger.
     /*FilterHolder is an Interface. */
    @LookupField(strategy = TREE, pathFilter="autochanger")
    protected FilterHolder autochanger;

    @LookupField(strategy = TREE)
    private FilterManager filterManager;

    @LookupField(strategy = TREE)
    private PersistencyService persistenceService;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private final boolean loadFilterLocationAtStartup = true;
    private final boolean saveFilterLocationAtShutdown = true;

    /**
     * carousel positionOnCarousel (angle) when this socket is at standby
     * standbyPosition should be : 360 - positionOnCarousel, but could be different
     */
    @ConfigurationParameter(range = "-4500000..4500000", description = " carousel position when this socket is at standby [carousel encoder step]", units = "unitless", category = "carousel")
    private volatile int standbyPosition;

    private int deltaPosition = 0;

    @Persist
    public volatile int filterID;

    private volatile FilterClampState clampsState = UNDEFINED;

    private long timeoutForUnlocking = 4000;
    private long timeoutForReleasing = 4000;

    private boolean available = true;

    /**
     * Build a new CarouselSocket with a tickMillis of 5000.
     *
     * @param id
     * @param clampXminus
     * @param clampXplus
     */
    public CarouselSocket(int id, CarouselClamp clampXminus, CarouselClamp clampXplus) {
        this.id = id;
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarouselSocket.class, path);
        registerAction(UNLOCKCLAMPS);
        registerAction(RELEASECLAMPS);
    }

    public int getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    public CarouselClamp getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClamp clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClamp getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClamp clampXplus) {
        this.clampXplus = clampXplus;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns filterID if a filter is in the socket.")
    public int getFilterID() {
        return filterID;
    }

    /**
     *
     * @return NO_FILTER if no filter at standby or filter name
     */
    public String getFilterName() {
        return filterManager.getFilterNameByID(filterID);
    }

    public void setFilterID(int filterID) {
        this.filterID = filterID;
        publishData();
    }

    public int getStandbyPosition() {
        return standbyPosition;
    }

    public int getDeltaPosition() {
        return deltaPosition;
    }

    public IOModuleStatus getIOModuleStatus() {
        return ioModuleStatus;
    }

    /**
     * Return true if this socket is at STANDBY position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if this socket is at STANDBY position on the carousel.")
    public boolean isAtStandby() {
        return carousel.getSocketAtStandbyID() == id;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return true if socket is ready to use or false if not.")
    public boolean isAvailable() {
        return available;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Set boolean available to true if socket is ready to use or false if not.")
    public void setAvailable(boolean available) {
        this.available = available;
        clampXminus.setAvailable(available);
        clampXplus.setAvailable(available);
    }

    /**
     * ***********************************************************************************************
     * END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        persistenceService.setAutomatic(loadFilterLocationAtStartup, saveFilterLocationAtShutdown);


        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
            // TODO check that alert is no more valid (sensors are OK / IO module is OK,
            // etc...)
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(CA_SENSOR_ERROR.getAlert(name), alwaysClear);
        
        this.clampXminus.setController(clampXminusController);
        this.clampXplus.setController(clampXplusController);
    }

    /**
     * Checking if autochanger is holding the filter is delegated to autochanger.
     *
     * @return true if autochanger is at STANDBY and holds a filter.
     */
    private boolean isAutochangerHoldingFilter() {
        return autochanger.isAtStandby() && autochanger.isHoldingFilter();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return clampXminusController.isInitialized() && clampXplusController.isInitialized();
    }

    /**
     * Returns the state of the clamps. If the state is being updated and waiting
     * for a response from a sensor, this methods waits until the state is updated.
     * If the state is not being updated, it returns immediatly the state.
     *
     * @return state
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return clamps state - doesn't read again sensors.")
    public FilterClampState getClampsState() {
        return clampsState;
    }

    /**
     * update state from sensors values.
     *
     */
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-carouselSocket")) {

            if (ioModuleSensor.getValue() >= 1 && ioModuleSensor.getValue() <= 7) {
                ioModuleStatus = IOModuleStatus.getStatusByCode(ioModuleSensor.getValue());
            } else {
                ioModuleStatus = IOModuleStatus.UNKNOWN_STATUS;
            }
            if (ioModuleSensor.getValue() < 4 && ioModuleSensor.getValue() > 0) {
                updateClampsState();
            } else {
                clampsState = UNDEFINED;
                // TODO: Has to be raiseAlarm for final carousel.
                this.raiseWarningOnlyIfNew(CA_SENSOR_ERROR,
                        " IO module status = " + ioModuleSensor.getValue() + " => " + ioModuleStatus, name);
            }
            updateDeltaPosition();
            publishData();
        }
    }

    public synchronized void updateDeltaPosition() {
        deltaPosition = carousel.getPosition() - this.standbyPosition % carousel.getFullTurn();
    }

    private void updateClampsState() {
        clampXminus.updateState();
        clampXplus.updateState();
        boolean inError = clampXminus.getClampState() == FilterClampState.ERROR
                || clampXplus.getClampState() == FilterClampState.ERROR;

        if (clampXminus.getClampState() == clampXplus.getClampState()) {
            this.clampsState = clampXminus.getClampState();
            FCSLOG.info(name + " : clamps are updated");

        } else if (clampXminus.getClampState() == UNCLAMPEDEMPTY && clampXplus.getClampState() == READYTOCLAMP) {
            if (!clampXminus.getController().isInState(OPERATION_ENABLE)
                    && !clampXplus.getController().isInState(OPERATION_ENABLE)) {
                this.clampsState = READYTOCLAMP;

            } else {
                this.clampsState = UNCLAMPEDEMPTY;
            }

        } else if (!inError) {
            FCSLOG.fine(name + " clampXminus is " + clampXminus.getClampState() + " and clampXplus is "
                    + clampXplus.getClampState() + " clampState = " + UNDEFINED);
            this.clampsState = UNDEFINED;

        } else {
            this.clampsState = FilterClampState.ERROR;
            String msg = name + ": clampState at Xminus side is different from clampState at Xplus side. ";
            msg = msg + clampXminus.getName() + " state=" + clampXminus.getClampState() + " ";
            msg = msg + clampXplus.getName() + " state=" + clampXplus.getClampState();
            this.raiseWarningOnlyIfNew(CA_SENSOR_ERROR, msg, name);
        }
        /* no alarm is raised anymore in updateState so we have to raise alarm now */
        clampXminus.getSensorErrorCounter().forEach((msg) -> {
            raiseAlarm(CA_SENSOR_ERROR, (String) msg, name);
        });
        clampXplus.getSensorErrorCounter().forEach((msg) -> {
            raiseAlarm(CA_SENSOR_ERROR, (String) msg, name);
        });
    }

    /**
     * update field filterID to be used only with whole FCS. In standalone mode,
     * filters can't be removed neither loaded into carousel so filterID never
     * changes.
     */
    public void updateFilterID() {
        if (this.isEmpty()) {
            filterID = 0;
        } else if (autochanger.isAtStandby()) {
            filterID = autochanger.getFilterID();
        }
        FCSLOG.info(name + " filter ID = " + carousel.getSocketAtStandby().getFilterID());
    }

    /**
     * check and update lock sensor offset1 for the 2 clamps
     *
     * @param sdo
     */
    public void checkAndUpdateOffset1(long sdo) {
        clampXminus.checkAndUpdateOffset1(TTC580Utils.getOffset1Xminus(sdo));
        clampXplus.checkAndUpdateOffset1(TTC580Utils.getOffset1Xplus(sdo));
    }

    /**
     * check and update presence sensor offset2 for the 2 clamps
     *
     * @param sdo
     */
    public void checkAndUpdateOffset2(long sdo) {
        clampXminus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xminus(sdo));
        clampXplus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xplus(sdo));
    }

    /*
     * The carousel socket is empty if there is no filter in the socket. That
     * appends when clamps of both side of the socket are empty. A clamp is empty
     * when no filter is engaged in the clamp.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is no filter in the socket.")
    public boolean isEmpty() {
        return clampXminus.getFilterPresenceStatus() == NOFILTER && clampXplus.getFilterPresenceStatus() == NOFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are LOCKED.")
    public boolean isClampedOnFilter() {
        return clampsState == CLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedOnFilter() {
        return clampsState == UNCLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is NO filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedEmpty() {
        return clampsState == UNCLAMPEDEMPTY;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if this socket is ready to clamp a filter.")
    public boolean isReadyToClamp() {
        return clampsState == READYTOCLAMP;
    }

    /**
     * Releases the 2 clamps of the socket.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Release the 2 clamps of this socket if the socket is at STANDBY position.")
    public void releaseClamps() {
        try (AutoTimed at = new AutoTimed("socket-releaseClamps")) {
            carousel.updateStateWithSensors();
            FCSLOG.info("Checking conditions for release clamp " + name + " on socket " + "at standby position.");
            if (!this.isAtStandby()) {
                throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
            }
            if (clampXminus.isFilterEngaged() || clampXplus.isFilterEngaged()) {
                throw new RejectedCommandException(
                        name + ": Can't release clamps because a filter is engaged in clamps.");
            }
            this.timeoutForReleasing = java.lang.Math.max(clampXminus.timeoutForReleasing,
                    clampXplus.timeoutForReleasing);
            FCSLOG.info(name + ": Releasing clamps at standby position.");
            this.executeAction(RELEASECLAMPS, timeoutForReleasing);
        }
    }

    /**
     * This method unclamps the 2 clamps at standby position.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Unlock the 2 clamps of this socket if the socket is at STANDBY position.")
    public void unlockClamps() {
        if (!carousel.isUnclampAllowedByPLC()) {
            throw new RejectedCommandException(name + " PLC does not allow to unlock clamps. Ckeck PLCCarouselPanel.");
        }
        try (AutoTimed at = new AutoTimed("socket-unlockClamps")) {
            carousel.updateStateWithSensors();
            if (!this.isAtStandby()) {
                throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
            }
            if (!this.isEmpty() && !this.isAutochangerHoldingFilter()) {
                throw new RejectedCommandException(
                        name + "cannot unlock clamps if NOT EMPTY and FILTER is NOT HELD by autochanger.");
            }
            this.timeoutForUnlocking = java.lang.Math.max(this.clampXminus.timeoutForUnlocking,
                    this.clampXplus.timeoutForUnlocking);
            this.executeAction(UNLOCKCLAMPS, timeoutForUnlocking);
            FCSLOG.info("Command unlockClamps completed");
            this.publishData();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns a String representation of a CarouselSocket.")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append(",FilterID in socket : ");
        sb.append(filterID).append("\n");
        return sb.toString();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == UNLOCKCLAMPS) {
            return this.clampsState == UNCLAMPEDONFILTER;

        } else if (action == RELEASECLAMPS) {
            return this.clampsState == READYTOCLAMP;

        } else {
            throw new IllegalArgumentException("Action on clamps on socket must " + "be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateSocketAtStandbyReadSensorsNoPublication();
        publishData();
    }

    /**
     * Start action of unlocking or releasing clamps.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(MobileItemAction action) {
        if (action == UNLOCKCLAMPS) {
            this.clampXminusController.goToOperationEnable();
            this.clampXplusController.goToOperationEnable();

            // Following a request from Guillaume Daubard, we first send a small current
            // to the clamp to prepare it and prevent an unstable state when we send the
            // full current to unlock.
            this.clampXminusController.writeCurrent((short) carousel.getCurrentToPrepareUnlock());
            this.clampXplusController.writeCurrent((short) carousel.getCurrentToPrepareUnlock());

            // tempo de 200ms pour la mécanique demandé par Guillaume.
            FcsUtils.sleep(200, getName());

            this.clampXminusController.writeCurrent((short) this.clampXminus.getCurrentToUnlock());
            this.clampXplusController.writeCurrent((short) this.clampXplus.getCurrentToUnlock());

        } else if (action == RELEASECLAMPS) {
            this.clampXminusController.goToSwitchOnDisabled();
            this.clampXplusController.goToSwitchOnDisabled();

        } else {
            throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        // TODO complete this
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        FCSLOG.debug(name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        carousel.updateStateWithSensors();
        FCSLOG.debug(name + " NOTHING BEING DONE HERE IF ACTION IS COMPLETED BUT IF TIMEOUT WE HAVE TO DISABLE CONTROLLERS");
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByCarouselSocket()));
    }

    /**
     * Create an object to be published on the STATUS bus by a CarouselSocket.
     *
     * @return
     */
    public StatusDataPublishedByCarouselSocket createStatusDataPublishedByCarouselSocket() {
        StatusDataPublishedByCarouselSocket status = new StatusDataPublishedByCarouselSocket();
        status.setAtStandby(this.isAtStandby());
        status.setIOStatus(ioModuleStatus);
        status.setClampsState(clampsState);
        status.setEmpty(this.isEmpty());
        status.setFilterID(filterID);
        status.setSocketID(id);
        status.setAvailable(available);
        if (this.isEmpty()) {
            status.setFilterName(NO_FILTER);
        } else {
            status.setFilterName(getFilterName());
        }
        return status;
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void shutdown() {
        super.shutdown();
        /* to save filter location in a file */
        persistenceService.persistNow();
    }

}
