
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.CarouselPLCSensor;
import org.lsst.ccs.subsystems.fcs.CarouselClampSensor;
import org.lsst.ccs.subsystems.fcs.CarouselSensor;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHYTTC580;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.utils.TTC580Utils;
import static org.lsst.ccs.subsystems.fcs.utils.TTC580Utils.*;

/**
 * A model for the CANopen device to read carousel sensors. This sensors are
 * read by PDO. (in response to a command "sync" sent to the CANbus)
 *
 * See on Confluence the description of data read on hyTTC580 :
 *
 * https://confluence.slac.stanford.edu/pages/viewpage.action?spaceKey=LSSTCAM&title=CAN+Open+communication+with+carousel+modules
 *
 * @author virieux
 */
public class CanOpenTTC580 extends CanOpenDevice implements TTC580Interface {

    public static final String LOCKSENSOR = "lockSensor";
    public static final String FILTERPRESENCE = "filterPresence";
    public static final String IO_MODULE_SENSOR = "ioModuleSensor";
    public static final String XMINUS = "Xminus";
    public static final String XPLUS = "Xplus";

    protected int cobid1;
    protected int cobid2;
    protected int cobid4;
 
    @LookupField(strategy = TREE)
    private AlertService alertService;

 
    /**
     * map of sensors attached to this device.
     */
    @LookupField(strategy = TREE)
    protected Map<String, SensorPluggedOnTTC580> sensorsMap = new HashMap<>();
    
    protected Map<String, CarouselPLCSensor> plcSensorsMap = new HashMap<>();

    /**
     * PDO1 represents sensors value for socket at STANDBY. pdo1 is coded on 64bits
     * and represents from right to left :
     * - 3 bits for socket identification
     * - 3 bits unused
     * - 2 bits unused
     * - 4 bits for State of filter at standby for clampXminus
     * - 4 bits for State of filter at standby for clampXplus
     * - 12 bits for ClampXminus lockSensor value
     * - 12 bits for ClampXminus filterPresenceSensor value
     * - 12 bits for ClampXplus lockSensor value
     * - 12 bits for ClampXplus filterPresenceSensor value
     *
     */
    protected long pdo1 = 0L;

    /**
     * PDO2 represents the same data than PDO1 but for another socket, not at
     * standby.
     *
     * - 3 bits for socket identification
     *
     * - 3 bits for io module status
     *
     * - 2 bits for socket safety switch states
     *
     * - 4 bits for clampXminus State
     *
     * - 4 bits for clampXplus State
     *
     * - 12 bits for ClampXminus lockSensor value
     *
     * - 12 bits for ClampXminus filterPresenceSensor value
     *
     * - 12 bits for ClampXplus lockSensor value
     *
     * - 12 bits for ClampXplus filterPresenceSensor value
     *
     *
     */
    protected long pdo2 = 0L;

    protected long pdo4 = 0L;

    public int getCobid1() {
        return cobid1;
    }

    public int getCobid2() {
        return cobid2;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByHYTTC580.class, path);
    }

    @Override
    public void init() {
        
        cobid1 = this.nodeID + 0x180;
        cobid2 = this.nodeID + 0x280;
        cobid4 = this.nodeID + 0x480;
                
        FCSLOG.info(name + " =====> initialization of my sensors map");
        
        if (sensorsMap.isEmpty()) {
            FCSLOG.error("NO SENSORS ARE FOUND FOR " + name);
            return;
        }
        StringBuilder sb = new StringBuilder();
        for (Iterator<SensorPluggedOnTTC580> it = sensorsMap.values().iterator(); it.hasNext();) {
            SensorPluggedOnTTC580 sensor = it.next();
            sb.append(sensor.getName());
            sb.append("====>");
            if (name.equals(sensor.getDeviceName())) {
                sb.append(" IS MY SENSOR ");
                sb.append(sensor.toString());
                sb.append("#\n");
            } else {
                it.remove();
            }
        }
        FCSLOG.info(sb.toString());
        FCSLOG.info(name + " =====> MY SENSORS \n" + sensorsMap.toString()); 
        /* Creation of a map with only CarouselPLCSensor*/
        sensorsMap.values().stream().filter((sensor) -> (sensor instanceof CarouselPLCSensor)).forEach((sensor) -> {
            plcSensorsMap.put(sensor.getName(), (CarouselPLCSensor) sensor);
        });
        
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR.getAlert(), alwaysClear);        
    }

    /**
     *
     * @return
     */
    @Override
    public long getPdo1() {
        return pdo1;
    }

    /**
     *
     * @param pdoVal
     */
    public void setPdo1(long pdoVal) {
        this.pdo1 = pdoVal;
    }

    @Override
    public long getPdo2() {
        return pdo2;
    }

    public void setPdo2(long pdo2) {
        this.pdo2 = pdo2;
    }

    @Override
    public long getPdo4() {
        return pdo4;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
        tcpProxy.addReceivedPDO(cobid4);
        initialized = true;
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        long beginTime = System.currentTimeMillis();
        FCSLOG.finest(name + " updatingFromPDO = " + pdo);
        boolean updated = false;
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
            updated = true;
            /* update from PDO1 */
            short socketID1 = getSocketId(pdo1);
            FCSLOG.finest(name + " updatingFromPDO1 = " + pdo1 + " binaire:" + Long.toBinaryString(pdo1));
            if (socketID1 == 0) {
                FCSLOG.finest(name + " no socket in STANDBY position");
            } else if (socketID1 == 7) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " error reading socketID for pdo1");
            } else if (socketID1 >= 1 && socketID1 <= 5) {
                /*
                 * The value of 1 for the io module status indicates the socket is in STANDBY
                 * position
                 */
                ((CarouselSensor) sensorsMap.get("carousel/socket"+socketID1+"/"+IO_MODULE_SENSOR + socketID1)).updateValue(1);
                updateSocketSensors(socketID1, pdo1);
            }
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
            updated = true;
            if (pdo2 == 0) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " pdo2 can't be 0 " + this.toString());
            } else {
                int ioModuleStatus = getIOModuleStatus(pdo2);
                /* ioModuleStatus not in safe state : update from PDO2 */
                if (ioModuleStatus != 4) {
                    FCSLOG.finest(name + " updatingFromPDO2 = " + pdo2 + " binaire:" + Long.toBinaryString(pdo2));
                    short socketID2 = getSocketId(pdo2);
                    if (socketID2 == 0) {
                        this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " socketID for pdo2 can't be 0");
                    } else if (socketID2 == 7) {
                        this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " error reading socketID for pdo2");
                    } else if (socketID2 >= 1 && socketID2 <= 5) {
                        ((CarouselSensor) sensorsMap.get("carousel/socket"+socketID2+"/"+IO_MODULE_SENSOR + socketID2)).updateValue(ioModuleStatus);
                        updateSocketSensors(socketID2, pdo2);
                    }
                } else {
                    FCSLOG.finest(name + ", pdo2 " + Long.toBinaryString(pdo2) + ": loss of communication with the TTC30s");
                }
            }
        }
        /* data for PLC debug */
        if (pdo.getPDOs().containsKey(cobid4)) {
            pdo4 = (long) pdo.getPDOs().get(cobid4);
            FCSLOG.info(name + " updateFromPDO pdo4 = " + pdo4);
            updated = true;
            if (pdo4 == 0) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " pdo4 can't be 0 " + this.toString());
            } else {
                updatePLCSensors(pdo4);
            }
        }
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateFromPDO duration = " + duration);
        if (updated) {
            this.publishData();
        }
    }

    /**
     * update the clamps sensors from pdoValue for a carousel socket given by its id.
     *
     * @param socketID
     * @param pdoValue
     */
    protected void updateSocketSensors(short socketID, long pdoValue) {
        /* carousel clamp sensors are CarouselClampSensor because of these 2 lines of code in CarouselClamp :*/
        /* private final SensorPluggedOnTTC580 filterPresenceSensor = new CarouselClampSensor(); */
        /* private final SensorPluggedOnTTC580 lockSensor = new CarouselClampSensor(); */
        FCSLOG.info(name + " updating socket sensors values for socketID=" + socketID);
        /* Xminus lock sensor update */        
        ((CarouselClampSensor) sensorsMap.get("carousel/socket"+socketID+"/clamp"+XMINUS+socketID+"/"+LOCKSENSOR + XMINUS + socketID)).updateValue(getLockXm(pdoValue));
        /* Xplus lock sensor update */
        ((CarouselClampSensor) sensorsMap.get("carousel/socket"+socketID+"/clamp"+XPLUS+socketID+"/"+LOCKSENSOR + XPLUS + socketID)).updateValue(getLockXp(pdoValue));
        /* Xminus filter presence sensor update */
        ((CarouselClampSensor) sensorsMap.get("carousel/socket"+socketID+"/clamp"+XMINUS+socketID+"/"+FILTERPRESENCE + XMINUS + socketID)).updateValue(getFilterPresenceXm(pdoValue));
        /* Xplus filter presence sensor update */
        ((CarouselClampSensor) sensorsMap.get("carousel/socket"+socketID+"/clamp"+XPLUS+socketID+"/"+FILTERPRESENCE + XPLUS + socketID)).updateValue(getFilterPresenceXp(pdoValue));
    }

    /**
     * Signals read on hyttc580 for PLC debug are read on pdo4
     *
     * @param pdoValue
     */
    private void updatePLCSensors(long pdoValue) {
        plcSensorsMap.values().stream().forEach((plcSensor) -> {
            plcSensor.updateValue(getSensorValue(pdoValue, plcSensor.getByteNumero(), plcSensor.getBitNumero()));
        });
    }


    /**
     * return socketId from PDO value socket identification are the first 3 bits
     * (reading from left to right).
     *
     * @param pdo
     * @return
     */
    @Override
    public short getSocketId(long pdo) {
        /* 61 = 64 - nbBits (nbBits =3) */
        return (short) ((pdo >>> 61));
    }

    /**
     * read lockSensorMinLocked : lock threshold which depends on ambient
     * temperature.
     *
     * @return lockSensorMinLocked
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "read lockSensorMinLocked : lock threshold for all clamps")
    @Override
    public long readLockSensorMinLocked() {
        return this.readSDO(0x6404, 5);
    }

    /**
     * Read offset1 for 2 clamps of socketID given as argument. Two 16-bits
     * words: from MSB: Offset for Balluff X+ Offset for Balluff X- for socketID
     * (presence sensor)
     *
     * @param socketID
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "read offset1 (balluff) for clamp presence sensors")
    @Override
    public long readOffset1SDO(byte socketID) {
        int subindex = Integer.parseInt(String.valueOf(socketID) + "0", 16);
        return this.readSDO(0x6404, subindex);
    }

    /**
     * Read offset2 for 2 clamps of socketID given as argument. Two 16-bits
     * words: from MSB: Offset for Baumer X+ Offset for Baumer X- for socketID
     * (clamp lock sensor)
     *
     * @param socketID
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "read offset2 (baumer) for clamp lock sensors")
    @Override
    public long readOffset2SDO(byte socketID) {
        int subindex = Integer.parseInt(String.valueOf(socketID) + "1", 16);
        return this.readSDO(0x6404, subindex);
    }

    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "for test and debug")
    @Override
    public long readInterlocksSDO() {
        int index = 0X6200;
        int subindex = 1;
        return readSDO(index, subindex);
    }

    /**
     * for tests
     *
     * @param sid
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "test and debug")
    public int getOffset1Xplus(byte sid) {
        return TTC580Utils.getOffset1Xplus(readOffset1SDO(sid));
    }

    /**
     * for tests
     *
     * @param sid
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "test and debug")
    public int getOffset1Xminus(byte sid) {
        return TTC580Utils.getOffset1Xminus(readOffset1SDO(sid));
    }

    /**
     * for tests
     *
     * @param sid
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "test and debug")
    public int getOffset2Xminus(byte sid) {
        return TTC580Utils.getOffset2Xminus(readOffset2SDO(sid));
    }

    /**
     * for tests
     *
     * @param sid
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "test and debug")
    public int getOffset2Xplus(byte sid) {
        return TTC580Utils.getOffset2Xplus(readOffset2SDO(sid));
    }


    /**
     * @param error
     * @return a string representation of the error code given as argument.
     */
    @Override
    public String displayError(int error) {
        StringBuilder sb = new StringBuilder();
        sb.append("ErrorSource=");
        sb.append(getErrorSource(error));
        sb.append("; ErrorType=");
        sb.append(getErrorType(error));
        sb.append("; ErrorCounter=");
        sb.append(getErrorCounter(error));
        sb.append("; ErrorCode=");
        sb.append(getErrorCode(error));
        return sb.toString();
    }

    /************************************************************************************
     * end of ERRORS
     * **********************************************************************************
     */

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description = "Return the string representation of the PDOs")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("\n/pdo1=");
        sb.append(pdo1);
        sb.append(", in hexa: 0x");
        sb.append(Long.toHexString(pdo1));
        sb.append(", in binary: ");
        sb.append(Long.toBinaryString(pdo1));
        sb.append("\n/pdo2=");
        sb.append(pdo2);
        sb.append(", in hexa: 0x");
        sb.append(Long.toHexString(pdo2));
        sb.append(", in binary: ");
        sb.append(Long.toBinaryString(pdo2));
        sb.append(printSensorsValues());
        return sb.toString();
    }

    /**
     * for tests
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "for tests and debug.")
    public String printSensorsValues() {
        StringBuilder sb = new StringBuilder("\nSensors values read from pdo1:\n");
        if (pdo1 == 0) {
            sb.append("No socket at Standby");
        } else {
            sb.append("Socket at Standby==>");
            sb.append(printSensorsForPdo(pdo1));
        }
        sb.append("\nSensors values read from pdo2:\n");
        sb.append("Socket not at Standby==>");
        sb.append(printSensorsForPdo(pdo2));
        return sb.toString();
    }

    private String printSensorsForPdo(long pdo) {
        StringBuilder sb = new StringBuilder("\n");
        sb.append("socketID=");
        sb.append(getSocketId(pdo));
        sb.append("/lockXminus=");
        sb.append(getLockXm(pdo));
        sb.append("/filterPresenceXminus=");
        sb.append(getFilterPresenceXm(pdo));
        sb.append("/lockXplus=");
        sb.append(getLockXp(pdo));
        sb.append("/filterPresenceXplus=");
        sb.append(getFilterPresenceXp(pdo));
        return sb.toString();
    }

    /**
     * Creates an object to be published on the STATUS bus by an hyttc580 device.
     *
     * @return
     */
    public StatusDataPublishedByHYTTC580 createStatusDataPublishedByHYTTC580() {
        StatusDataPublishedByHYTTC580 status = new StatusDataPublishedByHYTTC580(isBooted(), isInitialized());
        status.setPdo1(pdo1);
        status.setPdo2(pdo2);
        status.setPdo2(pdo4);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByHYTTC580()));
    }

}
