package org.lsst.ccs.subsystems.fcs.drivers;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.canopenjni.SDOException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.canopenjni.ConcurrentCallException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.BAD_SERIAL_NB;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_READING_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_MISSING;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.UNSYNC_BOOT;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_PLUTOGATEWAY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_PLUTOGATEWAY_NAME;

/**
 * The main job of the class CanOpenProxy is to check that the CANopen devices
 * booted on the CAN bus are the CANopen devices which are listed in the
 * configuration system.
 *
 * @author virieux
 */
public class CanOpenProxy implements BridgeToHardware, HasLifecycle, AlertRaiser,
        CanOpenEventListener {
    /* timeout for PDO*/
    private static final long PDO_TIMEOUT = 100;

    @LookupField(strategy = TOP)
    public Subsystem subs;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    protected String name;

    @LookupField(strategy = CHILDREN)
    protected CanOpenInterface canInterface;

    protected int master = 1;

    @ConfigurationParameter(description = "CANbus rate.", units = "unitless", category = "canbus")
    // value can be baud = "125k" for 125kbps or baud = "1m" for 1mbps;
    volatile String baud = "1m";

    @ConfigurationParameter(description = "CANbus name. Can be 0 for changer or 1 for loader.", units = "unitless", category = "canbus")
    volatile String busName = "0";

    @ConfigurationParameter(description = "CANbus master nodeID.", units = "unitless", category = "canbus")
    volatile int masterNodeID = 0x8;

    /**
     * A timeout for the booting process : during initialization, we wait for the
     * boot messages coming from CANbus field. When this amount of time is over, we
     * don't wait any more. This has to be tuned during the test bench. UNIT =
     * milliseconds
     *
     */
    @ConfigurationParameter(range = "1000..100000", description = "A timeout for the hardware booting process", units = "millisecond", category = "canbus")
    public volatile long hardwareBootTimeout = 5000;

    /**
     * Map of PieceOfHardware objects this CanOpenProxy manages. Key is device's
     * node ID.
     */
    protected final Map<Integer, PieceOfHardware> hardwareMapByNodeID = new HashMap<>();

    @LookupField(strategy = CHILDREN)
    protected final Map<String, PieceOfHardware> hardwareMapByName = new HashMap<>();

    @LookupField(strategy = CHILDREN)
    protected final Map<String, EPOSController> controllerMapByName = new HashMap<>();

    @LookupField(strategy = CHILDREN)
    protected final Map<String, PlutoGateway> plutoGatewayMapByName = new HashMap<>();

    @LookupField(strategy = TREE)
    private AlertService alertService;

    protected boolean hardwareBootProcessEnded = false;
    protected int bootedDeviceNB = 0;

    private PDOData pdoData = new PDOData();

    /**
     * a list of defined received PDO for this canopen proxy.
     */
    private ArrayList pdoList = new ArrayList(20);

    protected boolean canbusConnected = false;

    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    // Used because we have to wait until all the pieces of hardware are booted.

    private final Lock lock = new ReentrantLock();

    private final Condition bootingCompleted = lock.newCondition();

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public boolean isCanbusConnected() {
        return canbusConnected;
    }

    public CanOpenInterface getCanInterface() {
        return canInterface;
    }

    /**
     * @return PDOData
     */
    @Override
    public PDOData getPDOData() {
        return pdoData;
    }

    /**
     * ***********************************************************************************************
     * ******************** END OF SETTERS AND GETTERS
     * **********************************************
     * ***********************************************************************************************
     *
     */

    @Override
    public void init() {
        dataProviderDictionaryService.registerData(new KeyValueData(name, bootedDeviceNB));

        DataProviderInfo data = new DataProviderInfo("", DataProviderInfo.Type.TRENDING, name);
        data.addAttribute(DataProviderInfo.Attribute.UNITS, "no units");
        data.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "number of CAN open devices booted on this CAN bus");
        data.addAttribute(DataProviderInfo.Attribute.TYPE, "int");

        initialize();
        hardwareMapByName.values().stream().forEach((poh) -> {
            hardwareMapByNodeID.put(poh.getNodeID(), poh);
        });
        FCSLOG.info(name + ": init MODULE CanOpenProxy.");
        FCSLOG.info(name + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + hardwareMapByName.size());
        FCSLOG.info(this.toString());

        // we have to read the errors table for the maxon motor and can open devices.
        CanOpenErrorsTable.loadCanOpenErrorTables();

       ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(UNSYNC_BOOT.getAlert(), alwaysClear);
        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(CAN_BUS_READING_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(SDO_ERROR.getAlert(name), alwaysClear);
        hardwareMapByNodeID.values().stream().forEach((poh) -> {
            alertService.registerAlert(BAD_SERIAL_NB.getAlert(poh.getName()), alwaysClear);
            alertService.registerAlert(HARDWARE_MISSING.getAlert(poh.getName()), alwaysClear);
            alertService.registerAlert(HARDWARE_ERROR.getAlert(poh.getName()), alwaysClear);
            alertService.registerAlert(SDO_ERROR.getAlert(poh.getName()), alwaysClear);
        });        
        
    }

    /**
     * This method initializes the fields of the tcpProxy. It's used when we start
     * the Module and when we disconnect hardware.
     */
    protected void initialize() {
        this.bootedDeviceNB = 0;
        hardwareBootProcessEnded = false;
        hardwareMapByName.values().stream().forEach((poh) -> {
            poh.setBooted(false);
        });
    }

    /**
     * Check if all pieces of hardware in the configuration are booted with the
     * corect serial number. Raise alerts if not. Waits until all hardware is booted
     * or timeout is reached.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Check if all pieces of hardware are booted with the correct serial number.")
    @Override
    public void bootProcess() {
        FCSLOG.info(name + ": BEGIN OF HARDWARE BOOTING PROCESS");
        initialize();

        subs.getAgentService(AgentStateService.class).updateAgentState(FilterState.CAN_DEVICES_BOOTING,
                FilterReadinessState.NOT_READY);
        lock.lock();
        try {
            readDevicesInfo();
            waitForEndOfBooting();

        } catch (Exception ex) {
            this.raiseAlarm(HARDWARE_ERROR, " ERROR during hardware booting process due to " + ex, name);

        } finally {
            FCSLOG.debug(name + ": finally in bootProcess");
            hardwareBootProcessEnded = true;
            bootingCompleted.signalAll();
            lock.unlock();
        }
        if (this.bootedDeviceNB == hardwareMapByName.size()) {
            FCSLOG.info(name + " ALL HARDWARE is booted");
            subs.getAgentService(AgentStateService.class).updateAgentState(
                    FcsEnumerations.FilterState.READY, FcsEnumerations.FilterReadinessState.READY);
        } else if (this.bootedDeviceNB == 0) {
            FCSLOG.error(name + " NONE HARDWARE is booted");
            this.raiseAlarm(HARDWARE_ERROR, " NO HARDWARE booted - Power failure ?", name);

        } else if (this.bootedDeviceNB < hardwareMapByName.size()) {
            FCSLOG.error(name + " SOME HARDWARE is MISSING");
            hardwareMapByName.values().stream().forEach((poh) -> {
                poh.raiseAlarmIfMissing();
            });

        } else {
            FCSLOG.error(name + " this.bootedDeviceNB = " + " hardwareMapByName.size() = " + hardwareMapByName.size());
        }
        FCSLOG.info(name + " BOOTED HARDWARE=" + listBootedNodes());
        FCSLOG.info(name + ": END OF HARDWARE BOOTING PROCESS");
    }

    /**
     * Used with JNI interface to initialize canInterface to be able to received
     * PDOs.
     */
    public void initializePDOs() {
        // first step in our spiral development
        hardwareMapByNodeID.values().stream().forEach((poh) -> {
            try {
                poh.initializePDOs();
            } catch (DriverException ex) {
                this.raiseAlarm(HARDWARE_ERROR, " ERROR during PDO initialization : could not addReceivedPDOs",
                        poh.getName(), ex);
            }
        });
    }

    /**
     * Update pdoData in sending a sync command. And update all the PieceOfHardware
     * which are my children.
     */
    @Override
    public void updatePDOData() {
        long beginTime = System.currentTimeMillis();
        try {
            this.pdoData = this.canInterface.sync();
            long duration = System.currentTimeMillis() - beginTime;
            FCSLOG.info(name + " this.canInterface.sync() duration = " + duration);
        } catch (ConcurrentCallException ex) {
            this.raiseWarningOnlyIfNew(CAN_BUS_READING_ERROR, " couldn't update PDOs because " + ex, name);
        } catch (DriverException ex) {
            raiseAlarm(HARDWARE_ERROR, " couldn't update PDOs", ex);
        }

        if (!pdoData.isComplete()) {
            FCSLOG.info(name + "CANBUS READING ERROR : some PDOs are missing.");
            raiseWarningOnlyEveryTenMinutes(CAN_BUS_READING_ERROR,
                    " some PDOs are missing. " + "Power failure or bad configuration of a device ?", name);
        }

        hardwareMapByNodeID
            .values()
            .stream()
            .forEach(poh -> poh.updateFromPDO(pdoData));

        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updatePDOData duration = " + duration);
    }

    @Override
    public void start() {
        subs.getAgentService(AgentStateService.class).updateAgentState(
                FcsEnumerations.FilterState.WAITING_FOR_HARWARE_BOOTING,
                FcsEnumerations.FilterReadinessState.NOT_READY);
        connectToCANbus();
    }

    public void setPDOTimeout() throws DriverException {
        canInterface.setPdoTimeout(PDO_TIMEOUT);
    }

    public void connectToCANbus() {
        FCSLOG.info(name + " connecting to CANbus. " + this.toString());

        try {
            canInterface.init(master, baud, busName, masterNodeID); // for JNI
            setPDOTimeout();
            canbusConnected = true;
            FCSLOG.info(name + " canbusConnected= " + canbusConnected + " baud=" + baud);

        } catch (DriverException ex) {
            String msg = name + " could not init canInterface. " + this.toString();
            throw new FcsHardwareException(msg, ex);
        }

        try {
            //disableOperation to put brakes on autochanger linear rails
            //this is a temporary fix waiting for an update of the linear rails controllers software
            //in the mean time there is still a risk to drop a filter
            //TODO fetch controllers nodeID
            if (hardwareMapByNodeID.containsKey(0x2a)) {
                doDisableOperation(0x2a);
            }

            if (hardwareMapByNodeID.containsKey(0x2c)) {
                doDisableOperation(0x2c);
            }
            FcsUtils.sleep(100, name);

        } catch (DriverTimeoutException ex) {
            String msg = name + " could not disableOperation for AC trucks controllers "
                    + "after 2 tries; canbus0 is no more reachable; try reboot pc104.";
            FCSLOG.warning(msg, ex);
            this.raiseWarning(CAN_BUS_TIMEOUT, msg, ex);

        } catch (DriverException ex) {
            String msg = name + " could not disableOperation for AC trucks controllers "
                    + "after 2 tries; check 24Vclean or other failure for AC trucks controllers ? ";
            FCSLOG.warning(msg, ex);
            this.raiseWarning(SDO_ERROR, msg, ex);
//            throw new FcsHardwareException(msg, ex);
        }

        try {
            canInterface.scan();
            //no nedeed to wait here : scan is an asynchronous command
            //and the next command to be executed : bootProcess starts after a delay of 500ms

        } catch (DriverException ex) {
            throw new FcsHardwareException(name + " could not scan CANbus.", ex);
        }
    }

    public void doDisableOperation(int nodeID) throws DriverException {
        //we try 2 times
        try {
            canInterface.wsdo(nodeID, 0x6040, 0, 2, 7);
            FCSLOG.info(name + " disableOperation DONE for nodeID = 0x" + Integer.toHexString(nodeID));
        } catch (DriverTimeoutException ex) {
            FCSLOG.warning(name + " 2nd try to disableOperation for nodeID = 0x" + Integer.toHexString(nodeID), ex);
            canInterface.wsdo(nodeID, 0x6040, 0, 2, 7);
            FCSLOG.info(name + "  at 2nd try disableOperation DONE for nodeID = 0x" + Integer.toHexString(nodeID));
        } catch (DriverException ex) {
            String msg = name + " could not disableOperation for nodeID = 0x" + Integer.toHexString(nodeID);
            FCSLOG.warning(msg, ex);
            this.raiseWarning(SDO_ERROR, msg, ex);
            //commented on February 14th 2022 to let fcs starts and see cabling issues on quadbox
//            throw new FcsHardwareException(name + " could not disableOperation for nodeID = 0x" + Integer.toHexString(nodeID), ex);
        }
    }

    /**
     * check that all pieces of hardware is booted.
     */
    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart");
        bootProcess();
        initializePDOs();
        // for the GUI
        this.publishData();
        FCSLOG.info(name + " END postStart");
    }

    /**
     * This method is called in the bootProcess method during the initialization
     * process. It check devices at fixed rate with a scheduler until all devices
     * are booted or booting timeout is exceeded.
     */
    public void readDevicesInfo() {

        final long startTime = System.currentTimeMillis();
        final Runnable checkDevices = () -> {
            try {
                long duration = System.currentTimeMillis() - startTime;
                FCSLOG.fine("duration=" + duration + " : " + bootedDeviceNB + " booted devices.");
                if (duration > hardwareBootTimeout || this.bootedDeviceNB == hardwareMapByName.size()) {
                    hardwareBootProcessEnded = true;
                    cancelReadDevicesInfo();
                }
                // Retrieve some information as serial number, vendor, for the CANopen devices
                // with command (info,node_id)
                retrieveHardwareInfo();

            } catch (Exception ex) {
                FCSLOG.error(name + " error in devices boot process", ex);
            }
        };
        // we start to send info command after 500 milliseconds because some hardware
        // take time to boot.
        // we execute retrieveHardwareInfo every 250 milliseconds
        checkDevicesHandle = scheduler.scheduleWithFixedDelay(checkDevices, 500, 250, TimeUnit.MILLISECONDS);
    }

    /**
     * This stops the reading of the sensors.
     *
     */
    private void cancelReadDevicesInfo() {
        lock.lock();
        try {
            FCSLOG.debug(name + " => stop waiting for devices info.");
            bootingCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        checkDevicesHandle.cancel(true);
        FCSLOG.debug(name + " => readDevicesInfo canceled");
    }

    /**
     * This method waits until the booting process is completed. This methods is
     * called by postStart and has already acquired the lock.
     */
    private void waitForEndOfBooting() {
        while (!hardwareBootProcessEnded) {
            try {
                FCSLOG.info(name + " waiting until all pieces of hardware are booted.");
                bootingCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex);
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF BOOTING PROCESS");
    }

    /**
     * executed in pre-order : before controllers and other hardware shutdown.
     */
    @Override
    public void shutdown() {
        /* stop monitoring before stopping canInterface */
        List<String> taskNames = subs.getAgentService(AgentPeriodicTaskService.class).getAgentPeriodicTaskNames();
        //TODO check why we have to stop ?????
        if (taskNames.contains("monitor-update")) {
            subs.getAgentService(AgentPeriodicTaskService.class).setPeriodicTaskPeriod("monitor-update",
                    Duration.ofSeconds(-1));
        }
        if (taskNames.contains("monitorCurrent")) {
            subs.getAgentService(AgentPeriodicTaskService.class).setPeriodicTaskPeriod("monitorCurrent",
                    Duration.ofSeconds(-1));
        }
    }

    /**
     * This has to be executed in post order : after controllers and other hardware
     * shutdown. Called by MainModule.postShutdown. This shutdowns the scheduler.
     * This is executed during CLOSING phase when the CanOpenProxy is stopped. cf
     * checkStopped
     */
    @Override
    public void doShutdown() {
        FCSLOG.info(name + " is shutting down.");
        scheduler.shutdown();
        try {
            FCSLOG.info(name + " canInterface quit.");
            canInterface.quit();
            FCSLOG.info(name + " canInterface stop.");
            canInterface.stop();
            subs.getAgentService(AgentStateService.class).updateAgentState(FcsEnumerations.FilterState.OFF_LINE,
                    FcsEnumerations.FilterReadinessState.NOT_READY);
        } catch (DriverException ex) {
            FCSLOG.error("cannot shut down properly", ex);
        }
        FCSLOG.info(name + " is shutdown.");
    }

    /**
     * For a piece of hardware that this tcpProxy manages, this methods returns the
     * name of the device when the CANopen node id is given as argument.
     *
     * @param aNodeID
     * @return
     */
    public String getNodeName(int aNodeID) {
        if (this.hardwareMapByNodeID.get(aNodeID) == null) {
            return "UnknownDevice" + aNodeID;
        } else {
            return this.hardwareMapByNodeID.get(aNodeID).getName();
        }
    }

    /**
     * List the can open bootedNodes which are in the bootedNodes table.
     *
     * @return the list of can open bootedNodes and the information stored in
     *         this.bootedNodes.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Print the list of CANopen nodes which are booted on the CAN bus.")
    public String listBootedNodes() {
        if (this.bootedDeviceNB == 0) {
            return name + ": no booted CANopen devices.";
        } else {
            StringBuilder sb = new StringBuilder();
            for (PieceOfHardware poh : this.hardwareMapByName.values()) {
                if (poh.isBooted()) {
                    sb.append("\n==>");
                    sb.append(poh);
                }
            }
            return sb.toString();
        }
    }

    /**
     * Check that a nodeID entered by an end user at the console is a valid one:
     * nodeID should be in the hardwareMapByNodeID.
     *
     * @param nodeID
     */
    public void checkNodeID(int nodeID) {
        if (!hardwareMapByNodeID.containsKey(nodeID)) {
            throw new IllegalArgumentException(nodeID + " is not in the hardware list for tcpProxy " + name);
        }
    }

    /**
     * Command to be used by the end user at the console. Write a SDO message and
     * send it to the can open stack. The parameters are given in the following
     * format:
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @param size
     * @param value
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Send a CanOpen writeSDO command to the CANBus. "
            + "size represents the number of bytes on which the value is encoded. See device documentation.")
    public void writeSDO(int nodeID, int index, int subindex, int size, int value) {
        checkNodeID(nodeID);
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        writeSDO(nodeID, index, subindex, size, value, 1);
    }

    /**
     * Write a SDO message and send it to the can open stack. The parameters are
     * given in the following format:
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @param size
     * @param value
     * @param tryouts
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void writeSDO(int nodeID, int index, int subindex, int size, int value, int tryouts) {
        try {
            doWriteSDO(nodeID, index, subindex, size, value, tryouts);

        } catch (DriverException ex) {
            String msg = name +
                " error for request writeSDO for nodeID=0x" + Integer.toHexString(nodeID) +
                ",index=0x" + Integer.toHexString(index) +
                ",subindex=0x" + Integer.toHexString(subindex) +
                ",size=" + size +
                ",value=" + value;

            throw new SDORequestException(msg, ex);
        }
    }

    private void doWriteSDO(int nodeID, int index, int subindex, int size, int value, int tryouts) throws DriverException {
        try {
            canInterface.wsdo(nodeID, index, subindex, size, value);

        } catch (SDOException ex) {
            String deviceName = getNodeName(nodeID);
            String errorName = CanOpenErrorsTable.getErrorRegisterNameByCode(ex.getErrCode());
            String abortName = CanOpenErrorsTable.getCommErrorNameByCode(ex.getAbortCode());
            String msg = name
                    + " COULD NOT writeSDO for nodeID=0x"
                    + Integer.toHexString(nodeID) + "(" + nodeID + ")"
                    + ",index=0x" + Integer.toHexString(index)
                    + ",subindex=0x" + Integer.toHexString(subindex)
                    + ",errorCode=0x" + Integer.toHexString(ex.getErrCode())
                    + "=" + errorName
                    + ",abortCode=0x" + Integer.toHexString(ex.getAbortCode()) + "=" + abortName + ".";

            // TODO Verify the meaningfulness of error 0x85 1/2
            if (tryouts > 1 && ex.getErrCode() == 0x85) {
                String newMsg = msg + " Trying the command again (" + String.valueOf(tryouts) + " tries left)";
                this.raiseWarning(SDO_ERROR, newMsg, deviceName, ex);
                writeSDO(nodeID, index, subindex, size, value, tryouts - 1);

            } else if (ex.getErrCode() != 0x85) {
                throw new SDORequestException(ex.getMessage(), ex);

            } else {
                /* tryouts <=1 no more try*/
                String newMsg = msg + " no more try left (" + String.valueOf(tryouts) + " tries left)";
                this.raiseAlarm(SDO_ERROR, newMsg, deviceName, ex);
                throw new SDORequestException(newMsg, ex);
            }
        }
    }

    /**
     * Send a rsdo command to can interface and throw Exceptions or raise Alarms in
     * case of something went wrong.
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Send a CanOpen readSDO command to the CANBus.")
    public long readSDO(int nodeID, int index, int subindex) {
        checkNodeID(nodeID);
        return readSDO(nodeID, index, subindex, 1);
    }

    /**
     * Send a rsdo command to can interface and throw Exceptions or raise Alarms in
     * case of something went wrong.
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @param tryouts
     * @return
     */
    public long readSDO(int nodeID, int index, int subindex, int tryouts) {
        try {
            return doReadSDO(nodeID, index, subindex, tryouts);
        } catch (DriverException ex) {
            String msg = name
                    + " error for request readSDO for nodeID=0x" + Integer.toHexString(nodeID)
                    + ",index=0x" + Integer.toHexString(index)
                    + ",subindex=0x" + Integer.toHexString(subindex);
            throw new FcsHardwareException(msg, ex);
        }
    }

    private long doReadSDO(int nodeID, int index, int subindex, int tryouts) throws DriverException {
        try {

            return canInterface.rsdo(nodeID, index, subindex);

            /* catch an exception comming from canopenjni */
        } catch (SDOException ex) {
            String deviceName = getNodeName(nodeID);
            String errorName = CanOpenErrorsTable.getErrorSDONameByCode(ex.getErrCode());
            String abortName = CanOpenErrorsTable.getCommErrorNameByCode(ex.getAbortCode());
            String msg = name
                    + " error for request readSDO for nodeID=0x" + Integer.toHexString(nodeID) + "(" + nodeID + ")"
                    + ",index=0x" + Integer.toHexString(index)
                    + ",subindex=0x" + Integer.toHexString(subindex)
                    + ",errorCode=0x" + Integer.toHexString(ex.getErrCode())
                    + "=" + errorName
                    + ",abortCode=0x" + Integer.toHexString(ex.getAbortCode()) + "=" + abortName + ".";

            // Error 0x85 sent by canfestival library: aborted but not because of an abort command.
            if (tryouts > 1 && ex.getErrCode() == 0x85) {
                String newMsg = msg + " Trying the command again (" + String.valueOf(tryouts) + " tries left)";
                this.raiseWarning(SDO_ERROR, newMsg, deviceName, ex);
                return readSDO(nodeID, index, subindex, tryouts - 1);

            } else if (ex.getErrCode() != 0x85) {
                throw new SDORequestException(ex.getMessage(), ex);

            } else {
                /* tryouts <=1 no more try*/
                String newMsg = msg + " no more try left (" + String.valueOf(tryouts) + " tries left)";
                this.raiseAlarm(SDO_ERROR, newMsg, deviceName, ex);
                throw new SDORequestException(newMsg, ex);
            }
        }
    }

    /**************************************************************************************************
     * Methods delegated to canInterface
     * ************************************************************************************************
     */

    /**
     * Send a command sync to CAN bus to read sensors by PDO.
     *
     * @return
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Send a command sync to CAN bus to read sensors by PDO.")
    public PDOData sync() throws DriverException {
        PDOData pdoD = this.canInterface.sync();
        FCSLOG.info("PDOData=" + pdoD.toString());
        return pdoD;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = " init CAN interface.")
    public void init(int master, String baud, String busName, int nodeID) throws DriverException {
        canInterface.init(master, baud, busName, nodeID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "add a received PDO to canInterface.")
    public void addReceivedPDO(int cobId) throws DriverException {
        FCSLOG.info(name + " add PDO cobId = 0x" + Integer.toHexString(cobId));
        canInterface.addReceivedPDO(cobId);
        pdoList.add(cobId);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Clear all received PDO from canInterface. \nTo add again PDOs use command addReceivedPDO"
            + " for one PDO or use commande initializePDOs to add al received PDOs.")
    public void clearReceivedPDOs() throws DriverException {
        canInterface.clearReceivedPDOs();
        pdoList.clear();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Scan CAN bus.")
    public int scan() throws DriverException {
        return canInterface.scan();
    }

    /**
     * for end users from ccs-console.
     *
     * @param nodeID
     * @return result of command info,node_id
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Send a command info to CAN bus for nodeID to retrieve device information.")
    public String info(int nodeID) throws DriverException {
        String result = canInterface.info(nodeID);
        FCSLOG.info("result = " + result);
        if (hardwareMapByNodeID.containsKey(nodeID)) {
            PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
            processInfoMessage(poh, result);
        } else {
            throw new IllegalArgumentException(nodeID + " (0x" + Integer.toHexString(nodeID)
                    + ") : no such nodeID in subsystem " + subs.getName());
        }
        return result;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Send a command quit to canInterface to device.")
    public void quit() throws DriverException {
        canInterface.quit();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Send a command setNMTStateOperational to canInterface to device.")
    public void setNMTStateOperational(int nodeID) throws DriverException {
        canInterface.setNMTStateOperational(nodeID);
    }

    /**************************************************************************************************
     * end of Methods delegated to canInterface
     * ************************************************************************************************
     */

    /**
     * Identification of the hardware : we want to retrieve the information stored
     * in the hardware of the booted bootedNodes and update the array of bootedNodes
     * with this information. So for each booted node, we send a command : -
     * info,node_ID to the CWrapper and then we update the bootedNodes list with the
     * information returned in the command result. The serial number is a main
     * information to retrieve.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Identification of the hardware : we want to retrieve the information stored "
            + "in the hardware of the CANopen devices")
    public void retrieveHardwareInfo() {
        FCSLOG.info(name + ":Identification of the hardware");
        hardwareMapByName.values().stream().filter((poh) -> (!poh.isBooted())).forEach((poh) -> {
            updateDeviceInfo(poh);
        });
        publishData();
    }

    /**
     * send a message info,nodeID to the CANbus and update the CANopen device
     * information with data received from the device.
     *
     * @param poh
     */
    protected void updateDeviceInfo(PieceOfHardware poh) {
        int nodeID = poh.getNodeID();
        String commandInfo = "info," + Integer.toHexString(nodeID);
        try {
            //accelerobf has no serial number (no object 0x1018)
            //so we read deviceType to check if device is booted
            if ((nodeID == 0x50 && 0x20194 == poh.readDeviceType() && !FcsUtils.isSimu())) {
                FCSLOG.info(poh.getName() + " is booted");
                poh.setBooted(true);
                poh.initializeAndCheckHardware();
                this.bootedDeviceNB++;
            } else {
                String result = canInterface.info(nodeID);
                FCSLOG.info("result=" + result);
                processInfoMessage(poh, result);
            }

        } catch (DriverException ex) {
            this.raiseWarning(HARDWARE_MISSING, " no response to command : " + commandInfo + " ", poh.getName(), ex);

        } catch (FcsHardwareException ex) {
            this.raiseWarning(HARDWARE_ERROR, " Error in boot process for device.", poh.getName(), ex);
        }
    }

    /**
     * This method returns true if : all the hardware items are booted and
     * identified and the hardware have the node ID expected within the
     * configuration and the hardware is initialized.
     *
     * @return a boolean
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all CANopen devices are booted and identified.")
    @Override
    public boolean allDevicesBooted() {
        return hardwareBootProcessEnded && bootedDeviceNB == hardwareMapByNodeID.size();
    }

    @Override
    public void onBootMessage(int nodeID) {
        if (hardwareMapByNodeID.containsKey(nodeID)) {
            PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
            if (poh.isBooted() && hardwareBootProcessEnded) {
                /* the boot process is ended: something weird has just happened */
                String cause = " received an unsynchronous boot message for node ID = " + Integer.toHexString(nodeID)
                        + " Has the device been reset or powered on ?";
                this.raiseWarning(UNSYNC_BOOT, cause, poh.getName());
            }
        } else {
            this.raiseAlarm(UNSYNC_BOOT,
                    Integer.toHexString(nodeID) + ":UNKNOWN device. This device is not in the harware list.");
        }
    }

    /**
     * Process an emergency message received from a CANopen device on the status bus
     * : - ALARM or WARNING Alert is raised, - a message is logged
     *
     * @param nodeID
     * @param errCode
     * @param errReg
     */
    @Override
    public void onEmergencyMessage(int nodeID, int errCode, int errReg) {

        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(errCode);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);

        String deviceName = this.getNodeName(nodeID);
        EmergencyMessage emcyMsg = new EmergencyMessage(nodeID, deviceName, errCode, deviceErrorName, errReg,
                errorRegisterName);
        // Pass emergency message to affected device
        PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
        if (poh != null) {
            // The use of the scheduler is required in order to leave the reader thread as
            // soon as possible.
            subs.getScheduler().schedule(() -> {
                poh.onEmergencyMessage(emcyMsg);
            }, 0, TimeUnit.SECONDS);
        }
        FCSLOG.info(name + " received EMERGENCY message=" + emcyMsg.toString() + " for nodeID=0x"
                + Integer.toHexString(nodeID));
    }

    /**
     * Process the response to a command info,node_ID.
     *
     * @param message
     */
    void processInfoMessage(PieceOfHardware poh, String message) {
        String[] words = message.split(",", -1);
        String type = words[2];
        String vendor = words[3];
        String productCode = words[4];
        String revision = words[5];
        String serialNB = words[6];
        FCSLOG.info(name + "/" + poh.getName() + " informations read on device:" + " type=" + type + " vendor=" + vendor
                + " product code=" + productCode + " revision=" + revision + " serialNB=" + serialNB);

        /* because seneca constructor has fixed serial number to 0 for all its devices */
        boolean isSenecaDevice = "249".equals(vendor);
        if (isSenecaDevice) {
            FCSLOG.info(name + "/found a seneca device in message: " + message);
        }

        if (poh.getSerialNB().equals(serialNB) || isSenecaDevice) {
            // protect against multiple counting
            // because it can also be called from the info command at any time
            if (!poh.isBooted()) {
                poh.setBooted(true);
                this.bootedDeviceNB++;
            }

        } else {
            // should we raise the alert from here? We should allow recovery from a botched
            // message
            // if those message can happen.
            this.raiseAlarm(BAD_SERIAL_NB, " serial number read on device=" + serialNB + " ==>serial number expected="
                    + poh.getSerialNB() + " ==>device configuration=" + poh, poh.getName());
        }
    }

    /**
     * Return a String with the list of hardware expected in this subsystem. For
     * debug purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return a printed list of hardware expected in this subsystem.")
    @Override
    public String printHardwareList() {
        return hardwareMapByName.toString();
    }

    /**
     * Return a printed list of hardware with the initialization state. For debug
     * purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return a printed list of hardware with the initialization state.")
    @Override
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name + " CANopen devices : {");
        for (PieceOfHardware poh : this.hardwareMapByNodeID.values()) {
            sb.append(poh.printState());
            sb.append(";\n");
        }
        sb.append('}');
        return sb.toString();
    }

    /**
     * For the GUI : Return an Array containing hardware names handled by this
     * component.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return an Array containing hardware names handled by this component.")
    @Override
    public List<String> listHardwareNames() {
        List<String> res = new ArrayList<>();
        for (PieceOfHardware pieceOfHardware : this.hardwareMapByName.values()) {
            res.add(pieceOfHardware.getName());
        }
        Collections.sort(res);
        return res;
    }

    @Override
    public List<String> listAcSensorsNames() {
        if (plutoGatewayMapByName.containsKey(AC_PLUTOGATEWAY_NAME)) {
            return plutoGatewayMapByName.get(AC_PLUTOGATEWAY_NAME).listMySensorsNames();
        } else {
            return Collections.emptyList();
        }
    }
    
    @Override
    public List<String> listLoSensorsNames() {
        if (plutoGatewayMapByName.containsKey(LOADER_PLUTOGATEWAY_NAME)) {
            return plutoGatewayMapByName.get(LOADER_PLUTOGATEWAY_NAME).listMySensorsNames();
        } else {
            return Collections.emptyList();
        }
    }

    /**
     *
     * @return a list of pdo recorded by this CanOpenProxy
     */
    @Command(type = Command.CommandType.QUERY, description = "return a list of pdo recorded by this CanOpenProxy")
    public String printPDOList() {
        StringBuilder sb = new StringBuilder("Number of Received PDO = ");
        sb.append(pdoList.size());
        pdoList.forEach(cobid -> {
            sb.append("; cobid = 0x");
            sb.append(Integer.toHexString((int) cobid));
        });
        return sb.toString();
    }


    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Publish booting information for all CANopen devices.")
    @Override
    public void publishData() {
        FCSLOG.info(name + " is publishing:" + bootedDeviceNB);
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, bootedDeviceNB));
        hardwareMapByName.values().stream().forEach((device) -> {
            device.publishData();
        });
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("name=");
        sb.append(this.name);
        sb.append(" hardwareBootTimeout=");
        sb.append(this.hardwareBootTimeout);
        sb.append(" master=");
        sb.append(master);
        sb.append(" baud=");
        sb.append(baud);
        sb.append(" busName=");
        sb.append(busName);
        sb.append(" masterNodeID=");
        sb.append(masterNodeID);
        return sb.toString();
    }

    // -- BridgeToHardware interface methods -----------------------------------

    @Override
    public boolean isReady() {
        try {
            return canInterface.isReady() && allDevicesBooted();
        } catch (DriverException ex) {
            FCSLOG.error(ex);
            return false;
        }
    }

    @Override
    public AlertService getAlertService() {
        return subs.getAgentService(AlertService.class);
    }

    @Override
    public void connectHardware() {
        // Only for loader
    }

    @Override
    public void disconnectHardware() {
        // Only for loader
    }

    //TODO check also latches controllers when the issues with the ProtectionSystem will be solved
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Perform a check fault on all booted EPOS controllers")
    public void checkControllers() {
        controllerMapByName.values().stream().filter((epos) -> (epos.isBooted()))
                // TODO remove this line when latches controllers are OK again
                .filter((epos) -> (!epos.getName().contains("latch")))
                .forEach((epos) -> {
                    epos.checkFault();
                });
    }
}
