
package org.lsst.ccs.subsystems.fcs.drivers;

import java.text.DecimalFormat;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.PT100Interface;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * Used to measure temperatures in carousel motor and brakes.
 *
 * @author virieux
 */
public class CanOpenPT100 extends CanOpenDevice implements PT100Interface {

    public static int processValueIndex = 0x9130;

    Double[] temperatures = new Double[5];

    @Override
    public void init() {
        for (int i = 1; i < 5; i++) {
            temperatures[i] = Double.valueOf(0);
        }
    }

    @Override
    public double getTemperature(int i) {
        return temperatures[i];
    }

    /**
     * To configure this device to read temperature on the 4 channels : set SDO
     * index 0x6110, subindex = channel, size = 2, value = 30 (in decimal)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Configure this device to read temperature.")
    public void configure() {
        for (int i = 1; i <= 5; i++) {
            writeSDO(0x6110, i, 2, 30);
            FcsUtils.sleep(2, name);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "read temperatures on the device and update temperatures fields")
    @Override
    public void updateTemperatures() {
        for (int i = 1; i < 5; i++) {
            temperatures[i] = (double) readTemperature(i) / 1000;
            FcsUtils.sleep(2, name);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "read temperature, argument : 1..4")
    public long readTemperature(int subindex) {
        return this.readSDO(processValueIndex, subindex);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "print the 4 temperatures read by this device")
    @Override
    public String toString() {
        DecimalFormat formatter = new DecimalFormat("#######.000");
        StringBuilder sb = new StringBuilder("\n");
        for (int i = 1; i < 5; i++) {
            sb.append("temperature")
                .append(i)
                .append(" = ")
                    .append(formatter.format((double) temperatures[i]))
                    .append("\n");
        }
        return sb.toString();
    }

    /**
     * lifecycle commands
     */
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        this.initializeAndCheckHardware();
        FCSLOG.fine(name + " END postStart.");
    }

}
