
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

import java.util.logging.Logger;

/**
 *
 * @author virieux
 */
public class SimuCarousel extends Carousel {
    private static final Logger FCSLOG = Logger.getLogger(SimuCarousel.class.getName());

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/hyttc580")
    private SimuTTC580 hyttc580;

    private int deltaPositionMax = 1000;

    @Override
    public void postStart() {
        super.postStart();
        FCSLOG.info("===> INITIALISATION OF CLAMPS STATE : socket at STANBY empty; others loaded with filter<===");
        hyttc580.simulateSocketREADYTOCLAMP((short) 1);
        hyttc580.simulateSocketClampedOnFilter(false, (short) 2);
        hyttc580.simulateSocketClampedOnFilter(false, (short) 3);
        hyttc580.simulateSocketClampedOnFilter(false, (short) 4);
        hyttc580.simulateSocketClampedOnFilter(false, (short) 5);
        publishData();
        ((SimuCarouselController) carouselController).simulateSWITCH_ON_DISABLED();
    }


    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        int simuAbsTargetPos = this.absoluteTargetPosition % this.getFullTurn();
        if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            FCSLOG.info(name + " => Checking for carousel position to complete action - "
                    + " CurrentCarousel Position = " + this.position + " Target Position = "
                    + this.absoluteTargetPosition + " Delta Position expected = " + deltaPositionMax);
            return Math.abs(this.position - simuAbsTargetPos) < deltaPositionMax;
        }
        return false;
    }

    @Override
    public boolean isRotationAllowedByPLC() {
        return true;
    }

    @Override
    public boolean isUnclampAllowedByPLC() {
        return true;
    }

    /**
     *
     * Initialize clamps state for simulation : nothing to be done except
     * initialization of clampsStateInitialized because this method is called
     * each time carousel is rotated. For simulation the initialization of
     * clamps state is done in method postStart of SimuTTC580.
     */
    @Override
    public void initializeClampsState() {
        FCSLOG.info(name + " Initializing simulated clamps state....");
        clampsStateInitialized = true;
    }

    @Override
    public void initializeAndCheckClampsOffset() {
        //nothing to do in simulation
    }



    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
            description = "Update state from hyttc580 data.", timeout = 2000)
    @Override
    public void updateState() {
        long cmdBeginTime = System.currentTimeMillis();
        updateSocketAtStandbyState();
        //doesn't read again controller CPU because the following values have been refreshed by PDO.
        position = carouselController.getPosition();
        //if we use a scheduler for simulation, sometimes state is not updated for
        //socket not at standby
        updateSocketNotAtStandby();
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateState duration = " + duration);
    }

    @Override
    public void updateStateWithSensors() {
        super.updateStateWithSensors();
        /* useful otherwise only sockets 1 and 5 are displayed on the GUI */
        hyttc580.socketMapByID.values().stream().forEach((socket) -> {
            socket.publishData();
            socket.getClampXminus().publishData();
            socket.getClampXplus().publishData();
        });
    }

    public void simulateCarouselPosition(int pos) {
        ((SimuCarouselController) carouselController).simulatePosition(pos);
    }

    /**
     * return 0 if no socket is at standby when carousel is at position
     * carouselPosition return socket ID if a socket is at standby when carousel
     * is at position carouselPosition
     *
     * @param carouselPosition carousel position
     * @return socketID from socket which
     */
    public short getSocketAtStandbyID(int carouselPosition) {
        for (CarouselSocket socket : socketsMap.values()) {
            if (carouselPosition == socket.getStandbyPosition()) {
                return (short) socket.getId();
            }
        }
        return 0;
    }

    @Override
    public void releaseClamps() {
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't release clamps when no socket is halted at standby position.");
        } else if (socketAtStandby.getClampXminus().isFilterEngaged()
                || socketAtStandby.getClampXplus().isFilterEngaged()) {
            throw new RejectedCommandException(
                    name + ": Can't release clamps because a filter is engaged in clamps.");
        } else {
            getClampXminusController().goToSwitchOnDisabled();
            getClampXplusController().goToSwitchOnDisabled();
            hyttc580.simulateSocketREADYTOCLAMP((short) socketAtStandby.getId());
        }
    }

    @Override
    protected boolean areBrakesAllowingRotation() {
        return true;
    }
}
