
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;

import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedDuringMotionWithEncoderAndHoldingBrake;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;

/**
 * For carousel rotation controller.
 *
 * @author virieux
 */
public class CanOpenEPOSCarousel extends CanOpenEPOS implements EPOSControllerForCarousel {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    /**
     * define cobid to be able to read values from PDO data.
     */
    protected int cobid1;
    protected int cobid2;

    /**
     * PDO1 represents : current: 2 bytes positionActualValue: 4 bytes statusWord: 2
     * bytes
     */
    private long pdo1 = 0L;

    /**
     * PDO2 represents : FollowingError: 2 bytes VelocityActualValue 4 bytes
     * StatusWord: 2 bytes
     */
    private long pdo2 = 0L;

    public int positionSensorType;

    private boolean holdingBrakes = false;

    @Override
    public void setHoldingBrakes(boolean holdingBrakes) {
        this.holdingBrakes = holdingBrakes;
    }

    @Override
    public boolean isHoldingBrakes() {
        return holdingBrakes;
    }

    /**
     * set a pdo1 for simulation
     * @param pdo1
     */
    public void setPdo1(long pdo1) {
        this.pdo1 = pdo1;
    }

    /**
     * set a pdo2 for simulation
     * @param pdo2
     */
    public void setPdo2(long pdo2) {
        this.pdo2 = pdo2;
    }

    /**
     * for simulation
     * @return pdo1
     */
    public long getPdo1() {
        return pdo1;
    }

    /**
     * for simulation
     * @return pdo2
     */
    public long getPdo2() {
        return pdo2;
    }

    /**
     * for simulation only
     * @return cobid1
     */
    public int getCobid1() {
        return cobid1;
    }

    /**
     * for simulation only
     * @return cobid2
     */
    public int getCobid2() {
        return cobid2;
    }

    @Override
    public int getStatusWord() {
        return statusWord;
    }

    @Override
    public int getPositionSensorType() {
        return positionSensorType;
    }

    @Override
    public void setPositionSensorType(int positionSensorType) {
        this.positionSensorType = positionSensorType;
    }

    @Override
    public void init() {
        super.init();
        hasEncoder = true;

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        //Register Alert raised by interface EPOSControllerForCarousel
        alertService.registerAlert(HARDWARE_ERROR.getAlert(getName()), alwaysClear);

        cobid1 = nodeID + 0x180;
        cobid2 = nodeID + 0x280;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
    }

    @Override
    public void faultReset() {
        super.faultReset();
        /* go back to ssi sensor type to be able to read position again */
        if (positionSensorType == 8) {
            setPositionSensorTypeEncoderSSI();
        }
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        FCSLOG.finest(() -> name + " updatingFromPDO = " + pdo);
        boolean updated = false;
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
            updated = true;
            /* update from PDO1 */
            FCSLOG.finer(() -> name + " updatingFromPDO1 = " + pdo1 + " binaire:" + Long.toBinaryString(pdo1));
            statusWord = extractStatusWord(pdo1);
            averageCurrent = (short) extractCurrentAverage(pdo1);
            position = (int) extractPosition(pdo1);
            updateEposState(statusWord);
            FCSLOG.finer(() -> name + " position = " + position + " pdo1 = " + " binaire:" + Long.toBinaryString(pdo1));
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
            updated = true;
            /* update from PDO2 */
            FCSLOG.finer(() -> name + " updatingFromPDO2 = " + pdo2 + " binaire:" + Long.toBinaryString(pdo2));
            followingError = (short) extractFollowingError(pdo2);
            velocity = (int) extractVelocity(pdo2);
        }
        if (updated) {
            this.publishData();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update field statusWorld in reading a SDO (readStatusWord)")
    @Override
    public void updateStatusWord() {
        //nothing to do here because statusWord is updated by PDO
    }

    /**
     * @param pdo
     * @return statusWord from pdo
     */
    public static int extractStatusWord(long pdo) {
        return (int) pdo & 0xFFFF;
    }

    /**
     * @param pdo
     * @return current from pdo
     */
    public static int extractCurrentAverage(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return current from pdo
     */
    public static int extractPosition(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * @param pdo
     * @return following from pdo
     */
    public static int extractFollowingError(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return velocity from pdo
     */
    public static int extractVelocity(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * For tests and debug
     *
     * @return values updated by PDOs
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "print values updated by PDOs to debug and test")
    public String printValuesUpdatedByPDOs() {
        StringBuilder sb = new StringBuilder("=> pdo1 value = ");
        sb.append(pdo1);
        sb.append(" / pdo1 = 0x");
        sb.append(Long.toHexString(pdo1));
        sb.append(" / status word = 0x");
        sb.append(Long.toHexString(statusWord));
        sb.append("\n status word = 0b");
        sb.append(Long.toBinaryString(statusWord));
        sb.append("\n current average = ");
        sb.append(current);
        sb.append(" / current average = 0x");
        sb.append(Long.toHexString(current));
        sb.append("\n position = ");
        sb.append(position);
        sb.append(" / position = 0x");
        sb.append(Long.toHexString(position));
        sb.append("\n => pdo2 value = ");
        sb.append(pdo2);
        sb.append(" / pdo2 = 0x");
        sb.append(Long.toHexString(pdo2));
        sb.append("\n followingError = ");
        sb.append(followingError);
        sb.append(" / followingError = 0x");
        sb.append(Long.toHexString(followingError));
        sb.append(followingError);
        sb.append("\n Velocity = ");
        sb.append(velocity);
        sb.append(" / velocity = 0x");
        sb.append(Long.toHexString(velocity));
        return sb.toString();
    }

    public StatusDataPublishedByCarouselController createStatusDataPublishedByCarouselController() {
        StatusDataPublishedByCarouselController status = new StatusDataPublishedByCarouselController(isBooted(),
                isInitialized(), isInError(), getErrorRegister(), getErrorHistoryNB(), getLastErrorCode(),
                getLastErrorName());
        status.setMode(mode);
        status.setState(eposState);
        status.setCurrent(current);
        status.setPosition(position);
        status.setVelocity(velocity);
        status.setProfileAcceleration(profileAcceleration);
        status.setProfileDeceleration(profileDeceleration);
        status.setProfileVelocity(profileVelocity);
        status.setAverageCurrent(averageCurrent);
        status.setFollowingError(followingError);
        status.setPositionSensorType(positionSensorType);
        return status;
    }

    public StatusDataPublishedDuringMotionWithEncoderAndHoldingBrake createStatusDataPublishedDuringMotionWithEncoderAndHoldingBrake() {
        StatusDataPublishedDuringMotionWithEncoderAndHoldingBrake status = new StatusDataPublishedDuringMotionWithEncoderAndHoldingBrake();
        status.setCurrent(getCurrent());
        status.setAverageCurrent(getAverageCurrent());
        status.setVelocity(getVelocity());
        status.setFollowingError(getFollowingError());
        status.setPosition(getPosition());
        status.setHoldingBrakes(holdingBrakes);
        return status;
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByCarouselController()));
    }

    @Override
    public void publishDataDuringMotion() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedDuringMotionWithEncoderAndHoldingBrake()));
    }
}
