package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.DESCENDANTS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CLAMP_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.LO_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

import java.util.logging.Logger;

/**
 * This is the model for the loader in the Filter Exchange System. The loader is
 * used to load a filter into the camera or to unload a filter from the camera.
 *
 * CAMERA PROTECTION SYSTEM signal coming from autochanger :
 *
 * AP2: Autochanger at HANDOFF position
 *
 * AF0: Autochanger latches opened
 *
 * AF3: Filter locked in Autochanger
 *
 * @author virieux
 */
public class Loader implements FilterHolder, AlertRaiser, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(Loader.class.getName());
    private static final int TIMEOUT_FOR_MOVING_CARRIER = LoaderCarrier.TIMEOUT_FOR_MOVING_CARRIER;

    @LookupField(strategy = TOP)
    private Subsystem s;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    private String name;

    @LookupField(strategy = CHILDREN)
    private LoaderCarrier carrier;

    @LookupField(strategy = CHILDREN, pathFilter = LOADER_CLAMP_NAME)
    private LoaderClamp clamp;

    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = SIBLINGS, pathFilter = "autochanger")
    private FilterHolder autochanger;

    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderFilterPresenceSensors")
    private RedondantSensors loaderFilterPresenceSensors;

    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderOnCameraSensors")
    private RedondantSensors loaderOnCameraSensors;


    /**
     * CPS signal : autochanger_AF1 Filter detected in Autochanger
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/acAF1")
    private ComplementarySensors acAF1;

    /**
     * status signal coming from plutoGateway
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderCarrierRelayStatus")
    private DigitalSensor loaderCarrierRelayStatus;

    /**
     * status signal coming from plutoGateway
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderHooksRelayStatus")
    private DigitalSensor loaderHooksRelayStatus;

    /**
     * status signal coming from plutoGateway
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderFilterGoodPositionStatus")
    private DigitalSensor loaderFilterGoodPositionStatus;

    /**
     * filter distance coming from plutoGateway
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderFilterDistanceSensor")
    private ForceSensor loaderFilterDistanceSensor;

    @LookupField(strategy = TREE, pathFilter=".*\\/loaderPlutoGateway")
    private PlutoGatewayInterface loaderPlutoGateway;

    @LookupField(strategy = TREE, pathFilter=".*\\/hooksController")
    private EPOSController hooksController;

    @LookupField(strategy = TREE, pathFilter=".*\\/carrierController")
    private EPOSController carrierController;

    @LookupField(strategy = SIBLINGS, pathFilter = LOADER_TCPPROXY_NAME)
    private BridgeToHardware bridge;

    // Must be changed to 50 for the EMPTY FRAME
    @ConfigurationParameter(description = "maximum position of the filter", category = "loader")
    private volatile int filterDistanceMin = 80;

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public Subsystem getSubsystem() {
        return s;
    }

    /**
     * Returns carrier.
     *
     * @return carrier
     */
    public LoaderCarrier getCarrier() {
        return carrier;
    }

    /**
     * Returns clamp.
     *
     * @return
     */
    public LoaderClamp getClamp() {
        return clamp;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader CAN bus (can1) is connected")
    public boolean isCANbusConnected() {
        return bridge.isCanbusConnected();
    }

    public boolean isCarrierMotionAllowedByPLC() {
        return loaderCarrierRelayStatus.isOn();
    }

    public boolean isHooksMotionAllowedByPLC() {
        return loaderHooksRelayStatus.isOn();
    }

    @Override
    public void build() {
    }

    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(HARDWARE_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(LO_SENSOR_ERROR.getAlert(), alwaysClear);
    }


    /**
     *
     * @return true if hardware (controllers and plutoGateway) is correctly
     *         initialized and checkHardwareStateAndDoHomingIfPossible of the
     *         controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if hardware (controllers and plutoGateway) is correctly initialized"
            + "and homing of the controllers is done.")
    public boolean isInitialized() {
        boolean devicesInitialized = loaderPlutoGateway.isInitialized() && hooksController.isInitialized()
                && carrierController.isInitialized();
        return devicesInitialized && clamp.isHomingDone() && carrier.isInitialized();
    }

    /**
     * Returns the boolean field empty. If the empty boolean is being updated and
     * waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediately the
     * field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return !loaderFilterPresenceSensors.isOn();
    }

    /**
     * Returns the sensor value loaderOnCamera.
     *
     * @return connectedToCamera
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        return loaderOnCameraSensors.isOn();
    }

    /**
     * Return true if the autochanger is holding the filter. This command doesn't
     * read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        return autochanger.isHoldingFilter();
    }

    /**
     * Return true if a filter is present and it is held by the loader clamp.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if a filter is present and it is held by the loader clamp.")
    @Override
    public boolean isHoldingFilter() {
        return isClosedOnFilter();
    }

    /**
     * return true if a filter is present and hooks are closed.
     *
     * @return
     */
    public boolean isClampedOnFilter() {
        if (this.isCANbusConnected()) {
            this.updateStateWithSensors();
            this.clamp.updatePosition();
            return this.clamp.isClamped() && !this.isEmpty();
        } else {
            return false;
        }
    }

    /**
     * return true if a filter is present and hooks are closed.
     *
     * @return
     */
    public boolean isClosedOnFilter() {
        if (this.isCANbusConnected()) {
            this.updateStateWithSensors();
            return this.clamp.isClosed() && !this.isEmpty();
        } else {
            return false;
        }
    }

    public boolean isOpened() {
        return clamp.isOpened();
    }

    /**
     * Return true if loader clamp is opened.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader clamp is opened.")
    @Override
    public boolean isNotHoldingFilter() {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return this.clamp.isOpened();
    }

    /**
     * This command has to be executed after the initialization phase and after the
     * checkHardware command. It can't be automatically executed during
     * initialization phase because it's not compliant with the CCS requirement
     * which says that the initialization of the subsystem can't make move hardware.
     */
    @Deprecated //this command had not been tested on loader hardware
    public void locateHardware() {
        /*
         * check that plutoGateway and controllers are booted and initialized, and
         * controllers not in fault.
         */
        /* read sensors and update state */
        /* check that position read on carrier controller is confirmed by sensors. */
        carrier.initializeHardware();
        boolean devicesInitialized = loaderPlutoGateway.isInitialized() && hooksController.isInitialized()
                && carrierController.isInitialized();
        if (!devicesInitialized) {
            throw new FailedCommandException(
                    name + " couldn't locate hardware because devices " + "are not initialized.");
        }
        this.updateStateAndCheckSensors();
        /* check that clamp is not empty and closed. */
        checkClosedOnVoid();
        if (isEmpty() || (carrier.isAtHandoff() && autochanger.isHoldingFilter())) {
            /*
             * no filter in the loader /* OR a filter is in the loader, carrier is at
             * HANDOFF and autochanger is holding filter
             */
            clamp.open();

        } else {
            /* a filter is detected by filterPresenceSensor */
            if (!clamp.isClamped()) {
                clamp.clamp();
            }
        }
        this.updateFCSStateToReady();
    }

    public void checkClosedOnVoid() {
        /*
         * If the carrier is empty and the clamp is CLOSED, we can't start the loader
         * subsystem.
         */
        if (this.isEmpty() && clamp.isClosed()) {
            String msg = name + ": carrier is empty and clamp is CLOSED - can't start.";
            this.raiseAlarm(HARDWARE_ERROR, msg, name);
            throw new FcsHardwareException(msg);
        }
    }

    public void checkHaltRequired() {
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }
    }

    /**
     * initialize loader hardware after initialization. to be executed if during
     * boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Initialize loader hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        FCSLOG.info(() -> name + " BEGIN initializeHardware");

        // check that all hardware is booted and identified.
        bridge.bootProcess();

        try {
            this.postStart();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " couldn't initialize loader", ex);
        }
        FCSLOG.info(() -> name + " END initializeHardware");
    }

    /**
     * Check that plutoGateway is booted and initialize plutoGateway. Read sensors
     * and update clamp's lockStatus.
     */
    @Override
    public void postStart() {
        FCSLOG.info(() -> name + " postStart");
        if (loaderPlutoGateway.isBooted()) {
            initializeGateway();
            carrier.postStart();
            clamp.postStart();
        } else if (bridge.isCanbusConnected()) {
            // raise an Alert if loader CANbus is connected and plutoGateway is not booted.
            loaderPlutoGateway.raiseAlarmIfMissing();
        }
    }

    public void initializeGateway() {
        try {
            this.loaderPlutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize loaderPlutoGateway", ex);
        }
    }

    /**
     * Check if loader is connected on camera.
     *
     * @throws RejectedCommandException if not connected on camera
     */
    public void checkConnectedOnCamera() {
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(name + " Loader not connected - can't execute commands.");
        }
    }

    /**
     * Checks if the loader carrier can move.
     *
     * @param targetPosition
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the carrier can move.")
    public void checkConditionsForCarrierMotion(int targetPosition) {
        FCSLOG.info(() -> name + " checking pre-conditions for carrier motion");

        updateStateAndCheckSensors();

        /* loader can't move if no filter and clamp is closed. */
        checkClosedOnVoid();

        carrier.updatePosition();

        if (this.isEmpty()) {
            FCSLOG.info(() -> name + " no filter. Carrier can move.");
        } else {

            /*
             * if a filter is at HANDOFF in the loader, carrier can't move if it's held both
             * by loader and autochanger
             */
            if (carrier.isAtHandoff() && isAutochangerHoldingFilter() && (clamp.isClamped() || clamp.isClosed())) {
                throw new RejectedCommandException(name + " carrier can't move because a filter is in the loader"
                        + " and it's held by loader AND autochanger.");
            }

            /*
             * if loader is between Storage and Engaged with a filter and another filter is
             * in the autochanger, loader carrier can't move.
             */
            if (carrier.getPosition() < carrier.getEngagedPosition() && acAF1.isOn()) {
                throw new RejectedCommandException(name + " carrier can't move because a filter is "
                        + "in the loader AND another one is in the autochanger.");
            }

            checkFilterDistanceClamped();

            /* motion between STORAGE and ENGAGED position with a filter */
            if (targetPosition <= carrier.getEngagedPosition()
                    && carrier.isLooslyBetweenStorageAndEngaged()) {
                clamp.checkClamped();
                /* motion between HANDOFF and ENGAGED with a filter */
            } else if (targetPosition >= carrier.getEngagedPosition()
                    && carrier.isLooslyBetweenEngagedAndHandoff()) {
                clamp.checkUnclamped();
                /* motion from HANDOFF to STORAGE without a filter */
            } else if (carrier.isAtHandoff() && clamp.isOpened()) {
                FCSLOG.info(() -> name + " carrier can move empty to targetPosition=" + targetPosition);

                /* motion between HANDOFF and STORAGE with a filter */
            } else {
                throw new RejectedCommandException("Carrier has to stop first at ENGAGED position");
            }
        }
    }

    /**
     * Check if the clamp can be opened.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the clamp can be opened.")
    public void checkConditionsForOpeningHooks() {
        FCSLOG.info(() -> name + " checking pre-conditions for opening hooks");

        updateStateAndCheckSensors();

        if (!this.isEmpty()) {
            /*
             * if a filter is in the loader and carrier is NOT at HANDOFF, clamp can't be
             * open.
             */
            if (!carrier.isAtHandoff()) {
                String msg = name + ": carrier is loaded with a filter but not "
                        + "at handoff position - can't open clamp.";
                FCSLOG.severe(msg);
                throw new RejectedCommandException(msg);
            }
            /* carrier is at Handoff position */
            if (!isAutochangerHoldingFilter()) {
                String msg = name + ": A filter is in the loader but not held by autochanger " + "- can't open clamp.";
                FCSLOG.severe(msg);
                throw new RejectedCommandException(msg);
            }
        }
    }

    public void checkConditionsForClampingHooks() {
        if (!carrier.isAtEngaged()) {
            String msg = name + ": carrier is loaded with a filter but not " + "at ENGAGED position - can't clamp.";
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
        if (isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(name + " Autochanger is holding filter. Open autochanger latches before "
                    + "clamping loader hooks.");
        }
    }

    public void checkConditionsForRecoveryClampingHooks() {
        if (!carrier.isLooslyBetweenStorageAndEngaged()) {
            String msg = name + ": commands recoveryClamp or recoveryUnclamp can be executed only if "
                    + "carrier is between ENGAGED and STORAGE.";
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
    }

    public void checkConditionsForUnclampingHooks() {
        if (!carrier.isAtEngaged()) {
            String msg = name + ": carrier is loaded with a filter but not " + "at ENGAGED position - can't unclamp.";
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
    }


    /**
     * Check if the clamp can be closed. Clamp can be close if sensors are not in
     * error and a filter is in the loader.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the clamp can be closed.")
    public void checkLoaderNotEmpty() {
        FCSLOG.info(() -> name + " checking pre-conditions for closing hooks");
        updateStateAndCheckSensors();

        this.clamp.updatePosition();

        if (this.isEmpty()) {
            String msg = name + ": no filter in loader - can't execute close nor clamp command." + clamp.getName();
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
    }

    /**
     * Check filter distance when loader clamp is OPENED and carrier is going to
     * HANDOFF.
     *
     * Distance must be > filterDistanceMin otherwise raise and ALARM and launch
     * a FcsHardwareException.
     *
     * cf JIRA LSSTCCSFCS-344
     */
    public void checkFilterDistanceOpened() {
        if (clamp.isOpened() && carrier.isCarrierApproachingHandoff()
                && loaderFilterDistanceSensor.getVoltage() < filterDistanceMin) {
            String msg = name + " : filter position is < " + filterDistanceMin
                    + " distance to the filter too low; carrier motion must be stopped.";
            FCSLOG.severe(msg);
            this.raiseAlarm(HARDWARE_ERROR, msg, name);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * When a filter is CLAMPED in loader, check that
     * loaderFilterGoodPositionStatus is ON.
     *
     * Otherwise raise an ALARM and launch a FcsHardwareException.
     *
     * cf JIRA LSSTCCSFCS-294
     */
    public void checkFilterDistanceClamped() {
        if (clamp.isClamped() && !loaderFilterGoodPositionStatus.isOn()) {
            String msg = name + " : filter distance status is OFF :"
                    + " distance to the filter too high; carrier motion not allowed.";
            FCSLOG.severe(msg);
            this.raiseAlarm(HARDWARE_ERROR, msg, name);
            throw new FcsHardwareException(msg);
        }
    }

    public void updateStateAndCheckSensors() {
        updateStateWithSensors();
        autochanger.updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();
        clamp.checkSensors(LO_SENSOR_ERROR, clamp.getName());
    }

    /**
     * This methods updates the carrier and clamp state in reading all the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {
        FCSLOG.info(() -> name + " bridge=" + bridge.toString());
        FCSLOG.info(() -> name + " bridge CANbus connected=" + bridge.isCanbusConnected());

        if (!this.isCANbusConnected()) {
            FCSLOG.info(() -> name + " about to connect loader CANbus.");
            this.connectLoaderCANbus();
        }

        // CANbus can be still not connected after connectLoaderCANbus()
        if (this.isCANbusConnected()) {
            loaderPlutoGateway.checkBooted();
            loaderPlutoGateway.checkInitialized();

            this.loaderPlutoGateway.updateValues();
            this.carrier.updatePosition();
            this.clamp.updatePosition();
            this.clamp.updateState();
            this.clamp.publishData();
            this.carrier.publishData();
        }
    }

    /**
     * Update FCS state and FCS readiness state and publish on the status bus. Check
     * that Loader hardware is ready to be operated and moved. This means that : -
     * all CAN open devices are booted, identified and initialized, -
     * checkHardwareStateAndDoHomingIfPossible has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Force FCS state and FCS readiness state to be ready and publishes on the status bus. TO BE USED ONLY IN EXTREME CASES.")
    public void updateFCSStateToReady() {
        if (clamp.isHomingDone() && carrier.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    public void disconnectLoaderCANbus() {
        bridge.disconnectHardware();
    }

    public void connectLoaderCANbus() {
        this.bridge.connectHardware();
        postStart();
    }

    public void goToStorage() {
        carrier.goToStorage();
    }

    public void goToHandoff() {
        carrier.goToHandOff();
    }

    public void goToEngaged() {
        carrier.goToEngaged();
    }

    public void goToAbsolutePosition(int newPosition) {
        carrier.goToAbsolutePosition(newPosition);
    }

    /**
     * move a filter to HANDOFF position.
     *
     * This command executes sequence of actions :
     *
     * - check initial condition : a filter is in the loader
     *
     * - move carrier to ENGAGED
     *
     * - unclamp filter (not open, just unclamp to release constraint on filter)
     *
     * - move slowly filter to HANDOFF.
     *
     * This command ca be restarted at any position of the carrier. (see JIRA
     * LSSTCCSFCS-243)
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move a filter from STORAGE to HANDOFF. "
            + "This command executes sequence of actions :\n" + "- check initial conditions : a filter is in the loader \n"
            + "- move carrier to ENGAGED with high speed\n" + "- unclamp filter \n"
            + "- move slowly filter to HANDOFF. This command can be restarted after a stop in any position of the carrier.", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveFilterToHandoff() {
        if (this.isEmpty()) {
            throw new RejectedCommandException(name + " is empty. " + "Command moveFilterToHandoff is not relevant.");
        }
        if (carrier.isStrictlyBetweenStorageAndEngaged()) {
            if (!clamp.isClamped()) {
                throw new FcsHardwareException(name + " filter is not CLAMPED and carrier "
                        + "is between Storage and Engaged. Command moveFilterFromStorageToHandoff "
                        + "can't go on. See recovery procedure.");
            }
            /*
             * checkConditionsForCarrierMotion is called by goToEngaged and again by
             * goToHandOff
             */
            carrier.raiseProfileVelocity();
            carrier.raiseProfileAcceleration();
            carrier.raiseProfileDeceleration();
            carrier.goToEngaged();
        }
        if (carrier.isAtEngaged()) {
            /**
             * a little nap to let time to the protection system to update its
             * status, otherwise if we try to unclamp hooks before lpm
             * authorizes we have a UNDER_VOLTAGE error.
             */
            FcsUtils.sleep(50, name);

            /* in ENGAGED position the filter can be UNCLAMPED */
            clamp.unclamp();
        }
        if (carrier.isLooslyBetweenEngagedAndHandoff()) {
            if (!clamp.isClosed()) {
                throw new FcsHardwareException(name + " loader clamp should be CLOSED. Can't go one.");
            }
            /* move smoothly from ENGAGED to HANDOFF with filter */
            carrier.slowProfileVelocity();
            carrier.slowProfileAcceleration();
            carrier.slowProfileDeceleration();
            carrier.goToHandOff();
        }
    }

    /**
     * Move a filter from HANDOFF to STORAGE.
     *
     * - checks initial conditions : a filter is in the loader, autochanger
     * holding the filter (latches CLOSED), clamp is CLOSED
     *
     * - move filter slowly to ENGAGED
     *
     * - clamp
     *
     * - move filter to STORAGE with high speed.
     *
     * Can be restarted at any position of the carrier. (see JIRA
     * LSSTCCSFCS-243)
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move a "
            + "filter from HANDOFF to STORAGE : move filter slowly to ENGAGED"
            + "then clamp and move filter to STORAGE with high speed. "
            + "This command can be restarted in any position of carrier between "
            + "HANDOFF and STORAGE. This command can be restarted at any position of the carrier.", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveFilterToStorage() {
        if (this.isEmpty()) {
            throw new RejectedCommandException(name + " is empty. " + "Command moveFilterToStorage is not relevant.");
        }
        if (carrier.isStrictlyBetweenEngagedAndHandoff()) {
            FCSLOG.info(() -> name + " carrier is between HANDOFF and ENGAGED; about to move slowly to ENGAGED.");
            if (!clamp.isClosed()) {
                throw new RejectedCommandException(
                        name + " clamp is not CLOSED. "
                        + "Command moveFilterToStorage is not relevant."
                        + "Check that autochanger is holding the filter "
                        + "and do openAndHoming on the loader clamps before retrying.");
            }
            carrier.slowProfileVelocity();
            carrier.slowProfileAcceleration();
            carrier.slowProfileDeceleration();
            carrier.goToEngaged();
        }
        if (carrier.isAtEngaged()) {
            FCSLOG.info(() -> name + " carrier is at ENGAGED.");
            /**
             * a little nap to let time to the protection system to update its
             * status, otherwise if we try to clamp hooks before lpm authorizes
             * we have a UNDER_VOLTAGE error.
             */
            FcsUtils.sleep(100, name);

            /* At ENGAGED the filter can be CLAMPED */
            if (!clamp.isClamped()) {
                clamp.clamp();
            }
        }
        if (carrier.isLooslyBetweenStorageAndEngaged()) {
            if (!clamp.isClamped()) {
                throw new FcsHardwareException(name + " carrier is between STORAGE and ENGAGED "
                        + "and not CLAMPED. moveFilterFromHandoffToStorage can't go on. See recovery procedure.");
            }
            /**
             * a little nap to let the ProtectionSystem update its status
             * otherwise it prevents the execution of raiseProfileVelocity
             */
            FcsUtils.sleep(50, name);

            /* move filter to storage with high acceleration and high deceleration */
            carrier.raiseProfileVelocity();
            carrier.raiseProfileAcceleration();
            carrier.raiseProfileDeceleration();
            carrier.goToStorage();
        }
    }

    /**
     * This command can be executed when loader is at Handoff with a filter. The
     * autochanger must hold the filter.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "open loader hooks and move empty from HANDOFF to STORAGE Position at high speed.", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void openClampAndMoveEmptyToStorage() {
        if (!this.isEmpty() && !carrier.isAtHandoff()) {
            throw new RejectedCommandException(name + " command openClampAndMoveEmptyToStorage "
                    + "can't be executed when loaded with a filter and not at HANDOFF");
        }
        /*carrier is empty or carrier is loaded with a filter at HANDOFF*/
        if (carrier.isAtHandoff()) {
            clamp.open();
        }
        if (!clamp.isOpened()) {
            throw new FcsHardwareException(
                    name + " clamp is not opened. "
                    + "Command openClampAndMoveEmptyToStorage can't go on. See recovery procedure.");
        }
        /* clamp is open => loader can go at high speed to STORAGE */
        carrier.raiseProfileVelocity();
        carrier.raiseProfileAcceleration();
        carrier.raiseProfileDeceleration();
        carrier.goToStorage();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move empty from Storage to Handoff at high speed and then close clamp.", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveEmptyToHandoffAndClose() {
        if (!this.isEmpty() && !this.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + " is not empty. " + "Command moveEmptyToHandoffAndClose is not relevant. Use goToHandoff");
        }
        /* approach HANDOFF position with care */
        carrier.raiseProfileVelocity();
        carrier.raiseProfileAcceleration();
        carrier.slowProfileDeceleration();
        carrier.goToHandOff();
        /* must do the homing of the clamp controller if not already done. */
        if (!clamp.isHomingDone()) {
            clamp.open();
        }
        clamp.close();
    }

    /* FilterHolder methods */
    @Override
    public boolean isAtHandoff() {
        return carrier.isAtHandoff();
    }

    @Override
    public boolean isAtStandby() {
        return false;
    }

    @Override
    public boolean isAtOnline() {
        return false;
    }
    /* end of FilterHolder methods */

    public boolean isAtStorage() {
        return carrier.isAtStorage();
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Print list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(carrier.printHardwareState());
        sb.append("\n");
        sb.append(clamp.printHardwareState());
        return sb.toString();
    }

    @Override
    public int getFilterID() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

}
