
package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.subsystems.fcs.FCSCst;

import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;

/**
 * This class represents a 14bits sensor : the value returned by the sensor is
 * coded with 14 bits, so its value is between 0 and 32767.
 *
 * @author virieux
 */
public abstract class Sensor14bits implements Sensor, SensorPluggedOnTTC580 {
    private static final Logger FCSLOG = Logger.getLogger(Sensor14bits.class.getName());
    @LookupName
    protected String name;

    protected int value = FCSCst.SENSOR14BITS_MIN;

    // updatingValue is true while we are reading the value sent by the Sensor
    // this could take some time
    private volatile boolean updatingValue = false;
    private final Lock lock = new ReentrantLock();
    private final Condition valueUpdated = lock.newCondition();

    /**
     * This method returns value if the sensor is not updating otherwise it waits
     * until the update is completed and returns the new read value from the
     * hardware.
     *
     * @return value
     */
    @Override
    public int getValue() {
        lock.lock();
        try {
            while (updatingValue) {
                try {
                    // TODO put a timeout
                    valueUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.log(Level.SEVERE, name, ex);
                }
            }
            return value;
        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the name of this component
     *
     * @return
     */
    @Override
    public String getName() {
        return name;
    }

    /**
     * This method updates the sensor value : first it reads a new value from the
     * hardware or from the simulated sensor, then it checks if the value is in the
     * range between minimal value and maximal value. If it's ok it updates the
     * sensor value with the new value, otherwise it throws an exception.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException
     */
    @Override
    public void updateValue() {

        lock.lock();

        try {
            updatingValue = true;
            int newValue = readNewValue();
            if ((newValue < FCSCst.SENSOR14BITS_MIN) || (newValue > FCSCst.SENSOR14BITS_MAX)) {
                throw new SensorValueOutOfRangeException("ERROR in Sensor14bits during updateValue for ", name,
                        FCSCst.SENSOR14BITS_MIN, FCSCst.SENSOR14BITS_MAX, newValue);
            }
            this.value = newValue;

        } finally {

            updatingValue = false;
            valueUpdated.signal();
            lock.unlock();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns the value of the sensor stored in FCS memory. "
            + "Doesn't read again the ADC device.")
    public String printValue() {
        return Integer.toString(value);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns a printed format of this sensor.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/returned value=");
        sb.append(Integer.toString(value));
        return sb.toString();
    }

}
