package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.ANCESTORS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_READING_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;

import java.time.Duration;
import java.util.concurrent.ScheduledFuture;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.StrainGauge;
import org.lsst.ccs.subsystems.fcs.drivers.MonitorDeviceInterface;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AsyncTasks;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;


/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
/**
 * Represents the set of the 3 onlineClamps of autochanger. This class gathers
 * the method to send command to the 3 onlineClamps all together.
 *
 * It is no more a MobileItem since we use AutochangerOnlineClamp commands.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps implements ControlledBySensors, SignalHandler, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerThreeOnlineClamps.class.getName());

    @LookupName
    protected String name;

    @LookupPath
    protected String path;

    @LookupField(strategy = ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    protected Autochanger autochanger;

    @LookupField(strategy = TREE, pathFilter = ".*\\/onlineStrainGauge")
    private StrainGauge onlineStrainGauge;

    @LookupField(strategy = TREE, pathFilter = ".*\\/tempSensorsDevice1")
    private MonitorDeviceInterface tempSensorsDevice1;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    protected final AutochangerOnlineClamp onlineClampXminus;
    protected final AutochangerOnlineClamp onlineClampXplus;
    protected final AutochangerOnlineClamp onlineClampYminus;
    protected final AutochangerOnlineClamp[] clampsList;

    private FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    /* Tools for current ramp */
    protected ScheduledFuture<?> currentRampHandle;

    @LookupField(strategy = TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @ConfigurationParameter(description = "timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "autochanger")
    protected volatile int timeoutForLockingClamps = 15000;

    @ConfigurationParameter(description = "timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "autochanger")
    protected volatile int timeoutForUnlockingClamps = 20000;

    @ConfigurationParameter(description = "maximum time in milliseconds to lock the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToLockAllClamps = 3000;

    @ConfigurationParameter(description = "maximum time in milliseconds to unlock the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToUnlockAllClamps = 3000;

    @ConfigurationParameter(description = "maximum time in milliseconds to open the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToOpenClampsX = 2000;

    @ConfigurationParameter(description = "maximum time in milliseconds to close the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToCloseClampsX = 2000;

    @ConfigurationParameter(description = "minimal period for current ramps. should be > 50", units = "millisecond", range = "50..1000", category = "autochanger")
    protected volatile int minPeriod = 200;

    /**
     * **************************************************
     */
    /* configurable parameters for onlineStrainGauge */

    /* if strain < maxLockedStrain, clamps are LOCKED */
    @ConfigurationParameter(description = "if strain < maxLockedStrain, clamps are LOCKED", category = "autochanger")
    private volatile short maxLockedStrain = 1484;

    /* if strain < minLockedStrain, clamps may be in ERROR */
    @ConfigurationParameter(description = "if strain < minLockedStrain, clamps may be in ERROR", category = "autochanger")
    private volatile short minLockedStrain = 500;

    /* if maxLockedStrain <= strain < maxClosedStrain, clamps are CLOSED */
    /* if strain >= maxClosedStrain, clamps are OPENED */
    @ConfigurationParameter(description = "if strain >= maxClosedStrain, clamps are OPENED", category = "autochanger")
    private volatile short maxClosedStrain = 1572;
    /* end of configurable parameters for onlineStrainGauge */

    // TODO this parameter should be static ? question à poser à Aurélien
    @ConfigurationParameter(description = "UNDEFINED", category = "autochanger")
    private volatile double strainGain = 2.978;

    /*
     * strain read on strainGauge as a voltage. It measures the deformation of
     * onlineClampXplus. Units = mV
     */
    private short readStrain = 0;

    /* deformation computed from readStrain and temperature */
    private short normalizedStrain = 0;

    /**
     * Used to know if lockStatus has been initialized from gauge strain. At startup
     * this value is false. At first updateState this value is set to true.
     */
    private boolean lockStatusInitialized = false;

    /**
     * **************************************************
     */
    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClamp.
     *
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampYminus
     */
    public AutochangerThreeOnlineClamps(AutochangerOnlineClamp onlineClampXminus,
            AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampYminus;
        clampsList = new AutochangerOnlineClamp[] { onlineClampYminus, onlineClampXminus, onlineClampXplus };
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerThreeClamps.class, path);
    }

    @Override
    public void init() {

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(CAN_BUS_READING_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if lockStatus has been first initialized from strain.")
    public boolean isLockStatusInitialized() {
        return lockStatusInitialized;
    }

    /**
     *
     * @return true if homing has been done for the 3 controllers of clamps.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if homing has been done for the 3 ONLINE clamp controllers.")
    public boolean isHomingDone() {
        return onlineClampXminus.isHomingDone() && onlineClampXplus.isHomingDone() && onlineClampYminus.isHomingDone();
    }

    /**
     * set a minimal value for current ramps period
     *
     * @param minPeriod
     */
    @ConfigurationParameterChanger
    public void setMinPeriod(int minPeriod) {
        if (minPeriod < 50) {
            throw new IllegalArgumentException(minPeriod + " bad value for minPeriod. should be > 50");
        } else {
            this.minPeriod = minPeriod;
        }
    }

    /**
     *
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=LOCKED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are LOCKED.")
    public boolean isLocked() {
        return onlineClampXminus.isLocked() && onlineClampXplus.isLocked() && onlineClampYminus.isLocked();
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isClosed() {
        return onlineClampXminus.isClosed() && onlineClampXplus.isClosed() && onlineClampYminus.isClosed();
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are OPENED.")
    public boolean isOpened() {
        return onlineClampXminus.isOpened() && onlineClampXplus.isOpened() && onlineClampYminus.isOpened();
    }

    /**
     * @return true if the 3 clamps are in Travel between opened position and closed
     *         position.
     *
     *         //todo what is the logic here? Why not if just one is in travel ? And
     *         since Y is always handled //first, is this implementation not always
     *         false?
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel() && onlineClampXplus.isInTravel() && onlineClampYminus.isInTravel();
    }

    /**
     * Returns true if LockStatus=ERROR
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if one of the clamp is in error.")
    @Override
    public boolean isInError() {
        return onlineClampXminus.isInError() || onlineClampXplus.isInError() || onlineClampYminus.isInError();
    }

    /**
     * Return true if the 3 onlineClamps hardware is lockStatusInitialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
            + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized()
                && onlineClampYminus.isInitialized();
    }

    /**
     * do homing of the 3 clamps
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller", timeout = 6000)
    public void homing() {
        try (AutoTimed at = new AutoTimed("clamps homing")) {
            AsyncTasks asyncRun = FcsUtils.asyncRun();
            for (AutochangerOnlineClamp clamp : clampsList) {
                asyncRun.asyncRun(() -> clamp.openInCurrentModeAndHoming());
            }
            asyncRun.await();
        }
    }

    /**
     * test periodicTask.
     */
    public void testPeriodTask() {
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask(name + "-updateStateAndCheckSensors", this::updateStateAndCheckSensors)
                        .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofMillis(5000)));
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    public void closeClamps() {
        try (AutoTimed at = new AutoTimed("closeClamps")) {
            if (!isHomingDone()) {
                // TODO should be do the homing???
                throw new RejectedCommandException(name + " homing not done yet. Can't close.");
            }
            updateStateAndCheckSensors();

            if (isOpened()) {
                autochanger.checkConditionsForActioningOnlineClamps();
                onlineClampYminus.close();
                FcsUtils.parallelRun(() -> onlineClampXminus.close(), () -> onlineClampXplus.close());

            } else if (isClosed()) {
                FCSLOG.info(name + " clamps already CLOSED nothing to do");

            } else {
                throw new RejectedCommandException(name + " has to be unlocked before.");
            }
        }
    }

    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For final
     * pruducts AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    public void openClamps() {
        try (AutoTimed at = new AutoTimed("openClamps")) {
            if (!isHomingDone()) {
                throw new RejectedCommandException(name + " homing not done yet for the 3 clamps. Can't open.");
            }
            updateStateAndCheckSensors();
            if (isClosed()) {
                onlineClampYminus.checkReadyForAction();
                // First open clampY
                onlineClampYminus.open();
                // Then open clampX in parallel.
                FcsUtils.parallelRun(() -> onlineClampXminus.open(), () -> onlineClampXplus.open());

                // we brake + disable on clamp lock and clamp open

            } else if (isOpened()) {
                FCSLOG.info(name + " clamps already OPENED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be closed before.");
            }
        }
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps have
     * to be CLOSED. At the end of this action, the clamps are CLOSED but a strong
     * pressure to hold safely the clamps.
     */
    public void lockClamps() {
        try (AutoTimed at = new AutoTimed("lockClamps")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                // TODO can we do the three in parallel ? Should be tested and validated
                onlineClampYminus.lock();
                FcsUtils.parallelRun(() -> onlineClampXminus.lock(), () -> onlineClampXplus.lock());

                // check TODO we reproduce the former logic, the disable is done only when all
                // three clamps
                // are locked. Individual locks do not disable.
                // in pre-refactored code disable was done in endAction of 3claps, not endAction
                // of clamp...

                // change : we brake + disable on clamp lock and clamp open

                // FcsUtils.parallelRun(() -> onlineClampYminus.getController().goToSwitchOnDisabled(),
                //         () -> onlineClampXminus.getController().goToSwitchOnDisabled(),
                //         () -> onlineClampXplus.getController().goToSwitchOnDisabled());

                // FCSLOG.info("online clamps controllers " + onlineClampYminus.getController().isEnabled() + " "
                //         + onlineClampXminus.getController().isEnabled() + " "
                //         + onlineClampXplus.getController().isEnabled());
                if (onlineClampYminus.getController().isEnabled() || onlineClampXminus.getController().isEnabled()
                        || onlineClampXplus.getController().isEnabled()) {
                    FCSLOG.info("online clamps controllers sleep try again");
                    FcsUtils.sleep(50, "lockClamps");
                    FCSLOG.info("online clamps controllers " + onlineClampYminus.getController().isEnabled() + " "
                            + onlineClampXminus.getController().isEnabled() + " "
                            + onlineClampXplus.getController().isEnabled());
                }
            } else if (isLocked()) {
                FCSLOG.info(name + " is already LOCKED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " have to be CLOSED before being locked.");
            }
        }
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to decrease
     * pressure on the clamps. The clamps have to be LOCKED. At the end of this
     * action, the clamps are CLOSED with a small pressure of the clamp hardware on
     * the filter frame.
     */
    public void unlockClamps() {
        try (AutoTimed at = new AutoTimed("unlockClamps")) {
            updateStateAndCheckSensors();
            if (isLocked()) {
                onlineClampYminus.unlock();
                FcsUtils.parallelRun(() -> onlineClampXminus.unlock(), () -> onlineClampXplus.unlock());

                FcsUtils.parallelRun(() -> onlineClampYminus.getController().goToSwitchOnDisabled(),
                () -> onlineClampXminus.getController().goToSwitchOnDisabled(),
                () -> onlineClampXplus.getController().goToSwitchOnDisabled());

            } else if (isClosed()) {
                FCSLOG.info(name + " is already CLOSED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " have to be LOCKED before being unlocked.");
            }
        }
    }

    /**
     * to be used by setFilter
     *
     */
    public void lockFilterAtOnline() {
        try (AutoTimed at = new AutoTimed("lockFilterAtOnline")) {
            if (isLocked()) {
                FCSLOG.info(name + " onlineClamps LOCKED : nothing to do");
            } else if (isClosed()) {
                lockClamps();
            } else if (isOpened()) {
                if (!isHomingDone()) {
                    homing();
                }
                closeClamps();
                lockClamps();
            }
        }
    }

    /**
     * Use by command unclampAndMoveFilterToHandoff. For tests only.
     */
    public void unlockAndOpen() {
        if (isLocked()) {
            unlockClamps();
        }
        if (isClosed()) {
            if (isHomingDone()) {
                openClamps();
            } else {
                /* command homing open clamps in CURRENT mode*/
                homing();
            }
        }
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready.")
    public boolean myDevicesReady() {
        return onlineClampXminus.myDevicesReady() && onlineClampXplus.myDevicesReady()
                && onlineClampYminus.myDevicesReady();
    }

    private short computeNormalizedStrain() {
        /*
         * because computeNormalizedStrain is called in postStart we can't use
         * tempClampMotorXplus like temp = tempClampMotorXplus.getValue() / 10
         *
         */
        /* tempClampMotorXplus is always == 0 in postStart */
        double temp = 20;
        /* give a chance to tempSensorsDevice1 to be alive */
        if (!tempSensorsDevice1.isBooted()) {
            tempSensorsDevice1.updateDeviceInfo();
        }
        if (tempSensorsDevice1.isBooted()) {
            temp = (double) tempSensorsDevice1.readChannel(3) / 10;
        } else {
            this.raiseWarningOnlyIfNew(AC_SENSOR_ERROR,
                    " can't read temperature because tempSensorsDevice1 is not booted,"
                            + " temperature is supposed to be 20°C",
                    name);
        }
        /* give a chance to onlineStrainGauge to be alive */
        if (!onlineStrainGauge.isBooted()) {
            onlineStrainGauge.updateDeviceInfo();
        }
        if (onlineStrainGauge.isBooted()) {
            readStrain = onlineStrainGauge.readStrain();
        } else {
            this.raiseWarningOnlyIfNew(AC_SENSOR_ERROR,
                    " can't read strainGain because onlineStrainGauge is not booted", name);
        }
        normalizedStrain = (short) ((double) readStrain - (strainGain * (temp - 20)));
        FCSLOG.info("temperature=" + temp + " / ONLINE read strain=" + readStrain + " / normalized strain="
                + normalizedStrain);
        return normalizedStrain;
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setLockStatus(lockStatus);
        status.setOnlineClampStrain(normalizedStrain);
        status.setOnlineClampRawStrain(readStrain);
        status.setHomingDone(isHomingDone());
        return status;
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByThreeClamps()));
    }

    protected void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Update state from sensors values.")
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac3clamps")) {
            for (AutochangerOnlineClamp clamp : clampsList) {
                clamp.updateState();
            }
            normalizedStrain = computeNormalizedStrain();
            computeLockStatus();
            this.publishData();
        }
    }

    /**
     * compute lockStatus from the 3 ONLINE clamps lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isClosed()) {
            this.lockStatus = CLOSED;

        } else if (isLocked()) {
            this.lockStatus = LOCKED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else {
            this.lockStatus = UNKNOWN;
        }

    }

    /**
     * return a lock status computed from strain read on onlineStrainGauge.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return a lock status computed from the normalizedStrain value")
    public FcsEnumerations.LockStatus computeStrainGaugeLockStatus() {
        normalizedStrain = computeNormalizedStrain();
        FCSLOG.info(name + " ONLINE strain = " + normalizedStrain);

        if (normalizedStrain > maxClosedStrain) {
            return OPENED;

        } else if (normalizedStrain > maxLockedStrain) {
            return CLOSED;

        } else if (normalizedStrain > minLockedStrain) {
            return LOCKED;

        } else {
            return ERROR;
        }
    }

    /**
     * Compute lockStatus in reading gauge strain. This is done only the first time
     * updateState is called. After computeLockStatus is called.
     */
    private void computeLockStatusFromStrain() {

        FcsEnumerations.LockStatus strainLockStatus = computeStrainGaugeLockStatus();
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else if (isClosed()) {
            if (strainLockStatus == LOCKED) {
                this.lockStatus = LOCKED;

            } else if (strainLockStatus == CLOSED) {
                this.lockStatus = CLOSED;

            } else if (strainLockStatus == OPENED || strainLockStatus == ERROR) {
                // The strainLockStatus is either OPENED or ERROR
                this.lockStatus = INTRAVEL;

            } else {
                this.lockStatus = ERROR;
            }

        } else {
            /*
             * if lockStatus is not the same for the 3 clamps we are in this case. It can
             * happen during the closing (or the opening) of the 3 clamps when a clamp is
             * already closed but not the 2 others.
             */
            this.lockStatus = UNKNOWN;
        }
    }

    /**
     * initialize AC online clamps hardware after initialization. to be executed if
     * during boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Initialize AC online clamps hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        this.postStart();
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.postStart();
        }
    }

    @Override
    public void postStart() {
        FCSLOG.fine(() -> name + " BEGIN postStart.");
        if (onlineStrainGauge.isBooted()) {
            onlineStrainGauge.initializeAndCheckHardware();
        } else {
            onlineStrainGauge.raiseAlarmIfMissing();
        }
        if (tempSensorsDevice1.isBooted()) {
            try {
                computeLockStatusFromStrain();
                lockStatusInitialized = true;
                this.publishData();
            } catch (SDORequestException ex) {
                this.raiseWarning(CAN_BUS_READING_ERROR, "could not computeLockStatusFromStrain ", name, ex);
            }
        } else {
            tempSensorsDevice1.raiseWarningIfMissing();
        }

        FCSLOG.fine(() -> name + " END postStart.");
    }
}
