
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import java.util.logging.Logger;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.SWITCH_ON_DISABLED;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;

/**
 * This class is to simulate an EPOS Controller.
 *
 * @author virieux
 */
public class SimuEPOSController extends CanOpenEPOS {
    private static final Logger FCSLOG = Logger.getLogger(SimuEPOSController.class.getName());

    protected int targetPosition;
    protected int ssiPosition;
    protected int targetCurrent;
    protected int actualCurrent;

    /**
     * Build a new SimuEPOSControllerModule
     */
    public SimuEPOSController() {
        this.position = 0;
        this.mode = PROFILE_POSITION;
        this.simuEnabled = false;
        this.statusWord = 0x40;
        this.eposState = SWITCH_ON_DISABLED;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "For simulator only : Update position with a position given as argument.")
    @Override
    public void setPosition(int actualPosition) {
        this.position = actualPosition;
    }


    /**
     * overridden for simulation
     * @return
     */
    @Override
    public boolean isTargetReached() {
        return this.position == this.targetPosition
                || this.actualCurrent == this.targetCurrent;
    }

    @Override
    public void enable() {
        goToOperationEnable();
        this.simuEnabled = true;
    }

    @Override
    public void goToOperationEnable() {
        super.goToOperationEnable();
        this.simuEnabled = true;
    }

    @Override
    public void goToSwitchOnDisabled() {
        disableVoltage();
        this.simuEnabled = false;
    }

    /**
     *
     * @return
     */
    @Override
    public boolean isEnabled() {
        return simuEnabled;
    }

    /**
     * We assume that simulated hardware'subs controller has always correct parameters.
     * @return
     */
    @Override
    public boolean isParametersOK() {
        return true;
    }



    @Override
    public void defineAbsolutePosition(int position)  {
        this.position = position;
    }

    @Override
    public void writeParameters(EposMode mode) {
    }

    @Override
    public void writeTargetPosition(int position)  {
        this.position = position;
    }

    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.actualCurrent = aValue;
    }

    @Override
    public void stopAction()  {
    }

    @Override
    public long readParameter(EPOSEnumerations.Parameter parameter) {
        switch (parameter) {
            case ProfileVelocity:
                return getProfileVelocity();

            case ProfileAcceleration:
                return getProfileAcceleration();

            case ProfileDeceleration:
                return getProfileDeceleration();

            default:
                return 0;

        }
    }

    @Override
    public int readPosition()  {
        return this.position;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     */
    @Override
    public int readSSIPosition()  {
        return this.ssiPosition;
    }


    @Override
    public int readCurrent()  {
        return actualCurrent;
    }

    @Override
    public EposMode readMode()  {
        return mode;
    }

    @Override
    public int readStatusWord() {
        return statusWord;
    }



    @Override
    public void quickStop()  {
        FCSLOG.finest(() -> name + " quickStop done.");
    }

    @Override
    public int getErrorHistoryNB()  {
        return 0;
    }

    @Override
    public int readErrorRegister() {
        return 0;
    }

    @Override
    public void updateErrorHistory() {
        errorHistory = new int[0];
    }

    @Override
    public String displayErrorHistory()  {
        return "No ERROR";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
    }

    @Override
    public long readProfileVelocity() {
        //TODO return a value of current in random between min and max
        return 0;
    }

    @Override
    public String readParameters(EposMode mode)  {
        return getName() + ":no parameter to read.";
    }

    @Override
    public void checkParameters(EposMode aMode) {
        /*nothing to do because no controller CPU*/
    }


    @Override
    public void writeControlWord(int w)  {
        FCSLOG.finer(name + " writeControlWord : 0x" + Integer.toHexString(w) + " (in decimal:" + w + ")");
        switch (w) {
            case 7:
                //disableOperation
                this.statusWord = 0x23;
                break;
            case 0:
                //disableVoltage
                this.statusWord = 0x40;
                break;
            case 6:
                //shutdownController
                this.statusWord = 0x21;
                break;
            case 0xF:
                //switchOnEnableOperation
                this.statusWord = 0x27;
                break;
            default:
                this.statusWord = w;
                break;
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT,
            description = "Enable controller and go to relative position. Doesn't check condition. DANGER !!!! ")
    @Override
    public void enableAndWriteRelativePosition(int pos) {
        enable();
        changeMode(PROFILE_POSITION);
        writeTargetPosition(this.position + pos);
        writeControlWord(0x7F);
    }


    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart");
        FCSLOG.info(name + " END postStart");
    }

}
