package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutochangerLatch;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

import java.util.logging.Logger;


/**
 * To simulate a controller which can open or close a latch in the autochanger.
 * @author virieux
 */
public class SimuAutochangerLatchController extends SimuEPOSController {
    private static final Logger FCSLOG = Logger.getLogger(SimuAutochangerLatchController.class.getName());
    @ConfigurationParameter(isFinal = true, category = "controller")
    private volatile String latchName;

    private AutochangerLatch latch;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private SimuAutochangerPlutoGateway plutoGateway;

    @LookupPath
    private String nodePath;

    /**
     * Build a SimuAutochangerLatchController.
     */
    public SimuAutochangerLatchController() {
        this.mode = CURRENT;
    }

    @Override
    public void init() {
        if ( plutoGateway == null ) {
            throw new RuntimeException("Invalid initialization. An instance of SimuAutochangerPlutoGateway should have been "
                    + "picked up from the Lookup tree from the siblinds of this node: "+nodePath);
        }
        FCSLOG.finer(name + ":latchName=" + latchName);
        ComponentLookup lookup = subs.getComponentLookup();
        latch = (AutochangerLatch) lookup.getComponentByPath(latchName);
        if ( latch == null ) {
            throw new RuntimeException("Could not find latch "+latchName);
        }
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.targetCurrent = aValue;
        int currentInitial = this.actualCurrent;
        FCSLOG.finer("=>initial current=" + actualCurrent);
        int stepsNB = 10;
        int step = (targetCurrent - this.actualCurrent) / stepsNB;

        for (int i = 1; i <= stepsNB; i++) {
            if (i < stepsNB) {
                this.actualCurrent = (short) (currentInitial + (i * step));
            } //because in the last iteration the actual current could be
            //different from the target value.
            else {
                this.actualCurrent = aValue;
            }
            latch.updateCurrent();
            if (actualCurrent == -latch.getCurrentToOpen()) {
                FCSLOG.finer("plutoGateway=" + plutoGateway.toString());
                this.plutoGateway.simulateAutochangerLatchIsLocked(latchName);

            } else if (actualCurrent == latch.getCurrentToOpen()) {
                this.plutoGateway.simulateAutochangerLatchIsUnlocked(latchName);
            }
            latch.publishData();
            FCSLOG.finer("i=" + i + ",actualCurrent=" + actualCurrent);
            try {
                Thread.sleep(10);
                if (latch.getHaltRequired().get()) {
                    FCSLOG.finer(name + " STOP simulated latch motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                throw new FcsHardwareException(name + " sleep was interrupted.",ex);
            }

        }
    }

    @Override
    public void stopAction()  {
        writeCurrent((short)0);
    }


}
