
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

import java.util.logging.Logger;

/**
 * A model for an autochanger truck.
 * The goal of this class is to factorise the code which reads AC truck sensors.
 * On the Autochanger there are 2 trucks : one on each side :
 * truckXminus and truckXplus.
 * @author virieux
 */
public class AutochangerTruck implements MovedByEPOSController, HasLifecycle, AlertRaiser {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerTruck.class.getName());

    @LookupField(strategy = TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = TOP)
    private Subsystem s;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    private String name;

    @LookupPath
    protected String path;
    public static final String CURRENT_MONITOR_TASK_NAME = "-monitorCurrent";

    private final ComplementarySensors handoffPositionSensors;
    private final ComplementarySensors onlinePositionSensors;
    private final ComplementarySensors standbyPositionSensors;

    private final EPOSControllerWithBrake controller;

    private int position;

    /**
     * Builds a AutochangerTruckModule with 3 couple of complementary sensors to
     * detect position of the truck at ONLINE, HANDOFF and STANDBY position.
     *
     * @param controller
     * @param handoffPositionSensors
     * @param onlinePositionSensors
     * @param standbyPositionSensors
     */
    public AutochangerTruck(EPOSControllerWithBrake controller, ComplementarySensors handoffPositionSensors,
            ComplementarySensors onlinePositionSensors, ComplementarySensors standbyPositionSensors) {
        this.controller = controller;
        this.handoffPositionSensors = handoffPositionSensors;
        this.onlinePositionSensors = onlinePositionSensors;
        this.standbyPositionSensors = standbyPositionSensors;
    }

    @Override
    public EPOSController getController() {
        return controller;
    }

    @Override
    public Subsystem getSubsystem() {
        return s;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * *** lifecycle methods *************************************************
     */
    /**
     *
     */
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerTruck.class, path);
        FCSLOG.info(() -> name + " is controlled by " + controller.getName());
        /**
         * Monitor current read on controller.
         */
//        periodicTaskService
//                .scheduleAgentPeriodicTask(new AgentPeriodicTask(name + CURRENT_MONITOR_TASK_NAME, this::monitorCurrent)
//                        .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofSeconds(60)));
    }

    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(SDO_ERROR.getAlert(controller.getName()), alwaysClear);
    }


    /**
     * @return true if truck is at HANDOFF position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if truck is at HANDOFF position.")
    public boolean isAtHandoff() {
        return handoffPositionSensors.isOn();
    }

    /**
     * @return true if truck is at ONLINE position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if truck is at ONLINE position.")
    public boolean isAtOnline() {
        return onlinePositionSensors.isOn();
    }

    /**
     * @return true if truck is at STANDBY position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if truck is at STANDBY position.")
    public boolean isAtStandby() {
        return standbyPositionSensors.isOn();
    }

    /**
     * Return false if the 2 redondant position sensors at Standby are equal.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return false if the 2 redondant position sensors at Standby are equal. "
                    + "Doesn't read again sensors.")
    public boolean isStandbySensorsInError() {
        return standbyPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Handoff are equal.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return false if the 2 redondant position sensors at Handoff are equal."
                    + "Doesn't read again sensors.")
    public boolean isHandoffSensorsInError() {
        return handoffPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Online are equal.
     * Doesn't read again sensors.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return false if the 2 redondant position sensors at Online are equal."
                    + "Doesn't read again sensors.")
    public boolean isOnlineSensorsInError() {
        return onlinePositionSensors.isInError();
    }

    /**
     * Return true if position sensors are in error.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if position sensors are in error."
                + "Doesn't read again sensors.")
    public boolean isPositionSensorsInError() {
        return isOnlineSensorsInError() || isHandoffSensorsInError() || isStandbySensorsInError();
    }

    /**
     * Return truck position unregistered in field position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return truck position.")
    public int getPosition() {
        return position;
    }

    /**
     * Updates the field position of the truck in reading the CPU of the controller.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update truck position in reading controller.")
    public void updatePosition() {
        try {
            controller.updatePositionFromSDO();
            this.position = controller.getPosition();
            this.publishData();

        } catch (SDORequestException ex) {
            raiseWarning(SDO_ERROR, "=> ERROR IN READING CONTROLLER:", controller.getName(), ex);
        }
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerTruck createStatusDataPublishedByAutoChangerTruck() {
        StatusDataPublishedByAutochangerTruck s = new StatusDataPublishedByAutochangerTruck();
        s.setHandoffInError(handoffPositionSensors.isInError());
        s.setOnlineInError(onlinePositionSensors.isInError());
        s.setStandbyInError(standbyPositionSensors.isInError());
        s.setHandoffSensorValue(handoffPositionSensors.isOn());
        s.setOnlineSensorValue(onlinePositionSensors.isOn());
        s.setStandbySensorValue(standbyPositionSensors.isOn());
        s.setPosition(position);
        return s;
    }

    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByAutoChangerTruck()));
        controller.publishData();
    }

}
