package org.lsst.ccs.subsystems.fcs.common;

import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.HOMING;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_VELOCITY;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.VELOCITY;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.FAULT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.OPERATION_ENABLE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.QUICKSTOP;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.READY_TO_SWITCH_ON;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.SWITCHED_ON;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.SWITCH_ON_DISABLED;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.HomePosition;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.HomingMethod;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.PositionActualValue;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.PositionSensorType;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileVelocity;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.SSIEncoderActualPosition;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import java.util.Map;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ControllerStructure;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;

import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.CurrentThresholdHomingMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.HomeOffset;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedDuringMotionNoEncoder;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedDuringMotionWithEncoder;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 *
 * @author virieux cf doc EPOS2_Firmware_Specification § 3.2 Device Control
 */
public interface EPOSController extends PieceOfHardware, HasLifecycle {

    int getStatusWord();

    /**
     * Read StatusWord on controller CPU.
     *
     * @return
     */
    int readStatusWord();

    void updateStatusWord();

    short readControlWord();

    void setEposState(EposState state);

    EposState getEposState();

    /**
     * Write controlWord to the controller. Use to send or start commands. See EPOS
     * documentation.
     *
     * @param w
     * @throws FcsHardwareException
     */
    void writeControlWord(int w);

    /**
     * print status word bit by bit for end user and tests
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "print status word in binary.")
    default String printStatusWordInBinary() {
        int sw = readStatusWord();
        return "0b" + Integer.toBinaryString(sw);
    }

    /**
     * shutdowns controller cf doc EPOS2_Firmware_Specification § 3.2 Device Control
     * lower byte of control word: 0xxx x110. This is the transition command to go
     * to state READY_TO_SWITCH_ON.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Shutdown the controller - writeControlWord[6] and check that controller is in state READY_TO_SWITCH_ON.")
    default void shutdownController() {
        writeControlWord(6);
        checkState(READY_TO_SWITCH_ON);
        FCSLOG.debug(getName() + ": is READY_TO_SWITCH_ON.");
        publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if controller state is state given as argument.")
    default boolean isInState(EposState state) {
        return state == getEposState();
    }

    static boolean isReadyToSwitchOn(int statusWord) {
        // bit patterns from 3.2.1 State of the Drive in
        // x0xx xxx1 x010 0001
        // x010 0001
        // 0111 1111 = 0x7F
        // 0010 0001 = 0x21

        // return (statusWord & 0x7F) == 0x21;
        return (statusWord & 0x6F) == 0x21;
    }

    static boolean isSwitchedOn(int statusWord) {
        // x0xx xxx1 x010 0011
        // x010 0011
        // 0111 1111 = 0x7F
        // 0010 0011 = 0x23
        // return (statusWord & 0x7F) == 0x23;
        return (statusWord & 0x6F) == 0x23;
    }

    static boolean isSwitchOnDisabled(int statusWord) {
        // x0xx xxx1 x100 0000
        // x100 0000
        // 0111 1111 = 0x7F
        // 0100 0000 = 0x40
        // return (statusWord & 0x7F) == 0x40;
        return (statusWord & 0x4F) == 0x40;
    }

    static boolean isOperationEnabled(int statusWord) {
        // x0xx xxx1 x011 0111
        // x011 0111
        // 0111 1111 = 0x7F
        // 0011 0111 = 0x37
        // return (statusWord & 0x7F) == 0x37;
        return (statusWord & 0x6F) == 0x27;
    }

    static boolean isFault(int statusWord) {
        // ok this gather several fault states
        return (statusWord & 0x8) == 8;
    }

    /**
     * QUICKSTOP is state where bits 5 and 6 are == 0
     *
     * @param statusWord
     * @return
     */
    static boolean isQuickstop(int statusWord) {
        // x0xx xxx1 x001 0111
        // x001 0111
        // 0111 1111 = 0x7F
        // 0001 0111 = 0x17
        return (statusWord & 0x7F) == 0x17;

        // return (statusWord | 0x9F) == 0x9F;
    }

    /**
     * read statusWord by PDO, then compute and set eposState from statusWord.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read statusWord by SDO and update eposState.")
    default void updateEposState() {
        int statusWord = readStatusWord();
        updateEposState(statusWord);
    }

    default void updatePositionFromSDO() {
        setPosition(readPosition());
    }

    /**
     * Compute and set eposState from statusWord given as argument.
     *
     * @param statusWord
     */
    default void updateEposState(int statusWord) {
        if (isFault(statusWord)) {
            setEposState(FAULT);
        } else if (isSwitchOnDisabled(statusWord)) {
            setEposState(SWITCH_ON_DISABLED);
        } else if (isQuickstop(statusWord)) {
            FCSLOG.info("QUICKSTOP state " + getName() + " statusword " + Integer.toBinaryString(readStatusWord())
                    + " error register " + getErrorRegister() + " last error code " + getLastErrorCode());
            if (this instanceof CanOpenEPOS) {
                FCSLOG.info("QUICKSTOP 0x6007" + ((CanOpenEPOS) this).readSDO(0x6007, 0) + " 0x605E "
                        + ((CanOpenEPOS) this).readSDO(0x605E, 0));
            }
            setEposState(QUICKSTOP);
        } else if (isReadyToSwitchOn(statusWord)) {
            setEposState(READY_TO_SWITCH_ON);
        } else if (isSwitchedOn(statusWord)) {
            setEposState(SWITCHED_ON);
        } else if (isOperationEnabled(statusWord)) {
            setEposState(OPERATION_ENABLE);
        } else {
            setEposState(UNKNOWN_STATE);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check controller state until a timeout of 500ms. If controller never goes in state given as argument, throw an Exception.")
    default void checkState(EposState state) {
        long timeoutMillis = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean state_ok = false;
        while (!state_ok && duration <= timeoutMillis) {
            FcsUtils.sleep(2, getName());
            duration = System.currentTimeMillis() - timeStart;
            updateEposState();
            state_ok = isInState(state);
        }
        if (state_ok) {
            FCSLOG.info(getName() + String.format(" go to state %s duration = %d", state, duration));
        } else {
            String msg = getName()
                    + String.format(" couldn't go to state %s during time allocated of %d ms controller state is %s",
                            state, timeoutMillis, getEposState());
            FCSLOG.error(msg);

            FCSLOG.severe("controller " + getName() + " in " + state + " statusword "
                    + Integer.toBinaryString(readStatusWord()) + " error register " + getErrorRegister()
                    + " last error code " + getLastErrorCode() + " " + getLastErrorName());

            throw new FcsHardwareException(msg);
        }
    }

    /**
     * cf doc EPOS2_Firmware_Specification § 3.2 Device Control lower byte of
     * control word: 0xxx xx0x
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Disable voltage - writeControlWord[0] - and check that controller state "
            + "is SWITCH_ON_DISABLED. This command does not activate holding brake. "
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void disableVoltage() {
        try (AutoTimed at = new AutoTimed("disableVoltage-EPOS-" + getName())) {
            writeControlWord(0);
            checkState(SWITCH_ON_DISABLED);
            FCSLOG.info(getName() + " High-level power is switched off.");
            publishData();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Got to state SWITH_ON_DISABLED."
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void goToSwitchOnDisabled() {
        try (AutoTimed at = new AutoTimed("goToSwitchOnDisabled-EPOS-" + getName())) {
            checkFault();
            updateEposState();
            if (isInState(OPERATION_ENABLE)) {
                disableOperation();
                disableVoltage();
            } else if (isInState(SWITCHED_ON) || isInState(READY_TO_SWITCH_ON) || (isInState(QUICKSTOP))) {
                disableVoltage();
            } else if (isInState(SWITCH_ON_DISABLED)) {
                FCSLOG.info(getName() + " Already SWITCH_ON_DISABLED.");
            } else {
                disableVoltage();
                FCSLOG.error(getName() + String.format(" is in BAD STATE %s. This should not happen.", getEposState()));
            }
        }
    }

    /**
     * Disable operation. This command activates holding brake. This is a command
     * transition from state OPERATION_ENABLE to SWITCHED_ON.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Disable operation - writeControlWord[7] - and check that controller state "
            + "is SWITCHED_ON. This command activates holding brake. "
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void disableOperation() {
        try (AutoTimed at = new AutoTimed("EPOS-disableOperation-" + getName())) {
            writeControlWord(7);
            checkState(SWITCHED_ON);
            FCSLOG.info(getName() + " Drive function is disabled.");
            publishData();
        }
    }

    /**
     * Switch on and enable controller. cf doc EPOS2_Firmware_Specification §3.2
     * Device Control lower byte of control word: 0xxx1111 This is the transition
     * command from SWITCHED_ON to OPERATION_ENABLE.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Switch on and enable controller - writeControlWord[0xF] - and check "
            + "that controller state is in state OPERATION_ENABLE within 500ms.")
    default void switchOnEnableOperation() {
        writeControlWord(0xF);
        checkState(OPERATION_ENABLE);
        FCSLOG.info(getName() + " Drive function is enabled.");
        publishData();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Got to state OPERATION_ENABLE."
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void goToOperationEnable() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("goToOperationEnable-EPOS-" + getName())) {
            FCSLOG.info("goToOperationEnable entry state " + getEposState());
            checkFault();
            updateEposState();
            if (isInState(SWITCH_ON_DISABLED)) {
                shutdownController();
                switchOnEnableOperation();
            } else if (isInState(SWITCHED_ON) || isInState(READY_TO_SWITCH_ON)) {
                // } else if (isInState(SWITCHED_ON) || isInState(READY_TO_SWITCH_ON) ||
                // isInState(QUICKSTOP)) {
                switchOnEnableOperation();
            } else if (isInState(OPERATION_ENABLE)) {
                FCSLOG.info(getName() + " Already in state OPERATION_ENABLE.");
            }
//            else {
//                // after tests on hardware, this case should not happen.
//                throw new FcsHardwareException(
//                        getName() + String.format(" is in BAD STATE %s. This should not happen.", getEposState()));
//            }
        }
    }

    /***************************************************************************
     * End of Commands to change CANopen state
     ***************************************************************************
     */

    /**
     * Stop action and shutdownController(shutdown) controller.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Stop action and goToSwitchOnDisabled controller.")
    default void stopAction() {
        if (getMode() == HOMING) {
            this.writeControlWord(0x11F);

        } else if (getMode() == CURRENT) {
            /* stop current */
            writeCurrent((short) 0);

        } else if (getMode() == PROFILE_POSITION) {
            this.stopPosition();

        } else if (getMode() == PROFILE_VELOCITY) {
            this.stopVelocity();
        }
        goToSwitchOnDisabled();
        FCSLOG.info(getName() + " ACTION STOPPED");
    }

    /**
     * Send a quickStop command to the controller.
     *
     * see EPOS2_Application_Notes_Collection_En.pdf §8 page 8-104
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "QuickStop.")
    default void quickStop() {
        FCSLOG.debug(getName() + " running QUICKSTOP command.");
        switch (getMode()) {
            case HOMING:
            case PROFILE_POSITION:
            case VELOCITY:
                writeControlWord(0xB);
                break;
            case CURRENT:
                // quickStop has no effect in CURRENT mode so current has to be set to 0
                writeCurrent(0);
                break;
            case MASTER_ENCODER:
                // nothing to be done : command quickStop has to be sent to driver and that stops also follower.
                break;
            default:
                throw new IllegalArgumentException(getName() + " has invalid Epos mode:" + getMode());
        }
    }

    /**
     * This methods enables the controller : i.e. this makes it able to receive
     * commands. This is the transition command from SWITCH_ON_DISABLED to
     * OPERATION_ENABLE. It does : shutdownController, then switchOnEnableOperation.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "This command enables the controller : i.e. this makes it able to receive commands."
            + "it does shutdownController, then switchOnEnableOperation.")
    default void enable() {
        shutdownController();
        switchOnEnableOperation();
    }

    /**
     * Read statusWord, update eposState and returns true if the controller is in
     * state OPERATION_ENABLE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the controller is enabled (statusWord & 0x6F) == 0x27).")
    default boolean isEnabled() {
        int statusWord = readStatusWord();
        updateEposState(statusWord);
        return isOperationEnabled(statusWord);
    }
    /**
     * *************************************************************************************
     */
    /****************************************************************************************/
    /*
     * end of Methods to enable, shutdownController, shutdown controller /
     ****************************************************************************************/
    /****************************************************************************************/

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to read, write or check EPOS mode in the CPU of the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * This methods changes the mode to the new mode given as an argument. It writes
     * the new mode in the CPU of the EPOS controller and updates the field mode.
     *
     * @param newMode
     * @throws SDORequestException
     */
    void changeMode(EposMode newMode);

    /**
     * read the EPOS mode on the controller CPU.
     *
     * @return mode
     * @throws FcsHardwareException
     */
    EposMode readMode();

    /**
     * Return the EPOS mode stored in field mode. Doesn't read again controller CPU.
     *
     * @return
     */
    EposMode getMode();

    /**
     * return true if this controller is in the mode given as argument.
     *
     * @param aMode
     * @return
     */
    default boolean isInMode(EposMode aMode) {
        return aMode.equals(getMode());
    }

    default void checkEposMode(EposMode aMode) {
        EposMode myMode = readMode();
        if (aMode != myMode) {
            throw new FcsHardwareException(getName() + " is not in mode " + aMode);
        }
    }

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * end of Methods to read, write or check EPOS mode in the CPU of the controller
     * /
     ****************************************************************************************/
    /****************************************************************************************/

    /**
     * Read actual current on the controller CPU. (object 6078)
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    int readCurrent();

    /**
     * Read Average current on controller CPU. (object 2027)
     *
     * @return
     */
    int readCurrentAverageValue();

    /**
     * read velocity on the controller CPU.
     *
     * @return
     */
    int readVelocity();

    /**
     * read Following Error on controller CPU.
     *
     * @return
     */
    int readFollowingError();

    /**
     * In current mode this methods send a current to the motor. This make run the
     * motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "In current mode this methods send a current to the motor. index:0x2030 subindex:0 size:2")
    void writeCurrent(int aValue) throws EPOSConfigurationException;

    /**
     * Enable, set mode CURRENT and write current. For end user at console.
     *
     * @param aValue
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Enables controller and sets the current value in the EPOS CPU.")
    default void enableAndWriteCurrent(int aValue) {
        goToOperationEnable();
        writeCurrent(aValue);
    }

    /**
     * Defines the actual position as the absolute position which value is given as
     * an argument. Writes value 35 to set the Homing method as Actual (See EPOS
     * documentation)
     *
     * @param position
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Define the actual position as position given as argument.")
    default void defineAbsolutePosition(int position) {
        FCSLOG.debug(getName() + " Defining Absolute Position:" + position);
        int sleepTime = 50;
        this.changeMode(HOMING);
        this.writeParameter(HomePosition, position);
        this.writeParameter(HomingMethod, 35);
        /*
         * Application_Notes_Collection $8
         */
        // TODO replace by goToSwithOnDisabled
        this.writeControlWord(6);
        FcsUtils.sleep(sleepTime, getName());
        this.writeControlWord(0xF);
        FcsUtils.sleep(sleepTime, getName());
        /* start homing */
        this.writeControlWord(0x1F);
        checkHomingDone();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_EXPERT, description = "Check if homing has been done otherwise throws an Exception.")
    default void checkHomingDone() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " homing target not yet reached. duration=" + duration);
            duration = System.currentTimeMillis() - timeStart;
        }
        if (isTargetReached()) {
            writeControlWord(0xF);
            FCSLOG.info(getName() + " ==> END homing");
        } else {
            throw new FcsHardwareException(getName() + " couldn't do homing : "
                    + "target is not reached within timeout of " + timeout + "milliseconds.");
        }
    }

    default void definePositionFromNegativeLimitSwitch() {
        FCSLOG.debug(getName() + " Homing with homing method negative limit switch");
        this.changeMode(HOMING);
        this.writeParameter(HomePosition, 0);
        this.writeParameter(HomingMethod, 17);
        this.goToOperationEnable();
        /* start homing */
        this.writeControlWord(0x1F);
    }

    /**
     * Homing for loader clamp : this command opens clamps until a dead end.
     *
     * @param currentThreshold current to send to controller to do the homing
     */
    default void defineHomingCurrentThresholdNegativeSpeed(int currentThreshold) {
        FCSLOG.debug(getName() + " Homing with method Current Threshold Negative Speed");
        changeMode(HOMING);
        writeParameter(HomeOffset, 1000);
        writeParameter(CurrentThresholdHomingMode, currentThreshold);
        writeParameter(HomingMethod, -4);
        goToOperationEnable();
        /* start homing */
        writeControlWord(0x1F);
    }

    /**
     * Reads in the CPU the value of the parameter PositionActualValue and returns
     * it in a decimal format.
     *
     * @return value in decimal format
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "In PROFILE_POSITION Mode this methods returns the actual position. "
            + "index:0x6064 subindex:0")
    default int readPosition() {
        int pos = (int) readParameter(PositionActualValue);
        setPosition(pos);
        return pos;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read HomingMethod parameter")
    default byte readHomingMethod() {
        return (byte) readParameter(HomingMethod);
    }

    int getCurrent();

    int getPosition();

    int getVelocity();

    int getAverageCurrent();

    int getFollowingError();

    long getProfileAcceleration();

    long getProfileDeceleration();

    long getProfileVelocity();

    void setCurrent(int cur);

    void setPosition(int pos);

    void setVelocity(int vel);

    void setProfileVelocity(long velocity);

    void setProfileAcceleration(long acceleration);

    void setProfileDeceleration(long deceleration);

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read the position returned by the absolute encoder (single serial data) "
            + "index:0x2211 subindex:3")
    default int readSSIPosition() {
        return (int) this.readParameter(SSIEncoderActualPosition);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read the controller structure : dual loop or not. Should be 0 or 1"
            + "index:0x2220 subindex:0")
    default int readControllerStructure() {
        return (int) this.readParameter(ControllerStructure);
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter ProfileVelocity.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "In PROFILE_POSITION returns the value of the parameter ProfileVelocity. "
            + "index:0x6081 subindex:0")
    default long readProfileVelocity() {
        long profileVelocity = this.readParameter(ProfileVelocity);
        setProfileVelocity(profileVelocity);
        FCSLOG.debug(getName() + ":readProfileVelocity=" + profileVelocity);
        return profileVelocity;
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter
     * ProfileAcceleration.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "In PROFILE_POSITION returns the value of the parameter ProfileAcceleration. "
            + "index:0x6083 subindex:0")
    default long readProfileAcceleration() {
        long profileAcceleration = this.readParameter(ProfileAcceleration);
        setProfileAcceleration(profileAcceleration);
        FCSLOG.debug(getName() + ":readProfileAcceleration=" + profileAcceleration);
        return profileAcceleration;
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter
     * ProfileDeceleration.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "In PROFILE_POSITION returns the value of the parameter ProfileDeceleration. "
            + "index:0x6084 subindex:0")
    default long readProfileDeceleration() {
        long profileDeceleration = this.readParameter(ProfileDeceleration);
        setProfileDeceleration(profileDeceleration);
        FCSLOG.debug(getName() + ":readProfileDeceleration=" + profileDeceleration);
        return profileDeceleration;
    }

    /**
     * Read parameter PositionSensorType in CPU of controller and return read value.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read PositionSensorType. "
            + "index:0x2210 subindex:2")
    default int readPositionSensorType() {
        return (int) readParameter(PositionSensorType);
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "In PROFILE_POSITION mode this methods set the target position."
            + "index:0x607A subindex:0 size:4")
    void writeTargetPosition(int aValue);

    /**
     * change ProfileVelocity to a new value.
     *
     * @param newVelocity
     */
    default void changeProfileVelocity(int newVelocity) {
        FCSLOG.info(getName() + " changing ProfileVelocity to " + newVelocity);
        try {
            writeParameter(ProfileVelocity, newVelocity);
            setProfileVelocity(newVelocity);
        } catch (Exception ex) {
            String msg = getName() + " : could not change ProfileVelocity to value " + newVelocity;
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /**
     * change ProfileAcceleration to a new value.
     *
     * @param newAcceleration
     */
    default void changeProfileAcceleration(int newAcceleration) {
        FCSLOG.info(getName() + " changing ProfileAcceleration to " + newAcceleration);
        try {
            writeParameter(ProfileAcceleration, newAcceleration);
            setProfileAcceleration(newAcceleration);
        } catch (Exception ex) {
            String msg = getName() + " : could not change ProfileAcceleration to value " + newAcceleration;
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /**
     * change ProfileAcceleration to a new value.
     *
     * @param newDeceleration
     */
    default void changeProfileDeceleration(int newDeceleration) {
        FCSLOG.info(getName() + " changing ProfileDeceleration to " + newDeceleration);
        try {
            writeParameter(ProfileDeceleration, newDeceleration);
            setProfileDeceleration(newDeceleration);
        } catch (Exception ex) {
            String msg = getName() + " : could not change ProfileDeceleration to value " + newDeceleration;
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to read or write EPOS parameters in the CPU of the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * This methods returns true if the values of parameters stored in CPU are the
     * same than those stored in the configuration system.
     *
     * @return
     */
    boolean isParametersOK();

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param parameter
     * @return
     */
    long readParameter(Parameter parameter);

    /**
     * Write on the CPU of the controller a value given in hexadecimal format for a
     * parameter.
     *
     * @param parameter
     * @param val       the value.
     * @throws FcsHardwareException
     */
    void writeParameter(Parameter parameter, int val);

    /**
     * Check parameters for a given EPOS mode : compare the values stored in the
     * Configuration System and the values stored in controller CPU. Raise a
     * FcsHardwareException if some values are not equals.
     *
     * @param aMode
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void checkParameters(EposMode aMode);

    /**
     * This methods writes in the CPU of the EPOS devis the values of the parameters
     * set for the mode. Usually the values of the parameters are given by the
     * configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    void writeParameters(EposMode mode);

    /**
     * Write in the CPU of the EPOS device the values of the parameters set for the
     * mode in configuration system.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Write in the CPU of the EPOS device the values "
            + "of the parameters set for the mode.")
    default void writeParameters() {
        readMode();
        writeParameters(getMode());
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is
     * given as argument.
     *
     * @param parameterName name of the parameter to read
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Reads in the EPOS CPU the decimal value of the Parameter "
            + "which parameter name is given as argument.")
    default long readParameter(String parameterName) {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameter(param);
    }

    /**
     * Read the parameters for the actual mode.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read the parameters for the actual mode.")
    default String readParameters() {
        return readParameters(readMode().toString());
    }

    /**
     * This methods reads in the CPU of the EPOS the values of the parameters for a
     * given mode.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "This methods reads in the CPU of the EPOS the values "
            + "of the parameters for a given mode.")
    default String readParameters(String modeInString) {
        return readParameters(EposMode.valueOf(modeInString));
    }

    /**
     * Read all the value of parameters for a given EPOS mode and return a printed
     * list of this values.
     *
     * @param aMode
     * @return
     * @throws FcsHardwareException
     */
    default String readParameters(EposMode aMode) {
        String modeIS = aMode.toString();
        StringBuilder sb = new StringBuilder(getName() + " parameters VALUES in decimal format for mode ");
        sb.append(modeIS);
        sb.append("\n");
        Parameter[] params = aMode.getParameters();
        for (Parameter param : params) {
            sb.append(param.toString());
            sb.append("=");
            long valueInt = readParameter(param);
            sb.append(valueInt);
            sb.append("\n");
        }
        FCSLOG.debug(getName() + sb.toString());
        return sb.toString();
    }

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    default void writeParameters(Map<String, Integer> paramMap) {
        for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue();
            writeParameter(EPOSEnumerations.Parameter.valueOf(paramName), value);
        }
    }

    /**
     * values transmitted by EPOSController can be : UNSIGNED32, UNSIGNED16,
     * UNSIGNED8 for UNSIGNED parameter INTEGER32, INTEGER16, INTEGER8 for SIGNED
     * parameter This method convert a value sent by device to the value to be used
     * in fcs. Exemple : PolePairNumber is an UNSIGNED8. Real values are always > 0.
     * Value sent by device should not be changed and be always > 0 : 255 => 255
     *
     * HomingMethod is an INTEGER8 Real values can be negative. So value sent by
     * device is casted as byte to have the real value. 255 => -1
     *
     * @param param
     * @param value
     * @param myName
     * @return
     */
    static long convertEPOSValue(Parameter param, long value, String myName) {
        if (!param.isSigned()) {
            return value;
        } else if (param.getSize() == 4) {
            return (int) value;
        } else if (param.getSize() == 2) {
            return (short) value;
        } else if (param.getSize() == 1) {
            return (byte) value;
        } else {
            throw new EPOSConfigurationException(myName + ":\f" + param.getSize() + " => bad value for size");
        }
    }

    /****************************************************************************************/
    /****************************************************************************************/
    /*
     * END of Methods to read or write EPOS parameters in the CPU of the controller
     * /
     ****************************************************************************************/
    /****************************************************************************************/

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to check fault or reset fault in the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * Return true if controller is in fault.
     *
     * @return
     */
    boolean isInError();

    /**
     *
     * @return error register in printed format
     */
    String getErrorRegister();

    /**
     *
     * @return last code error occurred on that device
     */
    int getLastErrorCode();

    /**
     *
     * @return last error name occurred on that device
     */
    String getLastErrorName();

    /**
     * Return the number of errors that occurred on this controller.
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    int getErrorHistoryNB();

    /**
     * Return a printed list of errors that occurred on this controller.
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    String displayErrorHistory();

    /**
     * Check if the controller is in fault. Throw an Exception if the controller is
     * in fault.
     *
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    void checkFault();

    void faultReset();

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * End of Methods to check fault or reset fault in the controller /
     ****************************************************************************************/
    /****************************************************************************************/

    /**
     * In HOMING mode and PROFILE_POSITION mode this indicates that the position is
     * reached.
     *
     * @return true if the position is reached.
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "In HOMING mode and PROFILE_POSITION mode this indicates that the position is reached. "
            + "((readStatusWord() >> 10) & 1) == 1)")
    default boolean isTargetReached() {
        /* target is reached when bit10 of status word is 1 */
        return ((readStatusWord() >> 10) & 1) == 1;
    }

    /**
     * check if target is reached in HOMING or PROFILE_POSITION commands and throws
     * an Exception if not.
     *
     * @param timeout
     */
    default void checkTargetReached(long timeout) {
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " target is not yet reached.");
            duration = System.currentTimeMillis() - timeStart;
        }
        if (!isTargetReached()) {
            String msg = getName() + " couldn't reach target during time allocated of " + timeout + " ms";
            FCSLOG.info(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Enable controller and go to absolute position. Doesn't check condition.
     * DANGER !!!!
     *
     * @param pos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Enable controller and go to absolute position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteAbsolutePosition(int pos) {
        goToOperationEnable();
        writeTargetPosition(pos);
        writeControlWord(0x3F);
    }

    /**
     * Enable controller and go to relative position. Doesn't check condition.
     * DANGER !!!!
     *
     * @param pos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Enable controller and go to relative position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteRelativePosition(int pos) {
        goToOperationEnable();
        writeTargetPosition(pos);
        writeControlWord(0x7F);
    }

    /**
     *
     * Stop motion when in mode PROFILE_POSITION.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Stop motion when in mode PROFILE_POSITION. (writeControlWord(0x010F))")
    default void stopPosition() {
        if (isInMode(PROFILE_POSITION)) {
            this.writeControlWord(0x010F);
        } else {
            throw new RejectedCommandException(getName() + " is not in PROFILE_POSITION mode.");
        }
    }

    /**
     * Stop motion when in mode PROFILE_VELOCITY.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Stop motion when in mode PROFILE_VELOCITY. (writeControlWord(0x0107)")
    default void stopVelocity() {
        if (isInMode(PROFILE_VELOCITY)) {
            this.writeControlWord(0x0107);
        } else {
            throw new RejectedCommandException(getName() + " is not in PROFILE_VELOCITY mode.");
        }
    }

    default void updateDataDuringMotionFromSDO() {
        this.readCurrent();
        this.readCurrentAverageValue();
        this.readFollowingError();
        this.readPosition();
        this.readVelocity();
    }

    /**
     * Creates an object to be published on the STATUS bus by an EPOSController.
     *
     * @return
     */
    default StatusDataPublishedByEPOSController createStatusDataPublishedByEPOSController() {
        StatusDataPublishedByEPOSController status = new StatusDataPublishedByEPOSController(isBooted(),
                isInitialized(), isInError(), getErrorRegister(), getErrorHistoryNB(), getLastErrorCode(),
                getLastErrorName());
        status.setMode(getMode());
        status.setState(getEposState());
        status.setCurrent(getCurrent());
        status.setPosition(getPosition());
        status.setVelocity(getVelocity());
        status.setProfileAcceleration(getProfileAcceleration());
        status.setProfileDeceleration(getProfileDeceleration());
        status.setProfileVelocity(getProfileVelocity());
        status.setAverageCurrent(getAverageCurrent());
        status.setFollowingError(getFollowingError());
        return status;
    }


    default StatusDataPublishedDuringMotionNoEncoder createStatusDataPublishedDuringMotionNoEncoder() {
        StatusDataPublishedDuringMotionNoEncoder status = new StatusDataPublishedDuringMotionWithEncoder();
        status.setCurrent(getCurrent());
        status.setAverageCurrent(getAverageCurrent());
        status.setVelocity(getVelocity());
        return status;
    }

    default StatusDataPublishedDuringMotionWithEncoder createStatusDataPublishedDuringMotionWithEncoder() {
        StatusDataPublishedDuringMotionWithEncoder status = new StatusDataPublishedDuringMotionWithEncoder();
        status.setCurrent(getCurrent());
        status.setAverageCurrent(getAverageCurrent());
        status.setVelocity(getVelocity());
        status.setFollowingError(getFollowingError());
        status.setPosition(getPosition());
        return status;
    }

    /**
     * publish data during motion for trending data base.
     */
    void publishDataDuringMotion();

}
