package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.HashMap;
import java.util.Iterator;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import org.lsst.ccs.bus.data.Alert;

import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.McmState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the CCS
 * Master Control Module (MCM) in normal operation mode or from a Console.
 *
 * <P/>
 * Example of commands that can be sent to the FcsMain : <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>
 * The main goals of the FcsMain is :
 * <UL>
 * <LI>Receive commands from MCM or ccs-console for the whole subsystem.</LI>
 * <LI>Execute commands received from MCM or ccs-console.</LI>
 * <LI>Handle the logic of the whole subsystem.</LI>
 * <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 *
 * @author virieux
 *
 */
public class FcsMain extends MainModule {

    /**
     *
     */
    private static final long serialVersionUID = 7669526660659959402L;

    private static final Duration SETFILTER_MAX_DURATION = Duration.ofSeconds(90);
    private static final Duration LOAD_UNLOAD_FILTER_MAX_DURATION = Duration.ofSeconds(480);

    private static final String PUBLICATION_KEY_FOR_MCM = "fcs/mcm";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;

    @LookupField(strategy = CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = CHILDREN)
    private Autochanger autochanger;

    @LookupField(strategy = CHILDREN)
    private Loader loader;

    @LookupField(strategy = CHILDREN)
    private FilterManager filterManager;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private long setFilterDuration = 0;

    @Override
    public void build() {
        if (bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (bridgeToLoader.equals(bridge)) {
            throw new RuntimeException(
                    "In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();
        //Registering the KeyValudData published below
        dataProviderDictionaryService.registerData(getDataForMcm());
        agentStateService.updateAgentState(McmState.UNKNOWN);
    }

    @Override
    public void init() {
        super.init();
        /*
         * define a role for my subsystem in order to make FcsGUI listen to my subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs",
                "fcs");

        //description for duration/SETFILTER
        DataProviderInfo data = new DataProviderInfo("duration/SETFILTER", DataProviderInfo.Type.TRENDING, "duration/SETFILTER");
        data.addAttribute(DataProviderInfo.Attribute.UNITS, "milliseconds");
        data.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "duration of command setFilter");
        data.addAttribute(DataProviderInfo.Attribute.TYPE, "long");
        // register setFilterDuration with units, description, type.
        dataProviderDictionaryService.addDataProviderInfoToDictionary(data);

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_LOCKING_ISSUE.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert("carousel"), alwaysClear);
    }

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY);
            updateAgentState(FilterReadinessState.READY);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        // because of some sensors still missing, autochanger is never initialized
        // but we need the Filter to be ready for filter exchanges
        // TODO remove after tests
        updateAgentState(FilterState.READY);
        updateAgentState(FilterReadinessState.READY);
    }

    /**
     * For MCM and end user at console.
     *
     * Return the name of the filter which is at ONLINE or "NONE" if no filter
     * is at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns the "
            + "name of the filter which is at ONLINE or NONE if no filter is at ONLINE.")
    public String getMCMOnlineFilterName() {
        if (autochanger.isHoldingFilter() && autochanger.isAtOnline()) {
            return autochanger.getFilterOnTrucksName();

        } else {
            return "NONE";
        }
    }

    /**
     * This methods returns the filter which has for name the String given as
     * argument.
     *
     * @param filterName
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "returns the filter which has for name the String given as argument.")
    public Filter getFilterByName(String filterName) {
        return filterManager.getFilterByName(filterName);
    }

    /**
     * Return true if the changer is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    /**
     * Return true if the loader is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return autochanger.isLoaderConnected();
    }

    /**
     * Return true if the hardware of the changer is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        loader.disconnectLoaderCANbus();
        /* after loader disconnection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Connect the loader hardware.")
    public void connectLoaderCANbus() {
        loader.connectLoaderCANbus();
        /* after loader connection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * Check if a filter name given by an end user at the console is a valid Filter
     * Name. Delegate to filterManager.
     *
     * @param aName
     */
    public void checkFilterName(String aName) {
        filterManager.checkFilterName(aName);
    }

    /**
     * For Whole FCS GUI. Has to be overridden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    /**
     * For whole FCS GUI.
     * Override listLoSensorsNames of MainModule
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of names of sensors plugged on Loader pluto gateway.")
    @Override
    public List<String> listLoSensorsNames() {
        return bridgeToLoader.listLoSensorsNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return carousel.isFilterOnCarousel(filterID) || autochanger.isFilterOnAC(filterID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter on a given socket of the carousel and move the empty autochanger to HANDOFF position.", autoAck = false, timeout = 60000)
    public void storeFilterOnSocket(String socketName) {

        try (AutoTimed at = new AutoTimed("storeFilterOnSocket")) {
            updateStateWithSensors();
            FCSLOG.info(name + " === About to store filter on carousel ===");

            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }

            CarouselSocket desiredSocket = carousel.getSocketByName(socketName);

            if (!autochanger.isAtStandby()) {
                // carousel must be empty if the autochanger has a filter
                if (autochanger.isHoldingFilter() && !desiredSocket.isEmpty()) {
                    throw new RejectedCommandException(name + " the target carousel socket is not empty");
                }
            }

            // Make sure the autochanger is free to move
            if (autochanger.isAtOnline()) {
                AutochangerThreeOnlineClamps onlineClamps = autochanger.getOnlineClamps();
                if (onlineClamps.isLocked()) {
                    onlineClamps.unlockClamps();
                }
                // TODO throw an Exception if not Closed ?
                if (onlineClamps.isClosed()) {
                    if (onlineClamps.isHomingDone()) {
                        onlineClamps.openClamps();
                    } else {
                        onlineClamps.homing();
                    }
                }
                updateStateWithSensors();

                if (onlineClamps.isOpened()) {
                    FCSLOG.info(name + " === autochanger is free to move ===");
                } else {
                    throw new FcsHardwareException(name + " autochanger online clamps should be opened");
                }
            }

            carousel.rotateSocketToStandby(socketName);
            carousel.checkDeltaPosition();
            FCSLOG.info(name + " === carousel is ready to receive a filter at standby ===");

            if (!autochanger.isEmpty()) {
                autochanger.moveFilterToStandbyPlusDelta(carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
                /* because both subsystems sensors have changed we have to update again */
                updateStateWithSensors();
            }

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on autochanger = " + autochanger.getFilterID());
                FCSLOG.info("==================================");
                carousel.persistData();

            } else {
                recoveryLockingProcess();
                if (carousel.isHoldingFilterAtStandby()) {
                    carousel.getSocketAtStandby().updateFilterID();
                    carousel.persistData();
                } else {
                    String msg = name + ": carousel should be CLAMPED_ON_FILTER when autochanger is "
                            + "at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlert.CA_LOCKING_ISSUE, msg, name);
                    throw new FcsHardwareException(msg);
                }
            }
            if (autochanger.isAtStandby()) {
                // wait update local protection system
                autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff();
            } else {
                autochanger.goToHandOff();
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter in the carousel and move the empty autochanger to HANDOFF position."
            + " Initial State for autochanger: a filter at HANDOFF or ONLINE. Final State for autochanger : empty at HANDOFF.", autoAck = false, timeout = 60000)
    public void storeFilterOnCarousel() {
        try (AutoTimed at = new AutoTimed("storeFilterOnCarousel")) {
            updateStateWithSensors();
            FCSLOG.info(name + " === About to store filter on carousel ===");

            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }

            // carousel must be empty and available at STANDBY
            if (!carousel.isReadyToGrabAFilterAtStandby()) {
                throw new RejectedCommandException(name + " carousel should be at standby with an empty and available socket");
            }
            // autochanger should be at HANDOFF or ONLINE
            if (!(autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                throw new RejectedCommandException(name + " autochanger must be at HANDOFF or at ONLINE");
            }

            if (!autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(name + " autochanger must hold the filter");
            }

            // Make sure the carousel is aligned
            // change made on July 18 to clean the sequence of operations Carousel first then Autochanger
            carousel.checkDeltaPosition();

            // Make sure the autochanger is free to move
            if (autochanger.isAtOnline()) {
                if (autochanger.getOnlineClamps().isLocked()) {
                    autochanger.getOnlineClamps().unlockClamps();
                }
                // TODO throw an Exception if not Closed ?
                if (autochanger.getOnlineClamps().isClosed()) {
                    if (autochanger.getOnlineClamps().isHomingDone()) {
                        autochanger.getOnlineClamps().openClamps();
                    } else {
                        autochanger.getOnlineClamps().homing();
                    }
                }
            }
            updateStateWithSensors();

            if (autochanger.getOnlineClamps().isOpened()) {
                FCSLOG.info(name + " === autochanger is free to move ===");
            } else {
                throw new FcsHardwareException(name + " autochanger online clamps should be opened");
            }
            FCSLOG.info(name + " === carousel is ready to receive a filter at standby ===");
            autochanger.moveFilterToStandbyPlusDelta(carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
            /* because both subsystems sensors have changed we have to update again */
            updateStateWithSensors();

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on autochanger = " + autochanger.getFilterID());
                FCSLOG.info("==================================");
                carousel.persistData();

            } else {
                recoveryLockingProcess();
                if (carousel.isHoldingFilterAtStandby()) {
                    carousel.getSocketAtStandby().updateFilterID();
                    carousel.persistData();
                } else {
                    String msg = name + ": carousel should be CLAMPED_ON_FILTER when autochanger is "
                            + "at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlert.CA_LOCKING_ISSUE, msg, name);
                    throw new FcsHardwareException(msg);
                }
            }
            if (autochanger.isAtStandby()) {
                // wait update local protection system
                autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff();
            } else {
                throw new FcsHardwareException(
                        name + ": autochanger should be at STANDBY after " + "moveFilterToStandby() command");
            }
        }
    }

    private void recoveryLockingProcess() {
        try (AutoTimed at = new AutoTimed("recoveryLockingProcess")) {
            if (carousel.getClampXplus().isLocked() && !carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, name);
                carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false, description = "Unclamp filter from carousel, move autochanger to approachStandby position and release the carousel clamps", timeout = 15000)
    public void disengageFilterFromCarousel() {
        try (AutoTimed at = new AutoTimed("disengageFilterFromCarousel")) {
            updateStateWithSensors();
            if (!FilterReadinessState.READY.equals(getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }
            if (!carousel.isAtStandby()) {
                throw new RejectedCommandException(name + "Carousel should be at Standby");
            }
            if (!(autochanger.isAtStandby())) {
                throw new RejectedCommandException("autochanger should be at Standby Position");
            }
            // Unclamp the filter from the carousel
            carousel.unlockClamps();
            carousel.updateStateWithSensors();
            carousel.waitForProtectionSystemUpdate();

            if (!carousel.isUnclampedOnFilterAtStandby()) {
                String msg = name + "Carousel clamps still locked. Aborting autochanger movement because "
                        + "carousel is still holding the filter. bad state for socketAtStandby: "
                        + carousel.getClampsStateAtStandby() + " should be UNCLAMPED_ON_FILTER";
                raiseAlarm(FcsAlert.HARDWARE_ERROR, msg, name);
                throw new FcsHardwareException(name + msg);
            }
            autochanger.waitForProtectionSystemUpdate();
            // autochanger begins to leave standby position : Move autochanger to approach standby position
            autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();

            updateStateWithSensors();
            if (!carousel.isEmptyAtStandby()) {
                raiseAlarm(FcsAlert.HARDWARE_ERROR, "Carousel is still seeing the filter", name);
                throw new FcsHardwareException(
                        name + " aborting autochanger movement because carousel is still seeing filter.");
            }

            // Release the clamps
            FcsUtils.asyncRun(() -> {
                try {
                    carousel.releaseClamps();
                } catch (Exception e) {
                    this.raiseWarning(FcsAlert.HARDWARE_ERROR, "cannot release carousel clamps", "carousel");
                }
            });
            updateStateWithSensors();
        }
    }

    // SET FILTER PRECONDITIONS
    public boolean autochangerNotInTravel() {
        return autochanger.isAtHandoff() || autochanger.isAtOnline() || autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return autochanger.getLatches().isClosed() || autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return autochanger.getOnlineClamps().isLocked() || autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !autochanger.isAtOnline() || autochanger.isEmpty() || autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !autochanger.isAtStandby() || autochanger.isEmpty() || autochanger.isHoldingFilter()
                || carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return carousel.isHoldingFilterAtStandby() || carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return autochanger.isAtStandby() || autochanger.isEmpty() || carousel.isReadyToGrabAFilterAtStandby();
    }

    /**
     * move filter to online. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on
     * autochanger at HANDOFF or ONLINE => carousel socket at STANDBY must be
     * READY_TO_CLAMP && autochanger latches have to be CLOSED, if a filter is
     * on autochanger at STANDBY => carousel or autochanger must hold the
     * filter, if a filter is on autochanger at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilter(int filterID) {
        setFilterAtHandoffOrOnline(filterID, true, false);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilterByName(String filterName) {
        subs.helper()
                .precondition(filterManager.containsFilterName(filterName), "%s: Unknown filter name : %s", name, filterName)
                .precondition(FilterReadinessState.READY).action(() -> {
                        /* filterName is correct*/
            setFilter(filterManager.getFilterID(filterName));
//Just for MCM tests on PROTO.
//            setFilterAtHandoffOrOnline(filterManager.getFilterID(filterName), false);
                });
    }

    /**
     * move filter to HANDOFF. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on
     * autochanger at HANDOFF or ONLINE => carousel socket at STANDBY must be
     * READY_TO_CLAMP && autochanger latches have to be CLOSED, if a filter is
     * on autochanger at STANDBY => carousel or autochanger must hold the
     * filter, if a filter is on autochanger at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoff(int filterID) {
        setFilterAtHandoffOrOnline(filterID, false, false);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoffForLoader(int filterID) {
        setFilterAtHandoffOrOnline(filterID, false, true);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to ONLINE.", timeout = 180000, autoAck = false)
    public void setNoFilter() {
        setNoFilterAtHandoffOrOnline(true);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to HANDOFF.", timeout = 180000, autoAck = false)
    public void setNoFilterAtHandoff() {
        setNoFilterAtHandoffOrOnline(false);
    }

    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline, boolean forLoader) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter")) {

            updateStateWithSensors();
            publishDataForMcm();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            FCSLOG.info(name + ": filter to set online:" + filterID);
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    // CHECK it should be valid now
                    // loaderPresenceSensor is invalid
                    // .precondition(!isLoaderConnected(), "loader is connected - can't continue
                    // setFilter")
                    .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name,
                            filterID)
                    .precondition(isFilterInCamera(filterID), "%s: filter %s / filter name %s is out of camera", name, filterID, filterManager.getFilterNameByID(filterID))
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by AC or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(SETFILTER_MAX_DURATION)
                    .enterFaultOnException(true)
                    .action(() -> {
                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                            //LOADED
                            agentStateService.updateAgentState(McmState.LOADED);
                            FCSLOG.info(name + ": AC is holding a filter, filter name= "
                                    + autochanger.getFilterOnTrucksName());
                        } else {
                            agentStateService.updateAgentState(McmState.NO_FILTER);
                            FCSLOG.info(name + ": AC is not holding filter, filter name on AC"
                                    + autochanger.getFilterOnTrucksName());
                            FCSLOG.info(name + ": filterID: " + filterID + " is on socket:"
                                    + carousel.getFilterSocket(filterID).getId());
                        }
                        /**
                         * /* If the filter we want to move ONLINE is NOT on AC : /* - if AC is at
                         * STANDBY it has to be moved empty to HANDOFF /* - if AC is at HANDOFF or
                         * ONLINE, and another filter is on AC this filter has to be store on carousel,
                         * /* then AC can be moved empty at HANDOFF.
                         */
                        carousel.setControllerPositionSensorTypeEncoderSSI();
                        if (!autochanger.isFilterOnAC(filterID)) {
                            //UNLOADING
                            agentStateService.updateAgentState(McmState.UNLOADING);
                            FCSLOG.info(name + ": filterID: " + filterID + " is NOT on AC");
                            if (autochanger.isAtStandby()) {
                                FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                                autochanger.moveEmptyFromStandbyToHandoff();
                            } else {
                                /* autochanger can be at HANDOFF or ONLINE with a filter or empty */
                                /* if a filter is on AC */
                                if (autochanger.isHoldingFilter()) {
                                    FCSLOG.info(
                                            name + ": AC is holding filter: " + autochanger.getFilterOnTrucksName());
                                    FCSLOG.info(name + ": is going to store a filter on carousel");
                                    storeFilterOnCarousel();
                                }
                            }

                            /* Now autochanger is empty at Handoff or at ONLINE*/
                            if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                                throw new FcsHardwareException(name + ": autochanger is not empty at handoff or online");
                            }
                            // Rotate desired filter to standby
                            //ROTATING
                            agentStateService.updateAgentState(McmState.ROTATING);
                            carousel.rotateSocketToStandby(carousel.getFilterSocket(filterID).getName());
                            FcsUtils.sleep(100, name);
                            updateStateWithSensors();
                            if (carousel.isAtStandby()) {
                                FCSLOG.info(name + ": carousel is at standby");
                            } else {
                                throw new FcsHardwareException(name + ": carousel should be at standby after "
                                        + "rotateSocketToStandby command.");
                            }

                            /* At this point filterID should be on carousel at STANDBY */
                            if (carousel.getFilterIDatStandby() == filterID) {
                                // Carousel could have continued rotation due to unbalanced state
                                carousel.checkDeltaPosition();
                                // Go and grab filter on AC
                                // wait for autochanger PLC to update after carousel rotation
                                // otherwise trucks controller is in UNDER_VOLTAGE fault.
                                autochanger.waitForProtectionSystemUpdate();
                                //LOADING
                                agentStateService.updateAgentState(McmState.LOADING);
                                autochanger.grabFilterAtStandby();

                            } else {
                                throw new FcsHardwareException(name + " filterID at standby is "
                                        + carousel.getFilterIDatStandby() + " should be: " + filterID);
                            }
                            updateStateWithSensors();
                            if (!autochanger.isFilterOnAC(filterID)) {
                                throw new FcsHardwareException(name + " filter: " + filterID + " should be now on AC");
                            }
                        }
                        /* now filterID is on AC. AC is at STANDBY or at HANDOFF or ONLINE. */
                        if (autochanger.isAtStandby()) {
                            try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {

                                // Unclamp filter from carousel and move to approach position
                                disengageFilterFromCarousel();
                                updateStateWithSensors();

                                if (toOnline) {
                                    // Move the filter to online position
                                    autochanger.moveAndClampFilterOnline();
                                } else {
                                    // Move the filter to handoff position
                                    moveToHandoffWithHighVelocity(forLoader);
                                }
                            }
                        } else if (toOnline) {
                            if (autochanger.isAtHandoff()) {
                                autochanger.moveAndClampFilterOnline();

                            } else if (autochanger.isAtOnline()) {
                                autochanger.lockFilterAtOnline();
                            }
                        } else {
                            if (autochanger.isAtOnline() && autochanger.getOnlineClamps().isOpened()) {
                                try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {
                                    autochanger.getAutochangerTrucks().moveToApproachOnlinePositionWithLowVelocity();
                                    moveToHandoffWithHighVelocity(forLoader);
                                }
                            }
                        }
                String filterName = filterManager.getFilterNameByID(filterID);
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
                setFilterDuration = System.currentTimeMillis() - beginTime;
                FCSLOG.info("filter " + filterID + " is " + (toOnline ? "ONLINE" : "HANDOFF")
                        + ". setFilterDuration = " + setFilterDuration);
                this.publishData();
                subs.publishSubsystemDataOnStatusBus(
                        new KeyValueData("duration/SETFILTER", setFilterDuration));
                //LOADED
                agentStateService.updateAgentState(McmState.LOADED);
                publishDataForMcm();
                    }); // END of action
        }
    }

    private void moveToHandoffWithHighVelocity(boolean forLoader) {
        if (forLoader) {
            autochanger.getAutochangerTrucks().fastProfile();
            autochanger.getAutochangerTrucks().goToHandOffForLoader();
        } else {
            autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
        }
    }

    private void setNoFilterAtHandoffOrOnline(boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setNoFilter")) {
            updateStateWithSensors();
            publishDataForMcm();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by autochanger or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(SETFILTER_MAX_DURATION)
                    .enterFaultOnException(true)
                    .action(() -> {
                if (autochanger.isHoldingFilter()) {
                    FCSLOG.info(
                            name + ": autochanger is holding filter: " + autochanger.getFilterOnTrucksName());
                    FCSLOG.info(name + ": is going to store a filter on carousel");
                    //UNLOADING
                    agentStateService.updateAgentState(McmState.UNLOADING);
                    storeFilterOnCarousel();
                } else {
                    FCSLOG.info(name + " no filter to store on carousel");
                }
                /* Now autochanger is empty at Handoff or Online */
                if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                    throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                }
                if (toOnline) {
                    autochanger.goToOnline();
                }
                updateAgentState(FilterState.ONLINE_NONE);
                updateAgentState(McmState.NO_FILTER);
                publishDataForMcm();
                this.publishData();
            });
        }
    }

    /**
     * just for mcm testing
     *
     * @param state
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "change subsystem state.")
    public void changeState(String state) {
        updateAgentState(FilterState.valueOf(state));
        FCSLOG.fine("SUBSYSTEM STATE=" + isInState(FilterState.valueOf(state)));
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public Map<Integer, String> listFiltersOnChanger() {
        Map<Integer, String> lf = new HashMap();
        if (autochanger.getFilterID() != 0) {
            lf.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty()) {
                lf.put(socket.getFilterID(), socket.getFilterName());
            }
        });
        return lf;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public Map<Integer, String> listFiltersOnChanger2() {
        Map<Integer, String> lf = new HashMap();
        if (autochanger.getFilterID() != 0) {
            lf.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName() + ",autochanger");
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty()) {
                lf.put(socket.getFilterID(), socket.getFilterName() + "," + socket.getName());
            }
        });
        return lf;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list all filters names.")
    public List<String> listAllFilters() {
        return filterManager.getFilterNames();
    }

    public Map<String, Filter> getFiltersOnChanger() {
        Map<String, Filter> lf = new HashMap();
        if (autochanger.getFilterID() != 0) {
            lf.put("autochanger", filterManager.getFilterByID(autochanger.getFilterID()));
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0) {
                lf.put(socket.getName(), filterManager.getFilterByID(socket.getFilterID()));
            }
        });
        return lf;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public String printFilterOnChangerList() {
        StringBuilder sb = new StringBuilder("Filters on FilterChanger:\n");
        getFiltersOnChanger().values().stream().forEach((f) -> {
            sb.append("filterID:");
            sb.append(f.getFilterID());
            sb.append("/filterName:");
            sb.append(f.getName());
            sb.append("/filterFamily:");
            sb.append(f.getFamily());
            sb.append("=");
            sb.append(f.getFamily().getFamilyName());
            sb.append("\n");
        });
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public String printFilterLocalisation() {
        StringBuilder sb = new StringBuilder("Filters on FilterChanger: \n");
        Iterator iter = getFiltersOnChanger().entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry entree = (Map.Entry<String, Filter>) iter.next();
            Filter f = (Filter) entree.getValue();
            sb.append("filterID:");
            sb.append(f.getFilterID());
            sb.append("/filterName:");
            sb.append(f.getName());
            sb.append("/filterFamily:");
            sb.append(f.getFamily());
            sb.append(" is on: ");
            sb.append(entree.getKey());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Load a filter from the loader to the camera. The loader must hold a filter at
     * STORAGE position. The autochanger must be empty at HANDOFF, latches open. At
     * the end of this command the filter is held by autochanger at Handoff
     * position, and the loader carrier is empty at STORAGE.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.",
            timeout = 480000, autoAck = false)
    public void loadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilter")) {
            //
        // TODO check that the camera is in the position "horizontal".
        // to test if carousel is empty at standby is over engineering : it has to be
        // tested by autochanger before command StoreFilterOnCarousel
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // autochanger trucks must be empty at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isEmpty(),
                        " autochanger is not empty at HANDOFF position; can't load a filter.")
                // autochanger latches must be open.
                .precondition(autochanger.getLatches().isOpened(),
                        "Autochanger latches must be open before loadFilter command.")
                // The Loader carrier is retracted at "Storage" position and the filter is clamped
                .precondition(loader.isClampedOnFilter() && loader.isAtStorage(),
                        name + " can't load filter because loader is not holding a filter at storage position.")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {

            if (!autochanger.isPositionCorrectForExchangeWithLoader()) {
                autochanger.goToHandOffForLoader();
            }

            loader.moveFilterToHandoff();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            autochanger.closeLatches();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            loader.openClampAndMoveEmptyToStorage();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

                    }); // END of action
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera"
            + "and store it on socket which name is given as argument. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is store on carousel socket which name is given as argument, autochanger is empty at Handoff position, "
            + "and the loader carrier is empty at STORAGE.")
    public void loadFilterOnSocket(String socketName) {
        updateStateWithSensors();
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socket = carousel.getSocketsMap().get(socketName);
        if (!socket.isEmpty()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not empty. "
                    + "Select an empty socket.", socketName));
        }
        if (!socket.isAvailable()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not available. "
                    + "Select an available socket.", socketName));
        }
        loadFilter();
        carousel.rotateSocketToStandby(socketName);
        storeFilterOnCarousel();
    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Autochanger at HANDOFF. Loader
     * carrier must be empty at STORAGE. At the end of this command, autochanger is
     * empty at HANDOFF, latches are open, and loader carrier is holding the filter
     * at STORAGE (hooks are clamped).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unload a filter from the camera to the loader. The camera has to be at horizontal position. "
            + "The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. "
            + "\nAt the end of this command, autochanger is empty at HANDOFF, latches are open, "
            + "and loader carrier is holding the filter at STORAGE (hooks are clamped)",
            timeout = 480000, autoAck = false)
    public void unloadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilter")) {
            // TODO check that the camera is in the position "horizontal"
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // the auto changer trucks must hold a filter at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isHoldingFilter(),
                        " autochanger is not holding a filter at HANDOFF position; can't unload a filter.")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {


            if (!autochanger.isPositionCorrectForExchangeWithLoader()) {
                if (autochanger.isLinearRailMotionAllowed()) {
                    autochanger.goToHandOffForLoader();
                } else {
                    throw new RejectedCommandException("Autochanger trucks position is not correct for exchange with loader "
                            + " but command goToHandOffForLoader can't be executed because PLC doesn't allow to move trucks.");
                }
            }

            loader.moveEmptyToHandoffAndClose();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            autochanger.openLatches();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            loader.moveFilterToStorage();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

                    }); // END of action
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "check if controllers are in fault")
    @Override
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer :
     * carousel, autochanger and loader. Read also autochanger trucks and ONLINE
     * clamps position.
     *
     * @throws FcsHardwareException
     */
    @Override
    public synchronized void updateStateWithSensors() {
        // do not execute if last finished less than 100 ms ago.
        // we keep it at the end of actions but don't repeat it
        // synchronized : parallel operations should do it just once and n-1 will just
        // wait.
        if (System.currentTimeMillis() - lastUpdateStateWithSensors.get() < 100 && !FcsUtils.isSimu()) {
            FCSLOG.info(name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
            return;
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs")) {
            // Carousel
            carousel.updateStateWithSensors();
            if (carousel.isAtStandby()) {
                carousel.getSocketAtStandby().updateFilterID();
            }
            // Autochanger
            autochanger.updateStateWithSensors();
            if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.isHoldingFilter()
                    && autochanger.getOnlineClamps().isLocked()) {
                String filterName = filterManager.getFilterNameByID(autochanger.getFilterID());
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
            } else {
                updateAgentState(FilterState.valueOf("ONLINE_NONE"));
            }
            // update loader sensors is loader is connected on camera
            // modified for the May 2023 tests
            // TODO: make sure enforcing the loader presence on the AC is actually needed for updating the sensors
            if (/*autochanger.isLoaderConnected() && */loader.isCANbusConnected()) {
                loader.updateStateWithSensors();
            }
        }
    }

    AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
        autochanger.initializeHardware();
        loader.initializeHardware();
        postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        bridgeToLoader.publishData();
    }

    public void publishDataForMcm() {

        updateAgentState(autochanger.getAutochangerInclinationState());
        subs.publishSubsystemDataOnStatusBus(getDataForMcm());
    }

    private KeyValueData getDataForMcm() {
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData("filter_on_autochanger_id", autochanger.getFilterID());
        kvdl.addData("filter_on_autochanger_name", autochanger.getFilterOnTrucksName());
        kvdl.addData("autochanger_trucks_position", autochanger.getAutochangerTrucks().getPosition());
        kvdl.addData("autochanger_trucks_location", autochanger.getAutochangerTrucksLocation());
        return new KeyValueData(PUBLICATION_KEY_FOR_MCM, kvdl);
    }

    @Override
    public void postStart() {
        FCSLOG.info("Version du 18 juillet 2023 10:05");
        // TODO if we initializeClampsState here => httc580 is not booted, so fcs
        // doesn't start
        // check why
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(name + " is shutting down.");
        bridge.doShutdown();
        if (loader.isCANbusConnected()) {
            bridgeToLoader.doShutdown();
        }
        FCSLOG.info(name + " is shutdown.");
    }

    @Override
    public String vetoTransitionToNormalMode() {
        if (this.isChangerReady()) {
            publishDataForMcm();
            if (autochanger.isHoldingFilter() && autochanger.isAtOnline()) {
                updateAgentState(McmState.LOADED);
            } else {
                updateAgentState(McmState.NO_FILTER);
            }
            return super.vetoTransitionToNormalMode();
        } else {
            return "FilterChanger not ready";
        }
    }
}
