package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.ANCESTORS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileVelocity;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_TRUCKS_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_TO_ABSOLUTE_POSITION;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ALIGN_FOLLOWER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_FOLLOWER_TRUCK_ALONE;

/**
 * This is a model for autochanger trucks. Autochanger trucks move a filter from
 * STANDBY position to ONLINE position or HANDOFF position. ONLINE position is
 * just in front of the focal front. STANDBY position is where an exchange can
 * be done with the carousel. HANDOFF position is a position an exchange can be
 * done with the loader. The trucks are moved by 2 controllers, a driver and a
 * follower. Each controller has an absolute encoder.
 *
 * @author virieux
 */
public class AutochangerTwoTrucks extends MobileItem {

    @LookupField(strategy = ANCESTORS)
    private Subsystem subs;

    private final AutochangerTruck truckXminus;
    private final AutochangerTruck truckXplus;

    @ConfigurationParameter(description = "Which of the two trucks is master. Should be 'Xminus' if acTruckXminus is master, "
            + "or 'Xplus' if acTruckXplus is master. Used in the build method", category = "autochanger")
    private volatile String driverSide = "Xminus";

    @ConfigurationParameter(description = "minimal position for AC trucks", units = "micron", category = "autochanger")
    private volatile int minActualPositionValue = 0;

    @ConfigurationParameter(description = "maximal position for AC trucks", units = "micron", category = "autochanger")
    private volatile int maxActualPositionValue = 1007000;

    @ConfigurationParameter(description = "STANDBY position to be used to store a filter on carousel", units = "micron", category = "autochanger")
    private volatile int standbyPosition = 997500;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    /**
     * When going without a filter to STANDBY position to take a filter, AC trucks
     * must have less force (so go less far) than when going with a filter.
     * Otherwise, it makes it hard to unlock the filter from the carousel.
     */
    @ConfigurationParameter(description = "STANDBY position to be used when grabing a filter on Carousel", units = "micron", category = "autochanger")
    private volatile int standbyPositionEmpty = 997500;

    @ConfigurationParameter(description = "HANDOFF position", units = "micron", category = "autochanger")
    private volatile int handoffPosition = 70000;

    @ConfigurationParameter(description = "ONLINE position", units = "micron", category = "autochanger")
    private volatile int onlinePosition = 2600;

    @ConfigurationParameter(description = "time for trucks motion from ONLINE to STANDBY", units = "millisecond", category = "autochanger")
    private volatile long timeoutForTrucksMotion = 20000;

    @ConfigurationParameter(description = "time to align follower on driver controller", units = "millisecond", category = "autochanger")
    private volatile long timeoutForAlignFollower = 5000;
    /**
     * ****************** parameters to move filter to online position and adjust
     * position (docking process)
     *******
     */
    @ConfigurationParameter(description = "prelimary ONLINE position, used by process which"
            + "moves a filter at ONLINE position.", units = "micron", category = "autochanger")
    private volatile int prelimaryTargetPosition = 3367;

    @ConfigurationParameter(description = "natural position at ONLINE position used by process which"
            + "moves a filter at ONLINE position.", units = "micron", category = "autochanger")
    private volatile int naturalPosition = 3215;

    @ConfigurationParameter(description = "lowest position to be used by command moveAndClampFilterOnline.", units = "micron", category = "autochanger")
    private volatile int minTargetPosition = 1800;

    @ConfigurationParameter(description = "minimum value of voltage read on proximitySensor.", units = "V", category = "autochanger")
    private volatile double umin = 2.0;

    @ConfigurationParameter(description = "maximum value of voltage read on proximitySensor.", units = "V", category = "autochanger")
    private volatile double umax = 4.6;

    @ConfigurationParameter(description = "voltage target on proximitySensor.", units = "V", category = "autochanger")
    private volatile double utarget = 2.5;

    @ConfigurationParameter(description = "adjustment factor.", units = "micron/V", category = "autochanger")
    private volatile double adjustmentFactor = 1200;

    /**
     * *********** end of parameters to move filter to online position and adjust
     * position (docking process)
     *******
     */
    /**
     * *********** configurable parameters for the approach to standby position
     ************************
     */
    @ConfigurationParameter(description = "low speed to approach STANDBY position", category = "autochanger")
    private volatile int lowSpeed = 50;

    @ConfigurationParameter(description = "high speed to approach STANDBY position", category = "autochanger")
    private volatile int highSpeed = 260;

    @ConfigurationParameter(description = "low acceleration to approach STANDBY position", category = "autochanger")
    private volatile int lowAcceleration = 150;

    @ConfigurationParameter(description = "high acceleration to approach STANDBY position", category = "autochanger")
    private volatile int highAcceleration = 310;

    @ConfigurationParameter(description = "low deceleration to approach STANDBY position", category = "autochanger")
    private volatile int lowDeceleration = 50;

    @ConfigurationParameter(description = "high deceleration to approach STANDBY position", category = "autochanger")
    private volatile int highDeceleration = 310;

    @ConfigurationParameter(description = "approach STANDBY position", units = "micron", category = "autochanger")
    private volatile int approachStandbyPosition = 950000;

    @ConfigurationParameter(description = "approach ONLINE position", units = "micron", category = "autochanger")
    private volatile int approachOnlinePosition = 10000;

    private int driverPosition;
    private int followerPosition;
    private int deltaPosition;

    /**
     * position is reached if |targetPosition - driverPosition| <
     * positionRange
     */
    private int positionRange = 100;

    /**
     * position is reached at standby if |targetPosition - driverPosition| <
     * positionRangeAtStandby
     */
    @ConfigurationParameter(description = "position is reached at STANDBY if |targetPosition - driverPosition| < positionRangeAtStandby", units = "micron", category = "autochanger")
    private volatile int positionRangeAtStandby = 100;

    /**
     * position is reached if |targetPosition - driverPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached at ONLINE if |targetPosition - driverPosition| < positionRangeAtOnline", units = "micron", category = "autochanger")
    private volatile int positionRangeAtOnline = 50;

    /**
     * position is reached if |targetPosition - driverPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached at HANDOFF if |targetPosition - driverPosition| < positionRangeAtOnHandoff", units = "micron", category = "autochanger")
    private volatile int positionRangeAtHandoff = 100;

    /**
     * position is reached if |targetPosition - driverPosition| <
     * positionRange
     */
//    @ConfigurationParameter(description = "position is reached at HANDOFF if |targetPosition - driverPosition| < positionRangeAtHandoffForLoader", units = "micron", category = "autochanger")
    private volatile int positionRangeAtHandoffForLoader = 50;

    /**
     * position is reached if |targetPosition - driverPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached in TRAVEL if |targetPosition - driverPosition| < positionRangeAtOnline", units = "micron", category = "autochanger")
    private volatile int positionRangeInTravel = 150;

    @ConfigurationParameter(description = "Value in alignFollower when we want a strict alignment : alignment of follower is done when |deltaPosition| > deltaPositionForAlignStrict.", units = "micron", category = "autochanger")
    private volatile int deltaPositionForAlignStrict = 200;

    @ConfigurationParameter(description = "Value in alignFollower when we want a lenient alignment : alignment of follower is done when |deltaPosition| > deltaPositionForAlignLenient.", units = "micron", category = "autochanger")
    private volatile int deltaPositionForAlignLenient = 400;

    @ConfigurationParameter(description = "Value in alignFollower when we want a strict alignment for interactions with Loader : alignment of follower is done when |deltaPosition| > deltaPositionForAlignStrictForLoader.", units = "micron", category = "autochanger")
    private volatile int deltaPositionForAlignStrictLoader = 50;

    /**
     * proximity to preliminaryTargetPosition to allow to find the true ONLINE position
     */
    @ConfigurationParameter(description = "proximity to preliminaryTargetPosition to allow to find the true ONLINE position", units = "micron", category = "autochanger")
    private volatile int deltaPreliminaryTargetPosition = 200;

    private int absoluteTargetPosition = 0;
    private int followerTargetPosition = 0;

    private final EPOSControllerForLinearRail acTruckXminusController;
    private final EPOSControllerForLinearRail acTruckXplusController;

    private EPOSControllerForLinearRail linearRailDriverController;
    private EPOSControllerForLinearRail linearRailFollowerController;

    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = TREE)
    private Autochanger autochanger;

    @LookupField(strategy = TREE, pathFilter = ".*\\/proximitySensorsDevice")
    private ADCInterface proximitySensorsDevice;

    @ConfigurationParameter(description = "input number on the ADC where proximity sensor is plugged", range = "1..4", units = "unitless", category = "autochanger")
    private volatile int proximitySensorInput = 1;

    /* field to store voltage value read on proximity sensor device*/
    private double proximityVoltage = 0;

    /**
     * true if Handoff position sensors for TruckXminus and TruckXplus return value
     * 1
     */
    private boolean atHandoff = false;

    /**
     * true if Online position sensors for TruckXminus and TruckXplus return value 1
     */
    private boolean atOnline = false;

    /**
     * true if Standby position sensors for TruckXminus and TruckXplus return value
     * 1
     */
    private boolean atStandby = false;

    /**
     * Build a new AutoChangerTrucksModule.
     *
     * @param truckXminus
     * @param truckXplus
     * @param acTruckXminusController
     * @param acTruckXplusController
     */
    public AutochangerTwoTrucks(AutochangerTruck truckXminus, AutochangerTruck truckXplus,
            EPOSControllerForLinearRail acTruckXminusController, EPOSControllerForLinearRail acTruckXplusController) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.acTruckXminusController = acTruckXminusController;
        this.acTruckXplusController = acTruckXplusController;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutoChangerTrucks.class, path);
        for (MobileItemAction action : new MobileItemAction[]{
            ALIGN_FOLLOWER,
            MOVE_FOLLOWER_TRUCK_ALONE,
            MOVE_TO_ABSOLUTE_POSITION}) {
            registerAction(action);
        }
        /**
         * This initialization has to be make in build in order to be able to send
         * command to linearRailDriverController and to linearRailFollowerController in
         * postStart methods.
         */
        if ("Xminus".equals(driverSide)) {
            linearRailDriverController = acTruckXminusController;
            linearRailFollowerController = acTruckXplusController;
        } else {
            linearRailFollowerController = acTruckXminusController;
            linearRailDriverController = acTruckXplusController;
        }
    }

    @Override
    public void init() {
	super.init();

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(SDO_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(AC_TRUCKS_ERROR.getAlert(), alwaysClear);
    }

    @ConfigurationParameterChanger
    public void setNaturalPosition(int pos) {
        int minLimit = onlinePosition - 1500;
        int maxLimit = onlinePosition + 1500;
        if (pos < minLimit || pos > maxLimit) {
            throw new IllegalArgumentException(
                    pos + " bad value for natural position" + " must be > " + minLimit + " and < " + maxLimit);
        }
        this.naturalPosition = pos;
    }

    /**
     * for simu and tests
     *
     * @return
     */
    public String getDriverSide() {
        return driverSide;
    }

    /**
     * Return actual trucks driverPosition. Doesn't read again the driverPosition on
     * the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return actual trucks position (master position). Doesn't read again the position on the controller.")
    public int getPosition() {
        return driverPosition;
    }

    /**
     * Return HANDOFF position. Doesn't read again the position on the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * Return STANDBY position. Doesn't read again the position on the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return ONLINE position. Doesn't read again the position on the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return onlinePosition;
    }

    public int getApproachStandbyPosition() {
        return approachStandbyPosition;
    }

    /**
     * for tests
     *
     * @return
     */
    public int getApproachOnlinePosition() {
        return approachOnlinePosition;
    }

    /**
     * Return the timeout for trucks motion. If trucks motion duration is greater
     * than timeoutForTrucksMotion, an ALERT is raised.
     *
     * @return timeoutForTrucksMotion
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return timeout for trucks motion.")
    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it returns
     * its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoff() {
        return atHandoff;
    }

    public boolean isPositionCorrectForExchangeWithLoader() {
        return Math.abs(this.driverPosition - handoffPosition) < positionRangeAtHandoffForLoader;
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being updated
     * and waits for a response from a sensor, this methods waits until update is
     * completed. If the field atOnline is not being updated, it returns its
     * immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return atOnline;
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it returns
     * its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandby() {
        return atStandby;
    }

    /**
     * Return true if the homing of the trucks has been done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return linearRailFollowerController.isHomingDone() && acTruckXminusController.isHomingDone();
    }

    /**
     * Return true if TruckXminus position sensors or TruckXminus position sensors
     * are in error. Or position sensors are non consistant.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if TruckXminus position sensors or TruckXminus position sensors are in error.")
    public boolean isPositionSensorsInError() {
        return truckXminus.isPositionSensorsInError() || truckXplus.isPositionSensorsInError()
                || positionSensorsNonConsistant();
    }

    private boolean positionSensorsNonConsistant() {
        return ((isAtStandby() && isAtHandoff())
                || ((isAtStandby() && isAtOnline()) || ((isAtHandoff() && isAtOnline()))));
    }

    /**
     * At the end of a motion during tests, a bad positionning of the trucks in a
     * way that both sensors send the same value can be detected as an error. In
     * this case we don't raise an ALARM but a WARNING.
     *
     * @return true if position sensors error is transient
     *
     */
    public boolean isPositionSensorErrorsTransient() {
        int delta = 5000;
        boolean errorHandoff = truckXminus.isHandoffSensorsInError() || truckXplus.isHandoffSensorsInError();
        boolean errorOnline = truckXminus.isOnlineSensorsInError() || truckXplus.isOnlineSensorsInError();
        boolean errorStandby = truckXminus.isStandbySensorsInError() || truckXplus.isStandbySensorsInError();
        return ((errorHandoff && Math.abs(driverPosition - handoffPosition) < delta)
                || (errorOnline && Math.abs(driverPosition - onlinePosition) < delta)
                || (errorStandby && Math.abs(driverPosition - standbyPosition) < delta));
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (acTruckXplusController.isBooted() && acTruckXminusController.isBooted()) {
            initializeControllers();
            acTruckXplusController.updateEposState();
            acTruckXminusController.updateEposState();
            /**
             * because at startup of subsystem, brakes are activated by command
             * doDisableOperation in CanOpenProxy.connectToCANbus(), boolean
             * brakeActivated has not been initialized.
             */
            if (acTruckXminusController.isBrakeActivated()) {
                acTruckXminusController.setBrakeActivatedPub(true);
            } else {
                acTruckXminusController.setBrakeActivatedPub(false);
            }
            if (acTruckXplusController.isBrakeActivated()) {
                acTruckXplusController.setBrakeActivatedPub(true);
            } else {
                acTruckXplusController.setBrakeActivatedPub(false);
            }
            acTruckXminusController.readProfileVelocity();
            acTruckXminusController.readProfileAcceleration();
            acTruckXminusController.readProfileDeceleration();

            acTruckXminusController.publishData();
            acTruckXplusController.publishData();
        }

        if (proximitySensorsDevice.isBooted()) {
            proximitySensorsDevice.initializeAndCheckHardware();
            proximitySensorsDevice.publishData();
        } else {
            this.raiseAlarm(HARDWARE_ERROR, " proximitySensorsDevice not booted: it will be impossible to clamp at ONLINE.", name);
        }
        FCSLOG.fine(name + " END postStart.");
    }

    public void initializeControllers() {
        try {
            acTruckXplusController.initializeAndCheckHardware();
            acTruckXminusController.initializeAndCheckHardware();
            // configureControllers();
            homing();
            FCSLOG.info(name + " homing of AC trucks done.");

        } catch (FailedCommandException | FcsHardwareException ex) {
            // TODO voir quel controleur n'a pas pu etre initialisé.
            this.raiseAlarm(HARDWARE_ERROR, "Couldn't initialize controllers or do homing", name, ex);
        }
    }

    /**
     * Configure Autochanger trucks driver and follower controllers.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Configure Autochanger trucks driver and follower controllers.", alias = "initControllers")
    public void configureControllers() {
        FCSLOG.info(name + " Begin configuration of the controllers");
        acTruckXminusController.configureDigitalInputOfLinearRails();
        acTruckXplusController.configureDigitalInputOfLinearRails();
        acTruckXminusController.configureDigitalOutputOfLinearRails();
        acTruckXplusController.configureDigitalOutputOfLinearRails();
        FCSLOG.info(name + " End configuration of the controllers");
    }

    /**
     * Return true if both controllers are initialized and homing of trucks are
     * done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return acTruckXminusController.isInitialized() && acTruckXplusController.isInitialized() && this.isHomingDone();
    }

    /**
     * This method sends Exceptions if the conditions for trucks motion are not
     * filled. Autochanger trucks motion is allowed if : - controllers are
     * initialized - homing is done - trucks are empty OR (loaded with a filter but
     * latches are closed AND carousel or loader doesn't hold filter)
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion() {
        autochanger.checkFilterSafetyBeforeMotion();
        checkPrerequisitesForTrucksMotion();
    }

    private void checkPrerequisitesForTrucksMotion() {
        if (!acTruckXplusController.isInitialized() || !acTruckXminusController.isInitialized()) {
            String msg = name + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        updatePosition();
        // TODO do an alignFollower if allowed rather than fail
        checkDeltaPosition();
    }

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks are
     * already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    public void goToHandOff() {
        try (AutoTimed at = new AutoTimed("goToHandOff")) {
            moveToAbsoluteTargetPosition(this.handoffPosition);
        }
    }

    /**
     * Move Autochanger trucks to the Handoff position in order to load or unload a filter to or from the camera.
     * Autochanger trucks goes closer to handoff position than for exchange with carousel, in using as accuracy
     * configurable parameter positionRangeAtHandoffForLoader.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    public void goToHandOffForLoader() {
        if (Math.abs(driverPosition - handoffPosition) < positionRangeAtHandoffForLoader) {
            FCSLOG.info(name + " goToHandOffForLoader has nothing to do because position of trucks are already "
                    + "at a good position for loader. positionRangeAtHandoffForLoader = " + positionRangeAtHandoffForLoader);
            return;
        }
        try (AutoTimed at = new AutoTimed("goToHandOffForLoader")) {
            positionRange = positionRangeAtHandoffForLoader;
            absoluteTargetPosition = handoffPosition;
            FCSLOG.info(name + " accuracy of motion = " + positionRange + "microns");
            try (ActionGuard g = new ReleasedACBrakes()) {
                    checkConditionsForTrucksMotion();
                    checkControllersMode();
                    checkReadyForAction();
                    goToPosition(absoluteTargetPosition);
                    checkPositionSensors();
                    main.updateAgentState(FilterState.TRUCKS_STOPPED);
                publishData();
            }
        }
        alignFollowerForLoader();
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    public void goToOnline() {
        try (AutoTimed at = new AutoTimed("goToOnline")) {
            moveToAbsoluteTargetPosition(this.onlinePosition);
        }
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks are
     * already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    public void goToStandby() {
        try (AutoTimed at = new AutoTimed("goToOnline")) {
            moveToAbsoluteTargetPosition(this.standbyPosition);
        }
    }

    /**
     * Move the trucks to an absolute position given as argument. Doesn't align
     * follower at the end of motion.
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Move Autochanger trucks to the absolute position given as argument. "
            + "At the end of motion : doesn't align follower then activate brakes and disable controllers.", alias = "mobeABSPos", timeout = 20000)
    public void moveToAbsoluteTargetPosition(int targetPosition) {
        try (AutoTimed at = new AutoTimed("moveToAbsoluteTargetPosition")) {
            if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
                throw new IllegalArgumentException(name + "argument has to be between " + minActualPositionValue
                        + " and " + maxActualPositionValue);
            }

            this.updatePosition();
            if (driverPosition == targetPosition) {
                FCSLOG.info(name + " is already at target position=" + targetPosition);
            } else {
                try (ActionGuard g = new ReleasedACBrakes()) {
                    checkConditionsForTrucksMotion();
                    checkControllersMode();
                    checkReadyForAction();
                    absoluteTargetPosition = targetPosition;

                    FCSLOG.info(name + " going to absolute position: " + absoluteTargetPosition);
                    if (absoluteTargetPosition == handoffPosition) {
                        main.updateAgentState(FilterState.MOVING_TRUCKS_TO_HANDOFF);
                        positionRange = positionRangeAtHandoff;

                    } else if (absoluteTargetPosition == standbyPosition) {
                        main.updateAgentState(FilterState.MOVING_TRUCKS_TO_STANDBY);
                        positionRange = positionRangeAtStandby;

                    } else if (absoluteTargetPosition == onlinePosition) {
                        main.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE);
                        positionRange = positionRangeAtOnline;

                    } else if (absoluteTargetPosition == this.prelimaryTargetPosition) {
                        main.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE);
                        positionRange = deltaPreliminaryTargetPosition;

                    } else {
                        if (driverPosition > absoluteTargetPosition) {
                            main.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE);
                        } else {
                            main.updateAgentState(FilterState.MOVING_TRUCKS_TO_STANDBY);
                        }
                        positionRange = positionRangeInTravel;
                    }
                    FCSLOG.info(name + " accuracy of motion = " + positionRange + "microns");
                    goToPosition(absoluteTargetPosition);
                    checkPositionSensors();
                    main.updateAgentState(FilterState.TRUCKS_STOPPED);
                    publishData();
                }
            }
        }
    }


    public void moveAndClampFilterOnline() {
        try (AutoTimed at = new AutoTimed("moveAndClampFilterOnline")) {
            try (ActionGuard g = new ReleasedACBrakes()) {

                moveToApproachOnlinePositionWithHighVelocity();
                autochanger.waitForProtectionSystemUpdate();

                alignFollowerStrict();

                moveToTargetOnlinePosition();

            }
            /* close and lock clamps, do homing if not already done */
            autochanger.getOnlineClamps().lockFilterAtOnline();

            releaseTruckTensionOnClamps();

            autochanger.updateFCSStateToReady();

            this.publishData();

        }
    }

    /**
     * For end user at console for tests only. Not used in setFilter command.
     *
     *
     */
    public void unclampAndMoveFilterToHandoff() {
        /* unlock and open online clamps */
        if (!autochanger.isAtOnline()) {
            throw new RejectedCommandException(name + " autochanger has to be at ONLINE for this command.");
        }
        autochanger.getOnlineClamps().unlockAndOpen();
        if (autochanger.getOnlineClamps().isOpened()) {
            try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {
                moveToApproachOnlinePositionWithLowVelocity();
                moveToHandoffWithHighVelocity();
            }
        }
    }

    /**
     * Read the proximity sensors to adjust the new target online position once the
     * preliminaryTargetPosition has been reached
     * @return newTargetPosition = new target online position
     */
    // TODO: change this method to private ?
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_EXPERT, description = "Compute the actual online target position")
    public int findTargetOnlinePosition() {
        updatePosition();
        if (Math.abs(driverPosition - prelimaryTargetPosition) > deltaPreliminaryTargetPosition) {
            throw new RejectedCommandException(name + "Invalid position for this command. " +
                                               "Should be within a distance " + deltaPreliminaryTargetPosition +
                                               " of preliminaryTargetPosition = " + prelimaryTargetPosition +
                                               " to be executed.");
        }
        double umeasured = readProximitySensorDevice();

        if (umeasured > umax) {
            throw new FcsHardwareException(name + " voltage returned by proximity sensor is too high.");
        } else if (umeasured < umin) {
            throw new FcsHardwareException(name + " voltage returned by proximity sensor is too low.");
        }

        int newTargetPosition = (int) (prelimaryTargetPosition + adjustmentFactor * (this.utarget - umeasured));
        FCSLOG.info(name + " computed new target online position=" + newTargetPosition);

        newTargetPosition = Math.max(minTargetPosition, newTargetPosition);
        FCSLOG.info(name + " final new target online position=" + newTargetPosition);

        return newTargetPosition;
    }

    /**
     * Move a filter slowly to preliminaryTargetPosition, then computes the new
     * target position and finally reaches it.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Move a filter slowly to preliminaryTargetPosition, then computes the new "
            + "target position and finally reaches it", timeout = 5000)
    public void moveToTargetOnlinePosition() {
        try (AutoTimed at = new AutoTimed("moveToTargetOnlinePosition")) {
            try (ActionGuard g = new ReleasedACBrakes()) {
                slowProfile();
                moveToAbsoluteTargetPosition(prelimaryTargetPosition);

                int newTargetPosition = findTargetOnlinePosition();
                moveToAbsoluteTargetPosition(newTargetPosition);
            }
        }
    }

    /**
     * In the moveAndClampOnline process, release the tension on the clamps after
     * they are locked by moving the trucks to a move natural position than the
     * computed Online postion.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Release the tension on the clamps after they are locked by moving the trucks"
            + " to a move natural position than the computed Online postion.", timeout = 2000)
    public void releaseTruckTensionOnClamps() {
        double umeasured = readProximitySensorDevice();
        if (umeasured > umax) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too high.");
        } else if (umeasured < umin) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too low.");
        }
        goToPosition(naturalPosition);

        FCSLOG.info(name + " END process docking ONLINE.");
    }

    /**
     * Move a filter to standby position step by step. A first step with a high
     * speed to the approachStandbyPosition. a second step to standby position with
     * low speed.
     */
    public void moveFilterToStandby() {
        moveFilterToStandbyPlusDelta(0);
    }

    /**
     * Move a filter to standby position + deltaStandbyPosition step by step. A
     * first step with a high speed to the approachStandbyPosition. A second
     * step to standby position with low speed.
     *
     * @param deltaStandbyPosition
     */
    public void moveFilterToStandbyPlusDelta(int deltaStandbyPosition) {
        try (AutoTimed at = new AutoTimed("moveFilterToStandbyPlusDelta")) {
            updatePosition();
            autochanger.updateStateWithSensors();
            if (!(atHandoff || atOnline)) {
                throw new RejectedCommandException(name + "AC should be at Online or Handoff position");
            }
            main.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY);
            if (!atOnline) {
                alignFollowerLenient();
            }
            try (ActionGuard g = new ReleasedACBrakes()) {
                if (atOnline) {
                    moveToApproachOnlinePositionWithLowVelocity();
                    alignFollowerLenient();
                }
                moveToApproachStandbyPositionWithHighVelocity();
                alignFollowerStrict();
                moveToStandbyPlusDeltaWithLowVelocity(deltaStandbyPosition);
                FCSLOG.info("filter ID on AC after moveToStandbyWithLowVelocity = " + autochanger.getFilterID());
                publishData();
            }
        }
    }

    public void moveToApproachStandbyPositionWithLowVelocity() {
        try (AutoTimed at = new AutoTimed("moveToApproachStandbyPositionWithLowVelocity")) {
            slowProfile();
            moveToAbsoluteTargetPosition(approachStandbyPosition);
        }
    }

    public void moveToApproachOnlinePositionWithLowVelocity() {
        try (AutoTimed at = new AutoTimed("moveToApproachOnlinePositionWithLowVelocity")) {
            slowProfile();
            moveToAbsoluteTargetPosition(approachOnlinePosition);
        }
    }

    public void alignFollowerAndMoveEmptyFromApproachToHandoff() {
        try (AutoTimed at = new AutoTimed("alignFollowerAndMoveEmptyFromApproachToHandoff")) {
            alignFollowerStrict();
            fastProfile();
            moveToAbsoluteTargetPosition(handoffPosition);
        }
    }

    public void moveToApproachStandbyPositionWithHighVelocity() {
        try (AutoTimed at = new AutoTimed("moveToApproachStandbyPositionWithHighVelocity")) {
            fastProfile();
            moveToAbsoluteTargetPosition(approachStandbyPosition);
        }
    }

    public void moveToApproachOnlinePositionWithHighVelocity() {
        try (AutoTimed at = new AutoTimed("moveToApproachOnlinePositionWithHighVelocity")) {
            fastProfile();
            moveToAbsoluteTargetPosition(approachOnlinePosition);
        }
    }

    /**
     * To be able to push filter less or more into a socket, we need to add a
     * deltaStandbyPosition negative or positive to standbayPosition.
     *
     * @param deltaStandbyPosition
     *
     * See https://jira.slac.stanford.edu/browse/LSSTCCSFCS-507
     */
    public void moveToStandbyPlusDeltaWithLowVelocity(int deltaStandbyPosition) {
        try (AutoTimed at = new AutoTimed("moveToStandbyWithLowVelocity")) {
            FCSLOG.info(name + " === moveToStandbyWithLowVelocity ===");
            autochanger.checkCarouselDeltaPosition();
            slowProfile();
            if (autochanger.isEmpty()) {
                moveToAbsoluteTargetPosition(standbyPositionEmpty);
            } else {
                moveToAbsoluteTargetPosition(standbyPosition + deltaStandbyPosition);
            }
        }
    }

    public void moveToStandbyWithLowVelocity() {
        moveToStandbyPlusDeltaWithLowVelocity(0);
    }

    /**
     * to fix carousel unlocking mechanical issues at SLAC, a parameter
     * standbyPositionEmpty has been created.
     */
    public void moveToStandbyEmptyWithLowVelocity() {
        FCSLOG.info(name + " === moveToStandbyEmptyWithLowVelocity ===");
        autochanger.checkCarouselDeltaPosition();
        slowProfile();
        moveToAbsoluteTargetPosition(standbyPositionEmpty);
    }

    public void moveToHandoffWithHighVelocity() {
        fastProfile();
        moveToAbsoluteTargetPosition(handoffPosition);
    }

    /**
     * do go to position given as argument without any control. no control are done
     * user must check that this moving is possible.
     *
     * @param targetPosition
     */
    private void goToPosition(int targetPosition) {
        try (ActionGuard g = new ReleasedACBrakes()) {
            FCSLOG.info(name + " moving to position " + targetPosition);
            if (!isHomingDone()) {
                homing();
            }
            this.absoluteTargetPosition = targetPosition;
            /* see startAction */
            this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
            main.updateAgentState(FcsEnumerations.FilterState.TRUCKS_STOPPED);
            /**
             * update carousel and loader (fake or non fake) because it's no more done in
             * updateStateWithSensorsToCheckIfActionIsCompleted
             */
            main.updateStateWithSensors();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT,
            description = "goToPosition in bypassing all safety tests. Use this command only if you know what you are doing.")
    public void recoveryGoToPosition(int pos) {
        goToPosition(pos);
    }

    /**
     * activate both brakes then go to state SwitchOnDisabled for the 2 controllers.
     * For end * user.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Activate brakes and disable the 2 controllers.")
    public void activateBrakesAndDisable() {
        try (AutoTimed at = new AutoTimed("2Trucks-activateBrakesAndDisable")) {
            FcsUtils.parallelRun(() -> this.linearRailDriverController.activateBrake(),
                    () -> this.linearRailFollowerController.activateBrake());
            /*
             * This time is useful because brake activation delay is not controlled by the
             * LR controller
             */
            FcsUtils.sleep(autochanger.getWaitTimeForBrakeLR(), name);
            disableAllRails();
        }
    }

    private void disableAllRails() {
        try (AutoTimed at = new AutoTimed("2Trucks-disableAllRails")) {
            // basic implementation
            // linearRailDriverController.goToSwitchOnDisabled();
            // linearRailFollowerController.goToSwitchOnDisabled();
            // we want to do this in parallel

            // ??? and no need to wait so pure async run is ok seems to be an issue ?? Let's
            // remove for the moment

            // disable takes 100-150 ms for the rails.

            // FcsUtils.asyncRun(() -> linearRailDriverController.goToSwitchOnDisabled());
            // FcsUtils.asyncRun(() -> linearRailFollowerController.goToSwitchOnDisabled());

            FcsUtils.parallelRun(() -> linearRailDriverController.goToSwitchOnDisabled(),
                    () -> linearRailFollowerController.goToSwitchOnDisabled());

        }
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        // This PLC test is needed because the autochanger PLC is slow to update.
        // If we ask the controller to move when the relay is off the controller
        // goes in UNDER_VOLTAGE error state.
        autochanger.checkLinearRailMotionAllowed();
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                checkControllersMode();
                // enableControllersAndReleaseBrake(); useless now because it's done by ActionGuard.
                linearRailDriverController.writeTargetPosition(absoluteTargetPosition);
                linearRailDriverController.writeControlWord(0x3F);
                // TODO uncomment when monitoring will be reactivated
                // autochanger.increaseLinearRailsCurrentMonitoring();
                break;

            case ALIGN_FOLLOWER:
                linearRailFollowerController.enableAndReleaseBrake();
                linearRailFollowerController.writeTargetPosition(followerTargetPosition);
                linearRailFollowerController.writeControlWord(0x3F);
                break;

            case MOVE_FOLLOWER_TRUCK_ALONE:
                // linearRailFollowerController.enableAndReleaseBrake(); useless now because it's done by ActionGuard.
                linearRailFollowerController.writeTargetPosition(absoluteTargetPosition);
                linearRailFollowerController.writeControlWord(0x3F);
                break;

            default:
                assert false : action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.info(name + " ABORTING action: " + action.toString() + " within delay " + delay);
        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            quickStopAction(action, 0);
            // activateBrakesAndDisable();
            // autochanger.decreaseLinearRailsCurrentMonitoring();

        } else if (action == ALIGN_FOLLOWER) {
            stopAlignFollower();

        } else {
            // activateBrakesAndDisable();
        }

    }

    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.info(name + " ENDING action: " + action.toString());
        // Mandatory to update state because it's no more done in
        // updateStateWithSensorsToCheckIfActionIsCompleted
        autochanger.updateStateWithSensors();
        autochanger.carouselUpdateStandbyState();
        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            // handled by guards
            // activateBrakesAndDisable();
            // TODO uncomment when monitoring will be reactivated
            // autochanger.decreaseLinearRailsCurrentMonitoring();

        } else if (action == ALIGN_FOLLOWER) {
            stopAlignFollower();

        } else {
            // disableAllRails();
        }
        linearRailDriverController.checkFault();
        linearRailFollowerController.checkFault();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.linearRailDriverController.quickStop();
        this.linearRailFollowerController.quickStop();
    }

    /**
     * This method is used in the startAction method. It is was has to be done
     * before moving the trucks.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Enable and release brake for the 2 controllers.")
    public void enableControllersAndReleaseBrake() {
        try (AutoTimed at = new AutoTimed("2Trucks-enableControllersAndReleaseBrake")) {
            FcsUtils.parallelRun(() -> {
                linearRailDriverController.goToOperationEnable();
                linearRailDriverController.doReleaseBrake();
            }, () -> {
                linearRailFollowerController.goToOperationEnable();
                linearRailFollowerController.doReleaseBrake();
            });
        }
    }

    private void checkControllersMode() {
        if (!linearRailDriverController.isInMode(PROFILE_POSITION)) {
            throw new FcsHardwareException(
                    name + " : linearRailDriverController is not" + " in PROFILE_POSITION mode, can't move.");
        }

        if (!linearRailFollowerController.isInMode(MASTER_ENCODER)) {
            throw new FcsHardwareException(name + " : linearRailFollowerController is not"
                    + " in MASTER_ENCODER mode, can't move. Follower controller mode ="
                    + linearRailFollowerController.getMode());
        }
    }


    /**
     * move follower truck to driverPosition. When moving along the rails with follower
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignement. This consists in : setting follower controller mode to
     * PROFILE_POSITION, go to absolute driverPosition pos, and then set
     * follower controller mode back to MASTER_ENCODER mode.
     *
     * Note : switching the mode implies that brakes have to be set.
     * @param deltaMax
     */
    public void alignFollower(int deltaMax) {
        try (AutoTimed at = new AutoTimed("alignFollower")) {
            updatePosition();

            if (!isHomingDone()) {
                homing();
            }

            int startDeltaPosition = deltaPosition;

            FCSLOG.info("alignFollower delta " + deltaPosition + " (max " + deltaMax + ") "
                    + (Math.abs(deltaPosition) > deltaMax ? "ALIGN" : "no align"));

            if (Math.abs(deltaPosition) > deltaMax) {
                try (ActionGuard g = new ActivatedACBrakes()) {

                    // master controller must be disabled and brake activated
                    // this should be taken care of by the guard logic
                    if (linearRailDriverController.isEnabled()) {
                        FCSLOG.info("!!! driver controller was enabled");
                        this.activateBrakesAndDisable();
                    }
                    if (linearRailDriverController.isEnabled()) {
                        FCSLOG.warn("!!! driver controller was STILL enabled");
                        throw new FcsHardwareException(name + " driver controller must be disabled.");
                    }

                    doAlignFollower(driverPosition);
                    if (Math.abs(deltaPosition - startDeltaPosition) < 5) {
                        throw new FcsHardwareException(name + " alignFollower did nothing : is dirty power on ?");
                    }
                }
            }
        }
    }

    public void alignFollowerLenient() {
        alignFollower(deltaPositionForAlignLenient);
    }

    public void alignFollowerStrict() {
        alignFollower(deltaPositionForAlignStrict);
    }

    public void alignFollowerForLoader() {
        alignFollower(deltaPositionForAlignStrictLoader);
    }

    private void stopAlignFollower() {
        FCSLOG.info(name + " STOP ALIGN FOLLOWER");
        /* activate brake and wait until brake is activated */
        linearRailFollowerController.activateBrake();
        /* mandatory : we must let time to the controller to activate the brake */
        /*
         * otherwise if we disable voltage before brake are holding, a filter could
         * fall.
         */
        FcsUtils.sleep(autochanger.getWaitTimeForBrakeLR(), name);
        linearRailFollowerController.changeMode(MASTER_ENCODER);
        linearRailFollowerController.goToSwitchOnDisabled();
        FCSLOG.info(name + " END STOP ALIGN FOLLOWER");
    }

    private void doAlignFollower(int pos) {
        try (AutoTimed at = new AutoTimed("doAlignFollower")) {
            linearRailFollowerController.changeMode(PROFILE_POSITION);
            FCSLOG.info(name + " follower's alignment to be done. deltaPosition= " + deltaPosition);
            if (!linearRailFollowerController.isInMode(PROFILE_POSITION)) {
                throw new FcsHardwareException(
                        name + " can't align follower truck because" + " couldn't change its mode to PROFILE_POSITION.");
            }
            followerTargetPosition = pos;
            /* see startAction and postAction */
            this.executeAction(ALIGN_FOLLOWER, timeoutForAlignFollower);
        }
    }

    /**
     * move truck which is controlled by follower controller independently of truck
     * which is controlled by master controller
     *
     * @param newPos
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_EXPERT, description = "move truck which is controlled by follower controller independently of truck "
            + "which is controlled by driver controller")
    public void moveFollowerTruck(int newPos) {
        int followerPos = linearRailFollowerController.readPosition();
        if (Math.abs(followerPos - newPos) >= 2000) {
            throw new IllegalArgumentException(name + " newPos must be such as |newPos - followerPosition| < 2000");
        }
        // just to be safe.
        this.activateBrakesAndDisable();
        linearRailFollowerController.changeMode(PROFILE_POSITION);
        absoluteTargetPosition = newPos;
        this.executeAction(MOVE_FOLLOWER_TRUCK_ALONE, timeoutForTrucksMotion);
    }

    /**
     * This method is called at the end of actions to check that after a motion to
     * well known position as STANDBY, HANDOFF and ONLINE, position sensors confirm
     * the trucks position. Otherwise it throws a FailedCommandException.
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void checkPositionSensors() {
        /*
         * Read position sensors and check that values return are consistant with trucks
         * position otherwise throw an Exception.
         */
        autochanger.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(
                    name + ": check with sensors: STANDBY sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition=" + this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition=" + this.handoffPosition);
            FCSLOG.info("this.atHandoff=" + this.atHandoff);
            throw new FailedCommandException(
                    name + ": check with sensors: HANDOFF sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(
                    name + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    /**
     * Do homing of both controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Do homing for both  controllers.")
    public void homing() {
        try (AutoTimed at = new AutoTimed("AC2Trucks-homing")) {
            linearRailDriverController.homing();
            linearRailFollowerController.homing();
            updatePosition();
        }
    }

    /**
     * Updates the field position of the trucks in reading the CPU of the
     * controller.
     *
     * *********************************************************************
     * DRIVER CONTROLLER FOR truckXminus FOLLOWER CONTROLLER FOR truckXplus
     * *********************************************************************
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update trucks position in reading controller.")
    public void updatePosition() {
        this.truckXminus.updatePosition();
        this.truckXplus.updatePosition();
        this.driverPosition = linearRailDriverController.getPosition();
        this.followerPosition = linearRailFollowerController.getPosition();
        this.deltaPosition = this.driverPosition - this.followerPosition;
        this.publishData();
    }

    // TODO this should nearly never be called, we should align, except in places we
    // are not allowed to do so
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read position on controllers then compute difference and throw exception"
            + "if difference is > 1000.")
    public void checkDeltaPosition() {
        updatePosition();
        if (Math.abs(deltaPosition) > 1000) {
            String msg = "Too much difference between trucks position. " + "Alignment of Follower Truck to be done.";
            throw new FcsHardwareException(name + ":" + msg);
        }
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks driverPosition
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update trucks position in reading controller and return position.")
    public int readPosition() {
        updatePosition();
        return driverPosition;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return a voltage in mV which represents a distance between between filter"
            + "and filter beam.")
    public double readProximitySensorDevice() {
        readProximityVoltage();
        safeReadProximitySensorDevice();
        return proximityVoltage;
    }

    /**
    * Reads the proximity sensor current value
    *
    */
    private void readProximityVoltage() {
        this.proximityVoltage = proximitySensorsDevice.readVoltage(this.proximitySensorInput);
    }

    /**
    * Requested by Pierre Antilogus during the April 2023 testing phase
    * to have a window of 200ms to let the proximity sensors see the filter
    *
    */
    private void safeReadProximitySensorDevice() {
    FcsUtils.waitCondition(
        () -> proximityVoltage > 0,
        () -> readProximityVoltage(),
        "safeReadProximityVoltage",
        200);
    }

    /**
     * @return true if both controllers are initialized and homing of trucks are
     *         done.
     */
    @Override
    public boolean myDevicesReady() {
        return this.isInitialized();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                FCSLOG.info(name + " /driverPosition=" + driverPosition + " /position to reach = "
                        + absoluteTargetPosition + " /positionRange = " + positionRange);
                return Math.abs(this.driverPosition - this.absoluteTargetPosition) < positionRange;

            case ALIGN_FOLLOWER:
                FCSLOG.info(name + " linearRailFollowerController.isTargetReached()="
                        + linearRailFollowerController.isTargetReached() + " /driverPosition=" + driverPosition);
                return linearRailFollowerController.isTargetReached();

            case MOVE_FOLLOWER_TRUCK_ALONE:
                return linearRailDriverController.isTargetReached()
                        && Math.abs(this.driverPosition - this.absoluteTargetPosition) < positionRange;

            default:
                assert false : action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            acTruckXminusController.checkFault();
            acTruckXplusController.checkFault();
            // TODO do not read position 2 times
            updatePosition();
            acTruckXminusController.updateDataDuringMotionFromSDO();
            acTruckXplusController.updateDataDuringMotionFromSDO();
            acTruckXminusController.publishDataDuringMotion();
            acTruckXplusController.publishDataDuringMotion();
        } catch (SDORequestException ex) {
            raiseWarning(SDO_ERROR, "=> ERROR IN READING CONTROLLERS:", name, ex);
        }
    }

    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac2Trucks")) {
            synchronized (this) {
                this.atHandoff = truckXminus.isAtHandoff() && truckXplus.isAtHandoff();
                this.atOnline = truckXminus.isAtOnline() && truckXplus.isAtOnline();
                this.atStandby = truckXminus.isAtStandby() && truckXplus.isAtStandby();
            }
            updatePosition();
        }
    }

    /**
     * change speed & acceleration profile to slow
     *
     */
    public void slowProfile() {
        slowProfileVelocity();
        slowProfileAcceleration();
        slowProfileDeceleration();

        checkProfileVelocity();
        checkProfileAcceleration();
        checkProfileDeceleration();

        linearRailDriverController.publishData();
        linearRailFollowerController.publishData();
    }

    /**
     * change speed and acceleration profile to fast
     */
    public void fastProfile() {
        raiseProfileVelocity();
        raiseProfileAcceleration();
        raiseProfileDeceleration();

        checkProfileVelocity();
        checkProfileAcceleration();
        checkProfileDeceleration();
        linearRailDriverController.publishData();
        linearRailFollowerController.publishData();
    }

    /**
     * change ProfileVelocity parameter to lowSpeed
     *
     */
    public void slowProfileVelocity() {
        linearRailDriverController.changeProfileVelocity(lowSpeed);
        linearRailFollowerController.changeProfileVelocity(lowSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + lowSpeed);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    public void raiseProfileVelocity() {
        linearRailDriverController.changeProfileVelocity(highSpeed);
        linearRailFollowerController.changeProfileVelocity(highSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + highSpeed);
    }

    /**
     * check that ProfileVelocity has same value in driver and in follower.
     */
    private void checkProfileVelocity() {
        int followerV = (int) linearRailFollowerController.readParameter(ProfileVelocity);
        int driverV = (int) linearRailDriverController.readParameter(ProfileVelocity);
        if (followerV != driverV) {
            throw new FcsHardwareException(name + " driver and follower ProfileVelocity are not the same."
                    + " DRIVER velocity =" + driverV + " and" + " FOLLOWER velocity =" + followerV);
        }
    }

    /**
     * check that ProfileAcceleration has same value in driver and in follower.
     */
    private void checkProfileAcceleration() {
        long followerAcc = linearRailFollowerController.readParameter(ProfileAcceleration);
        long driverAcc = linearRailDriverController.readParameter(ProfileAcceleration);
        if (followerAcc != driverAcc) {
            throw new FcsHardwareException(name + " driver and follower ProfileAcceleration are not the same."
                    + " DRIVER acceleration =" + driverAcc + " and" + " FOLLOWER acceleration =" + followerAcc);
        }
    }

    /**
     * check that ProfileDeceleration has same value in driver and in follower.
     */
    private void checkProfileDeceleration() {
        long followerDec = linearRailFollowerController.readParameter(ProfileDeceleration);
        long driverDec = linearRailDriverController.readParameter(ProfileDeceleration);
        if (followerDec != driverDec) {
            throw new FcsHardwareException(name + " driver and follower ProfileDeceleration are not the same."
                    + " DRIVER deceleration =" + driverDec + " and" + " FOLLOWER deceleration =" + followerDec);
        }
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    public void raiseProfileAcceleration() {
        linearRailDriverController.changeProfileAcceleration(highAcceleration);
        linearRailFollowerController.changeProfileAcceleration(highAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + highAcceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    public void slowProfileAcceleration() {
        linearRailDriverController.changeProfileAcceleration(lowAcceleration);
        linearRailFollowerController.changeProfileAcceleration(lowAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + lowAcceleration);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    public void raiseProfileDeceleration() {
        linearRailDriverController.changeProfileDeceleration(highDeceleration);
        linearRailFollowerController.changeProfileDeceleration(highDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + highDeceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    public void slowProfileDeceleration() {
        linearRailDriverController.changeProfileDeceleration(lowDeceleration);
        linearRailFollowerController.changeProfileDeceleration(lowDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + lowDeceleration);
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setDriverPosition(driverPosition);
        s.setFollowerPosition(followerPosition);
        s.setHomingDone(isHomingDone());
        s.setAtHandoff(atHandoff);
        s.setAtOnline(atOnline);
        s.setAtStandby(atStandby);
        s.setInError(isPositionSensorsInError());
        s.setProximityVoltage(proximityVoltage);
        s.setProximityDistance((int) (adjustmentFactor * (utarget - proximityVoltage)));
        return s;
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByAutoChangerTrucks()));
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (linearRailDriverController.isBooted() && linearRailFollowerController.isBooted()) {
            this.activateBrakesAndDisable();
        }
        FCSLOG.info(name + " is shutdown.");
    }

    public class ReleasedACBrakes extends ActionGuard {
        public Runnable getPreAction() {
            return () -> enableControllersAndReleaseBrake();
        }

        public Runnable getPostAction() {
            return () -> activateBrakesAndDisable();
        }
    }

    public class ActivatedACBrakes extends ActionGuard {
        public Runnable getPreAction() {
            return () -> activateBrakesAndDisable();
        }

        public Runnable getPostAction() {
            return () -> enableControllersAndReleaseBrake();
        }

        public Class<? extends ActionGuard> requiredEnclosure() {
            return ReleasedACBrakes.class;
        }
    }

}
