
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.HOMING;
import org.lsst.ccs.subsystems.fcs.LoaderClamp;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CLAMP_NAME;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is to simulate an EPOS Controller which controls the loader hooks.
 *
 * @author virieux
 */
public class SimuLoaderClampController extends SimuEPOSController {

    @LookupField(strategy = TREE, pathFilter = ".*\\/" + LOADER_CLAMP_NAME)
    private LoaderClamp clamp;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private SimuLoaderPlutoGateway plutoGateway;

    @LookupPath
    private String nodePath;
    
    static int POSITION_CLAMPED = 350000;
    static int POSITION_OPENED = 0;

    @Override
    public void init() {
        if ( plutoGateway == null ) {
            throw new RuntimeException("Invalid initialization. An instance of SimuAutochangerPlutoGateway should have been "
                    + "picked up from the Lookup tree from the siblinds of this node: "+nodePath);
        }
    }
    
    @Override
    public void writeCurrent(int aValue)  {
        this.targetCurrent = aValue;
        int currentInitial = this.actualCurrent;
        FCSLOG.debug("=>initial current=" + actualCurrent);
        int stepsNB = 10;
        int step = (targetCurrent - this.actualCurrent) / stepsNB;

        for (int i = 1; i < stepsNB; i++) {

            this.actualCurrent = (short) (currentInitial + (i * step));
            clamp.updateCurrent();
            if (actualCurrent == clamp.getCurrentToClamp()) {
                FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
                this.position = POSITION_CLAMPED;
                this.plutoGateway.simulateLoaderClampIsClosed();

            } else if (actualCurrent == clamp.getCurrentToOpen()) {
                this.position = POSITION_OPENED;
                this.plutoGateway.simulateLoaderClampIsOpened();
            }
            clamp.publishData();
            FCSLOG.debug("i=" + i + ",actualCurrent=" + actualCurrent);
            try {
                Thread.sleep(10);
                if (clamp.getHaltRequired().get()) {
                    FCSLOG.debug(name + " STOP simulated clamp motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                throw new FcsHardwareException(name + " sleep was interrupted.", ex);
            }

        }
        this.actualCurrent = aValue;
        if (actualCurrent == clamp.getCurrentToClamp()) {
            FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
            this.position = POSITION_CLAMPED;
            this.plutoGateway.simulateLoaderClampIsClamped();

        } else if (actualCurrent == clamp.getCurrentToOpen()) {
            this.position = POSITION_OPENED;
            this.plutoGateway.simulateLoaderClampIsOpened();
        }
        clamp.publishData();
    }

    @Override
    public void writeTargetPosition(int positionToReached) {
        this.targetPosition = positionToReached;
        int positionInitiale = this.position;
        FCSLOG.info("=>position intitiale=" + position);
        FCSLOG.info("=>targetPosition=" + targetPosition);
        FCSLOG.info("clamp.getTargetPositionToClamp()=" + POSITION_CLAMPED);
        FCSLOG.info("clamp.getRelativePositionToUnclamp()=" + clamp.getRelativePositionToUnclamp());
        int stepsNB = 10;
        int step = (targetPosition - this.position) / stepsNB;

        for (int i = 1; i < stepsNB; i++) {
            this.position = positionInitiale + (i * step);
            clamp.updatePosition();
            simulateSensors();
            FCSLOG.info("i=" + i + ",position=" + position);
            try {
                Thread.sleep(10);
            } catch (InterruptedException ex) {
                throw new RejectedCommandException(name + " sleep was interrupted.");
            }
        }
        this.position = targetPosition;
        FCSLOG.info("i=" + 10 + ",position=" + position);
        simulateSensors();
    }

    private void simulateSensors() {
        if (position == POSITION_OPENED) {
            FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
            this.plutoGateway.simulateLoaderClampIsOpened();

        } else if (position == clamp.getRelativePositionToClose() ||
                position == POSITION_CLAMPED + clamp.getRelativePositionToUnclamp()) {
            this.plutoGateway.simulateLoaderClampIsClosed();

        } else if (position == POSITION_CLAMPED) {
            this.plutoGateway.simulateLoaderClampIsClamped();

        } else {
            this.plutoGateway.simulateLoaderClampIsInTravel();
        }
        clamp.publishData();
    }

    @Override
    public void changeMode(EposMode newmode) {
        super.changeMode(newmode);
        if (HOMING.equals(newmode)) {
            writeCurrent(clamp.getCurrentToOpen());
        }
    }
}


