
package org.lsst.ccs.subsystems.fcs.simulation;

import static java.lang.Math.abs;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.Autochanger;

import org.lsst.ccs.subsystems.fcs.AutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoLatches;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * To simulate a controller which moves autochanger trucks along its linear
 * rails.
 *
 * @author virieux
 */
public class SimuAutochangerLinearRailController extends SimuEPOSController implements EPOSControllerForLinearRail {

    @ConfigurationParameter(description = "minimal position of the encoder ribbon.", category = "controller")
    private volatile int encoderRibbonMinValue;

    @LookupField(strategy = TREE)
    private Autochanger autochanger;
    private AutochangerTwoTrucks trucks;
    
    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private SimuAutochangerPlutoGateway plutoGateway;

    @LookupPath
    private String nodePath;
    
    @LookupField(strategy = TREE)
    private SimuTTC580 hyttc580;

    @LookupField(strategy = TREE, pathFilter = "carousel")
    private FilterHolder carousel;
    private AutochangerTwoLatches latches;

    /* this field is null for follower controller, it's initialized to follower controller for the driver controller.*/
    private SimuAutochangerLinearRailController followerController;

    //in the simulation, homing is always done.
    private boolean homingDone = true;

    private boolean brakeActivatedPub;

    @Override
    public boolean isBrakeActivatedPub() {
        return brakeActivatedPub;
    }

    @Override
    public void setBrakeActivatedPub(boolean brakeActivatedToPublish) {
        this.brakeActivatedPub = brakeActivatedToPublish;
    }

    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
    }

    public int getEncoderRibbonMinValue() {
        return encoderRibbonMinValue;
    }


    private void setTrucksPosition(int pos) {
        this.position = pos;
        if (isDriver()) {
            this.followerController.setPosition(pos);
        } 
        this.ssiPosition = this.encoderRibbonMinValue + pos;
    }
    
    private boolean isDriver() {
        return this.getName().contains(trucks.getDriverSide());
    }

    @Override
    public void init() {
        if ( plutoGateway == null ) {
            throw new RuntimeException("Invalid initialization. An instance of SimuAutochangerPlutoGateway should have been "
                    + "picked up from the Lookup tree from the siblinds of this node: "+nodePath);
        }
        ComponentLookup lookup = subs.getComponentLookup();
        this.trucks = (AutochangerTwoTrucks) lookup.getComponentByPath("autochangerTrucks");
        this.latches = (AutochangerTwoLatches) lookup.getComponentByPath("latches");
        this.setSsiPosition(encoderRibbonMinValue);
        /* if this controller is Master controller, field followerController is initialized to linearRailFollowerController*/
        if (isDriver()) {
            if ("Xminus".contains(trucks.getDriverSide())) {
                this.followerController = (SimuAutochangerLinearRailController) lookup.getComponentByPath("acTruckXplusController");
            } else {
                this.followerController = (SimuAutochangerLinearRailController) lookup.getComponentByPath("acTruckXminusController");
            }
        }
        if (!isDriver()) {
            this.mode = MASTER_ENCODER;
        }
    }


    /**
     * Simulates a command writeTargetPosition : the position given as argument
     * is the position to be reached. This method updates position incremently
     * to simulate a real trucks with a motor controller.
     *
     * @param positionToReached
     */
    @Override
    public void writeTargetPosition(int positionToReached) {
        this.targetPosition = positionToReached;
        int positionInitiale = this.position;
        FCSLOG.debug("AC trucks are moving.");
        FCSLOG.debug(name + "=> initial position = " + position);
        int stepsNB;
        if (abs(targetPosition - this.position) > 10000) {
            stepsNB = 20;
        } else {
            stepsNB = 5;
        }

        int stepHeight = (targetPosition - this.position) / stepsNB;

        simulateTrucksMotion(positionInitiale, stepsNB, stepHeight);

        setTrucksPosition(targetPosition);
        trucks.updatePosition();
        trucks.publishData();
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());

        if (position == trucks.getHandoffPosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsAtHandoff");
            this.plutoGateway.simulateAutochangerTrucksAreAtHandoff();

        } else if (position == trucks.getStandbyPosition()) {
            this.plutoGateway.simulateAutochangerTrucksAreAtStandby();
            if (carousel.isHoldingFilter()) {
                FCSLOG.info("simulateFilterIsOnAutochanger");
                this.plutoGateway.simulateFilterIsOnAutochanger();
            }

        } else if (position == trucks.getOnlinePosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsOnline");
            this.plutoGateway.simulateAutochangerTrucksAreOnline();

        } else if (position <= trucks.getApproachOnlinePosition()
                && position > trucks.getOnlinePosition()) {
            this.plutoGateway.simulateOnlineSensorsOn();

        } else {
            FCSLOG.info("simulateAutochangerTrucksIsInTravel");
            this.plutoGateway.simulateAutochangerTrucksAreInTravel();
        }
        if (hyttc580 != null && autochanger.isHoldingFilter()) {
            updateSocketAtStandbyState(positionInitiale, position, autochanger.getFilterID());
        }
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());
    }

    /**
     * simulates FilterPresenceSensors according to the position of the trucks
     * and publishes trucks data.
     *
     * @param positionInitiale
     * @param stepHeight
     */
    private void simulateFilterPresenceSensors(int positionInitiale, int stepHeight) {
        /* as AC is going to STANDBY, if there is a filter the carousel, simulated AC must "see" the filter*/
        if (positionInitiale < trucks.getStandbyPosition()
                && position > trucks.getStandbyPosition() - (2 * stepHeight)
                && carousel.isHoldingFilter()) {
            FCSLOG.info("simulateFilterIsOnAutochanger");
            this.plutoGateway.simulateFilterIsOnAutochanger();
            //on autochanger-standalone, carousel.getFilterID() == 0
            if (carousel.getFilterID() != 0) {
                this.plutoGateway.simulateFilterIDOnAutochanger(carousel.getFilterID());
            }

            /* as AC is going away from STANDBY simulated AC must be updated because it can't see the filter anymore*/
        } else if (positionInitiale == trucks.getStandbyPosition()
                && position < trucks.getStandbyPosition() - (2 * stepHeight)
                && latches.isOpened()) {
            FCSLOG.info("simulateAutochangerIsEmpty");
            this.plutoGateway.simulateAutochangerIsEmpty();
        }
        autochanger.updateStateWithSensors();
        trucks.publishData();
    }

    /**
     * simulates trucks motion : increment or decrement position value from an
     * initialPosition, with a number of steps and a step height.
     *
     * @param positionInitiale
     * @param stepsNB number of steps
     * @param stepHeight
     */
    private void simulateTrucksMotion(int positionInitiale, int stepsNB, int stepHeight) {
        for (int i = 1; i < stepsNB; i++) {
            setTrucksPosition(positionInitiale + (i * stepHeight));
            trucks.updatePosition();
            FCSLOG.info("trucks Position=" + trucks.readPosition());
            simulateFilterPresenceSensors(positionInitiale, stepHeight);
            FCSLOG.debug(name + "i=" + i + ",position=" + position);
            try {
                Thread.sleep(100);
                if (trucks.getHaltRequired().get()) {
                    FCSLOG.debug(name + " STOP simulated trucks motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                FCSLOG.error(ex);
                throw new RejectedCommandException(name + " sleep was interrupted.");
            }
        }
    }

    /**
     * update socket at standby to be executed only if hyttc580 is not null
     *
     * @param initialPosition
     * @param finalPosition
     */
    private void updateSocketAtStandbyState(int initialPosition, int finalPosition, int filterID) {

        /* case : disengage filter from carousel*/
        if (finalPosition == trucks.getApproachStandbyPosition()
                && initialPosition == trucks.getStandbyPosition()) {
            hyttc580.updateDisengageFromAC();
        }

        if (finalPosition == trucks.getStandbyPosition()
                && initialPosition == trucks.getApproachStandbyPosition()) {
            FCSLOG.finest("filter ID on AC before hyttc580.updateStoreFilterOnCarousel = " + autochanger.getFilterID());
            hyttc580.updateStoreFilterOnCarousel(filterID);            
            FCSLOG.finest("filter ID on AC after hyttc580.updateStoreFilterOnCarousel = " + autochanger.getFilterID());
            plutoGateway.simulateFilterIDOnAutochanger(filterID);
        }
    }

    @Override
    public void configureDigitalInputOfLinearRails() {
    }

    @Override
    public void configureDigitalOutputOfLinearRails() {
    }

    @Override
    public void homing() {
        homingDone = true;
    }

    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    @Override
    public void activateBrakeAndDisable() {
        activateBrake();
        /* mandatory otherwise issues !!!*/
        shutdownController();

    }

    @Override
    public void doReleaseBrake() {
        brakeActivatedPub = false;
    }

    @Override
    public boolean isBrakeActivated() {
        return brakeActivatedPub;
    }

    @Override
    public void activateBrake() {
        brakeActivatedPub = true;
    }
}
