
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.ANCESTORS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class represents a model for the 2 Latches of the autochanger. The
 * Latches hold a filter on the trucks. There is one latch on each side of the
 * filter. Because the ACTION open (or close) for 2 latches is different from
 * opening (or closing) one Latch, then the other one, it's more convenient to
 * have a class to represent 2 Latches at a time.
 *
 * @author virieux
 */
public class AutochangerTwoLatches implements AlertRaiser, SignalHandler, HasLifecycle {

    @LookupName
    protected String name;

    @LookupField(strategy = ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    private Autochanger autochanger;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;
    
    private final AutochangerLatch latchXminus;
    private final AutochangerLatch latchXplus;

    private volatile LockStatus lockStatus = UNKNOWN;
    private volatile FilterPresenceStatus filterPresenceStatus = FilterPresenceStatus.UNKNOWN;

    /**
     * Build an Object AutochangerTwoLatches from 2 AutochangerLatch.
     *
     * @param latchXminus
     * @param latchXplus
     */
    public AutochangerTwoLatches(AutochangerLatch latchXminus, AutochangerLatch latchXplus) {
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerTwoLatches.class, name);
    }

    @Override
    public void init() {

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);
    }
    
    /**
     * Returns true if the 2 latches controllers are initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if the 2 latches are initialized.")
    public boolean isInitialized() {
        return latchXminus.isInitialized() && latchXplus.isInitialized();
    }

    /**
     * return filterPresenceStatus
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Return if both latches are CLOSED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if both latches are CLOSED.")
    public boolean isClosed() {
        return this.latchXminus.getLockStatus() == CLOSED && this.latchXplus.getLockStatus() == CLOSED;
    }

    /**
     * Return if both latches are OPENED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if both latches are OPENED.")
    public boolean isOpened() {
        return this.latchXminus.getLockStatus() == OPENED && this.latchXplus.getLockStatus() == OPENED;
    }

    /**
     * Returns true if one of the latches is in ERROR state. Doesn't read again
     * sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if one of the latches is in ERROR state.")
    public boolean isInError() {
        return this.latchXminus.getLockStatus() == ERROR || this.latchXplus.getLockStatus() == ERROR;
    }

    /**
     *
     * @return true if both latches are in Travel
     */
    private boolean isInTravel() {
        return this.latchXminus.getLockStatus() == INTRAVEL && this.latchXplus.getLockStatus() == INTRAVEL;
    }

    /**
     * Returns true if there is no filter is latches. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if there is no filter is latches.")
    public boolean isEmpty() {
        return this.latchXminus.isEmpty() && this.latchXplus.isEmpty();
    }

    public boolean isFilterEngagedInError() {
        return latchXminus.isFilterEngagedInError() || latchXplus.isFilterEngagedInError();
    }

    /**
     * Return true if autochanger is holding a filter. A filter is ENGAGED and
     * latches are CLOSED. Does not read again sensors values.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return true if autochanger is holding a filter.")
    public boolean isHoldingFilter() {
        return isFilterEngaged() && this.isClosed();
    }

    /**
     *
     * @return true if both latches detect filter presence.
     */
    private boolean isFilterEngaged() {
        return !this.latchXminus.isEmpty() && !this.latchXplus.isEmpty();
    }

    /**
     * Checks sensors.
     *
     * @throws FcsHardwareException if some latch sensors are in error or if latches
     *                              state is not equal on both side of the filter.
     * @throws FcsHardwareException
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        latchXminus.checkSensors(AC_SENSOR_ERROR, "latchXminus");
        latchXplus.checkSensors(AC_SENSOR_ERROR, "latchXplus");
        if (latchXminus.getLockStatus() != latchXplus.getLockStatus()) {
            String msg = name + " sensors are in ERROR: latchXminus state=" + latchXminus.getLockStatus()
                    + " latchXplus state=" + latchXplus.getLockStatus();
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(
                    name + " LockStatus is different on both side. Can't close neither open latches.");
        }
        if (latchXminus.isEmpty() != latchXplus.isEmpty()) {
            String msg = name + " Error in filterPresenceSensors of autochanger : the 2 sensors send different values."
                    + " latchXminus.isEmpty=" + this.latchXminus.isEmpty() + " latchXplus.isEmpty="
                    + this.latchXplus.isEmpty();
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg + "Can't close neither open latches.");
        }
    }

    /**
     * Update lockStatus from latchXminus state and latchXplus state.
     *
     */
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-latches")) {
            synchronized (this) {
                this.latchXminus.updateState();
                this.latchXplus.updateState();
                computeLockStatus();
                computeFilterPresenceStatus();
            }
            this.publishData();
        }
    }

    /**
     * Compute a global lock status for the 2 latches.
     */
    private void computeLockStatus() {

        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isClosed()) {
            this.lockStatus = CLOSED;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else {
            /* if lockStatus is not the same for the 2 latches */
            /*
             * can happen when closing (or opening) when a latch is already closed and not
             * the other
             */
            this.lockStatus = UNKNOWN;
        }
    }

    private void computeFilterPresenceStatus() {
        /* compute filter presence */
        if (this.latchXminus.isEmpty() != this.latchXplus.isEmpty()) {
            filterPresenceStatus = FilterPresenceStatus.ERROR;

        } else if (this.isEmpty()) {
            filterPresenceStatus = FilterPresenceStatus.NOFILTER;

        } else if (isFilterEngaged()) {
            filterPresenceStatus = FilterPresenceStatus.ENGAGED;

        } else {
            filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
        }
    }

    /**
     * Close latches
     *
     * @throws FcsHardwareException
     */
    public void close() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("closeLatches")) {
            FcsUtils.parallelRun(() -> latchXminus.close(), () -> latchXplus.close());
        }
    }

    /**
     * Open the 2 latches.
     *
     * @throws FcsHardwareException
     */
    public void open() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("openLatches")) {
            FcsUtils.parallelRun(() -> latchXminus.open(), () -> latchXplus.open());
        }
    }

    /**
     * initialize AC latches hardware after initialization. to be executed if during
     * boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Initialize AC latches hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        latchXminus.postStart();
        latchXplus.postStart();
    }

    /**
     * Creates an object to be published on the STATUS bus by AutochangerTwoLatches.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerTwoLatches createStatusDataPublishedByTwoLatches() {
        StatusDataPublishedByAutochangerTwoLatches status = new StatusDataPublishedByAutochangerTwoLatches();
        status.setLockStatus(lockStatus);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterName(autochanger.getFilterOnTrucksName());
        status.setFilterId(autochanger.getFilterID());
        return status;
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByTwoLatches()));
    }
}
