
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileVelocity;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ENGAGED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * This is a model for the carrier in the loader.
 *
 * @author virieux
 */
public class LoaderCarrier extends MobileItem implements MovedByEPOSController, ControlledBySensors {

    public static final int TIMEOUT_FOR_MOVING_CARRIER = 120000;

    public static final String cmdDesc = " Check and change ProfileVelocity, ProfileAcceleration and "
            + "ProfileDeceleration if needed. low velocity when filter is in loader, high velocity otherwise, "
            + "low acceleration when filter is in loader and between handoff and engaged, high acceleration otherwise, "
            + "low deceleration when filter is in loader and between handoff and engaged, high deceleration otherwise.";


    @LookupField(strategy=CHILDREN, pathFilter="loaderHandoffSensors")
    private BinarySensor handoffSensors;

    @LookupField(strategy=CHILDREN, pathFilter="loaderStorageSensors")
    private BinarySensor storageSensors;

    @LookupField(strategy=CHILDREN, pathFilter="loaderEngagedSensors")
    private BinarySensor engagedSensors;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private int position = 0;

    private int absoluteTargetPosition = 0;

    private int carrierMotionTimeout;

    @ConfigurationParameter(description = "delta position : used to know if carrier position "
            + "is in a range of 2*deltaPosition around a given position", units = "micron", category = "loader")
    private volatile int deltaPosition = 50;

    @ConfigurationParameter(description = "Loader Handoff position", units = "micron", category = "loader")
    private volatile int handoffPosition = 1037500;

    @ConfigurationParameter(description = "Loader Engaged position", units = "micron", category = "loader")
    private volatile int engagedPosition = 987500;

    @ConfigurationParameter(description = "Loader Storage position", units = "micron", category = "loader")
    private volatile int storagePosition = 0;

    @ConfigurationParameter(description = "For the Loader GUI : Maximum current to be sent to the Loader Carrier controller.", units = "mA", category = "loader")
    private volatile int maxCurrent = 1000;

    @ConfigurationParameter(description = "For the Loader GUI : Loader Carrier Maximum speed. [rpm]", units = "unitless", category = "loader")
    private volatile int maxSpeed = 227;

    @ConfigurationParameter(description = "timeout in milliseconds to go from storage to handoff on loader", units = "millisecond", category = "loader")
    private volatile long timeoutForGoingToHandOff = 120000;

    @ConfigurationParameter(description = "timeout in milliseconds to go from storage to engaged on loader", units = "millisecond", category = "loader")
    private volatile long timeoutForGoingToEngaged = 120000;

    @ConfigurationParameter(description = "timeout in milliseconds to go from handoff to storage on loader", units = "millisecond", category = "loader")
    private volatile long timeoutForGoingToStorage = 120000;

    @ConfigurationParameter(description = "low speed to go with a filter from ENGAGED to HANDOFF position [rpm]", units = "unitless", category = "loader")
    private volatile int lowSpeed = 50;

    @ConfigurationParameter(description = "high speed to move carrier when it's empty or with a filter from ENGAGED to STORAGE position [rpm]", units = "unitless", category = "loader")
    private volatile int highSpeed = 187;

    @ConfigurationParameter(description = "low acceleration to move carrier with a filter from ENGAGED to STORAGE position", category = "loader")
    private volatile int lowAcceleration = 30;

    @ConfigurationParameter(description = "high acceleration to move carrier empty from ENGAGED to STORAGE position", category = "loader")
    private volatile int highAcceleration = 100;

    @ConfigurationParameter(description = "low deceleration to move carrier with a filter from ENGAGED to STORAGE position", category = "loader")
    private volatile int lowDeceleration = 30;

    @ConfigurationParameter(description = "high deceleration to move carrier with a filter from ENGAGED to STORAGE position", category = "loader")
    private volatile int highDeceleration = 50;

    @LookupField(strategy = TREE, pathFilter = ".*\\/carrierController")
    private EPOSController carrierController;

    @LookupField(strategy = TREE)
    private Loader loader;

    @LookupField(strategy = TREE, pathFilter = "loaderTcpProxy")
    private BridgeToHardware loaderTcpProxy;

    private volatile boolean initialized = false;
    private volatile boolean homingDone = true;

    private int speed;
    private int current;
    private long profileVelocity;
    private long profileAcceleration;
    private long profileDeceleration;

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderCarrier.class, name);
        registerAction(MOVE_LOADERCARRIER_TO_HANDOFF);
        registerAction(MOVE_LOADERCARRIER_TO_ENGAGED);
        registerAction(MOVE_LOADERCARRIER_TO_STORAGE);
        registerAction(MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION);
    }

    @Override
    public void init() {
	super.init();
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(SDO_ERROR.getAlert(name), alwaysClear);
    }

    /**
     *
     * @return carrier position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns carrier position.")
    public int getPosition() {
        return position;
    }

    /**
     * returns current for tests
     *
     * @return
     */
    protected int getCurrent() {
        return current;
    }

    @Override
    public EPOSController getController() {
        return carrierController;
    }

    /**
     *
     * @return true if carrier is initialized and ready to receive commands.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if carrier is initialized and ready to receive commands.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Returns max speed - Used by GUI
     *
     * @return max speed in rpm/mn (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the max speed in rpm/mn (format decimal).")
    public int getMaxSpeed() {
        return maxSpeed;
    }

    /**
     * Returns max current - Used by GUI
     *
     * @return max current in mA (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the max current in mA (format decimal).")
    public int getMaxCurrent() {
        return maxCurrent;
    }

    /**
     *
     * @return handoff position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the handoff position in microns (format decimal).")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     *
     * @return
     */
    public int getEngagedPosition() {
        return engagedPosition;
    }

    /**
     *
     * @return storage position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the storage position in microns (format decimal).")
    public int getStoragePosition() {
        return storagePosition;
    }

    /**
     * Used by GUI
     *
     * @return timeout for going to Handoff action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the timeout for going to Handoff in millis (format decimal).")
    public long getTimeoutForGoingToHandOff() {
        return timeoutForGoingToHandOff;
    }

    /**
     * Used by GUI
     *
     * @return timeout for going to Storage action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the timeout for going to Storage in millis (format decimal).")
    public long getTimeoutForGoingToStorage() {
        return timeoutForGoingToStorage;
    }

    /**
     * returns false because position sensors can't be in error see
     * LSSTCCSFCS-267 and LSSTCCSFCS-242.
     *
     * @return
     */
    @Override
    public boolean isInError() {
        return false;
    }

    /**
     * This command is to be used by end user in case the carrier controller has
     * lost its position. This must not be launched automatically. Tested with
     * success in may 2018 on prototype.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Homing of carrier controller. Should be done only if necessary.")
    public void homing() {
        carrierController.definePositionFromNegativeLimitSwitch();
        carrierController.changeMode(PROFILE_POSITION);
        FCSLOG.info(name + " homing done.");
    }

    /**
     * initialize controller
     */
    public void initializeController() {
        try {
            carrierController.initializeAndCheckHardware();
            carrierController.changeMode(PROFILE_POSITION);
            this.initialized = true;
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize controller", name, ex);
        }
    }

    @Override
    public void postStart() {
        FCSLOG.info(name + " postStart");
        carrierMotionTimeout = (int) timeoutForGoingToStorage;
        if (carrierController.isBooted()) {
            initializeController();
            profileVelocity = readProfileVelocity();
            profileAcceleration = carrierController.readParameter(ProfileAcceleration);
            profileDeceleration = carrierController.readParameter(ProfileDeceleration);
            if (profileVelocity == lowSpeed) {
                carrierMotionTimeout = 4 * (int) timeoutForGoingToStorage;
            } else {
                carrierMotionTimeout = (int) timeoutForGoingToStorage;
            }
        }
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and
     * initialized.
     *
     * @return
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }

    /**
     * Returns the boolean field atStorage. If the atStorage boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atStorage is updated. If the field atStorage is not being updated, it returns
     * immediatly the field atStorage.
     *
     * @return atStorage
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at STORAGE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStorage() {
        return this.storageSensors.isOn()&& this.isCarrierAlmostAtPosition(storagePosition);
    }

    /**
     * Return true if the carrier is at HANDOFF position. This command doesn't read
     * again the sensors but waits if sensors are beeing updated. At HANDOFF
     * engagedSensors and handoffSensors are ON.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at HANDOFF position. "
            + "This command doesn't read again the sensors." + "At HANDOFF engagedSensors and handoffSensors are ON.")
    public boolean isAtHandoff() {
        return this.handoffSensors.isOn() && this.isCarrierAlmostAtPosition(handoffPosition);
    }

    /**
     * Return true if carrier is at ENGAGED position. Wait for response if sensors
     * are being updated otherwise returns immediatly. At ENGAGED position,
     * engagedSensors are ON and handoffSensors are OFF.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at ENGAGED position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtEngaged() {
        return this.engagedSensors.isOn() && this.isCarrierAlmostAtPosition(engagedPosition);
    }

    /**
     * return true if position < engagedPosition - deltaPosition
     * storagePosition is 0
     * engagedPosition is > 0 and between storage and handoff we have :
     * STORAGE(0)------|xxENGAGED(987500)xx|xxxxxxHANDOFF(1037500)
     *
     * deltaPosition is a configuration parameter which value is around 50.
     *
     * @return
     */
    public boolean isStrictlyBetweenStorageAndEngaged() {
        updatePosition();
        return (position < engagedPosition - deltaPosition);
    }

    /**
     * return true if position >= engagedPosition - deltaPosition
     * storagePosition is 0 engagedPosition is > 0 and between storage and
     * handoff we have :
     *
     * STORAGE(0)xxxxxx|--ENGAGED(987500)--|----HANDOFF(1037500)
     *
     * deltaPosition is a configuration parameter which value is around 50.
     *
     * @return
     */
    public boolean isLooslyBetweenEngagedAndHandoff() {
        updatePosition();
        return position >= engagedPosition - deltaPosition;
    }

    /**
     * return true if position > engagedPosition + deltaPosition storagePosition
     * is 0 engagedPosition is > 0 and between storage and handoff we have :
     * STORAGE(0)xxxxxx|xxENGAGED(987500)xx|------HANDOFF(1037500)
     *
     * deltaPosition is a configuration parameter which value is around 50.
     *
     * @return
     */
    public boolean isStrictlyBetweenEngagedAndHandoff() {
        updatePosition();
        return position > engagedPosition + deltaPosition;
    }

    /**
     * return true if position <= engagedPosition + deltaPosition.
     *
     * storage is 0 engagedPosition is > 0 and between storage and handoff we
     * have :
     *
     * STORAGE(0)------|--ENGAGED(987500)--|xxxxxxHANDOFF(1037500)
     *
     * deltaPosition is a configuration parameter which value is around 50.
     *
     * @return
     */
    public boolean isLooslyBetweenStorageAndEngaged() {
        updatePosition();
        return position <= engagedPosition + deltaPosition;
    }

    /**
     * Move the carrier to Handoff position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, alias = "goToHandoff",
            description = "Move the carrier to Handoff position. This command ", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void goToHandOff() {
        if (this.isAtHandoff()) {
            FCSLOG.info(name + " is already at Handoff position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(handoffPosition);
            this.absoluteTargetPosition = this.handoffPosition;
            this.executeAction(MOVE_LOADERCARRIER_TO_HANDOFF,
                    timeoutForGoingToHandOff);
            afterMotion();
        }
    }

    /**
     * Move the carrier to Handoff position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to Engaged position." + cmdDesc,
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void goToEngaged() {
        if (this.isAtEngaged()) {
            FCSLOG.info(name + " is already at Engaged position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(this.engagedPosition);
            this.absoluteTargetPosition = this.engagedPosition;
            this.executeAction(MOVE_LOADERCARRIER_TO_ENGAGED,
                    carrierMotionTimeout);
            afterMotion();
        }
    }

    /**
     * Moves loader carrier to storage position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to STORAGE position." + cmdDesc,
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void goToStorage() {
        if (this.isAtStorage()) {
            FCSLOG.info(name + " is already at Storage position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(this.storagePosition);
            this.absoluteTargetPosition = this.storagePosition;
            this.executeAction(MOVE_LOADERCARRIER_TO_STORAGE,
                    carrierMotionTimeout);
            afterMotion();
        }
    }

    /**
     * For end user. Move the carrier to absolute position given as argument (in
     * decimal format).
     *
     * @param absolutePosition
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to absolute position given as argument (in decimal format)." + cmdDesc,
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void goToAbsolutePosition(int absolutePosition) {
        this.updatePosition();
        if (position == absolutePosition) {
            FCSLOG.info(name + " is already at position " + absolutePosition);
        } else {
            loader.checkConditionsForCarrierMotion(absolutePosition);

            this.absoluteTargetPosition = absolutePosition;
            this.executeAction(MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION,
                    timeoutForGoingToStorage);
            afterMotion();
        }
    }

    private void checkAndChangeProfileVelocity() {
        boolean betweenEngagedAndHandoff = position >= engagedPosition && absoluteTargetPosition >= engagedPosition;
        long velocity = this.readProfileVelocity();
        if ((loader.isClosedOnFilter()) && betweenEngagedAndHandoff) {
            if (velocity > this.lowSpeed) {
                FCSLOG.info(name + " velocity too high with a filter in loader : slowing ProfileVelocity");
                this.slowProfileVelocity();
            }
        }
    }

    private void checkAndChangeProfileAcceleration() {
        long acceleration = carrierController.readParameter(ProfileAcceleration);
        boolean betweenEngagedAndHandoff = position >= engagedPosition && absoluteTargetPosition >= engagedPosition;
        if (loader.isClosedOnFilter() && betweenEngagedAndHandoff) {
            if (acceleration > this.lowAcceleration) {
                FCSLOG.info(name + " ProfileAcceleration too high with a filter in the loader : slowing ProfileAcceleration");
                this.slowProfileAcceleration();
            }
        }
    }

    private void checkAndChangeProfileDeceleration() {
        long deceleration = carrierController.readParameter(ProfileDeceleration);
        boolean betweenEngagedAndHandoff = position >= engagedPosition && absoluteTargetPosition >= engagedPosition;
        if (loader.isClosedOnFilter() && betweenEngagedAndHandoff) {
            if (deceleration > this.lowDeceleration) {
                FCSLOG.info(name + " ProfileDeceleration too high with a filter in the loader. I slow it.");
                this.slowProfileDeceleration();
            }
        }
        /* storage position is 0 */
        boolean fromStorageToHandoff = absoluteTargetPosition > position;
        if (loader.isEmpty() && fromStorageToHandoff) {
            if (deceleration > this.lowDeceleration) {
                FCSLOG.info(name + " ProfileDeceleration too high to go to handoff without a filter.");
                this.slowProfileDeceleration();
            }
        }
    }

    /**
     * For tests done by end user only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For tests only. Slow down profile velocity, acceleration and deceleration in loader controller.")
    public void setSlowMode() {
        slowProfileVelocity();
        slowProfileAcceleration();
        slowProfileDeceleration();
        /* for GUI */
        publishData();
    }

    /**
     * For tests done by end user only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For tests only. Raise profile velocity, acceleration and deceleration in loader controller.")
    public void setFastMode() {
        raiseProfileVelocity();
        raiseProfileAcceleration();
        raiseProfileDeceleration();
        /* for GUI */
        publishData();
    }


    /**
     * change ProfileVelocity parameter to lowSpeed
     *
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileVelocity parameter to lowSpeed")
    public void slowProfileVelocity() {
        carrierController.changeProfileVelocity(lowSpeed);
        profileVelocity = lowSpeed;
        carrierMotionTimeout = (int) (4 * timeoutForGoingToStorage);
        FCSLOG.info(name + " new ProfileVelocity=" + lowSpeed);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileVelocity parameter to highSpeed")
    public void raiseProfileVelocity() {
        carrierController.changeProfileVelocity(highSpeed);
        profileVelocity = highSpeed;
        carrierMotionTimeout = (int) timeoutForGoingToStorage;
        FCSLOG.info(name + " new ProfileVelocity=" + highSpeed);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileAcceleration "
            + "parameter to highAcceleration")
    public void raiseProfileAcceleration() {
        carrierController.writeParameter(ProfileAcceleration, highAcceleration);
        profileAcceleration = highAcceleration;
        FCSLOG.info(name + " new ProfileAcceleration=" + highAcceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileAcceleration "
            + "parameter to lowAcceleration")
    public void slowProfileAcceleration() {
        carrierController.writeParameter(ProfileAcceleration, lowAcceleration);
        profileAcceleration = lowAcceleration;
        FCSLOG.info(name + " new ProfileAcceleration=" + lowAcceleration);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileAcceleration "
            + "parameter to highAcceleration")
    public void raiseProfileDeceleration() {
        carrierController.writeParameter(ProfileDeceleration, highDeceleration);
        profileDeceleration = highDeceleration;
        FCSLOG.info(name + " new ProfileDeceleration=" + highDeceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "change ProfileAcceleration "
            + "parameter to lowAcceleration")
    public void slowProfileDeceleration() {
        carrierController.writeParameter(ProfileDeceleration, lowDeceleration);
        profileDeceleration = lowDeceleration;
        FCSLOG.info(name + " new ProfileDeceleration=" + lowDeceleration);
    }

    /**
     * read ProfileVelocity parameter in carrierController CPU. just for end user
     * facility
     *
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description = "read ProfileVelocity parameter in carrierController CPU")
    public long readProfileVelocity() {
        return carrierController.readParameter(ProfileVelocity);
    }

    /**
     * Tests if action is completed. During motion of the carrier, this methods
     * tests if the desired position has been reached. If yes, it returns true, else
     * false.
     *
     * @param action to be tested
     * @return true if required position has been reached
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        return this.position == this.absoluteTargetPosition;
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update carrier position in reading controller.")
    public void updatePosition() {
        this.position = carrierController.readPosition();
        this.publishData();
    }

    @Override
    public void updateCurrent() {
        this.current = this.carrierController.readCurrent();
    }

    /**
     * To display position for end user. Updates carrier position in reading
     * controller and returns it.
     *
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "To update and display position for end user."
                    + "Updates carrier position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carrierController.checkFault();
            position = carrierController.readPosition();
            updateCurrent();
            loader.updateStateWithSensors();
            /* read actual velocity */
            speed = carrierController.readVelocity();
            loader.checkFilterDistanceOpened();
        } catch (SDORequestException ex) {
            this.raiseWarning(SDO_ERROR, "error in updateStateWithSensorsToCheckIfActionIsCompleted: ", name, ex);
        }
    }

    /**
     * Starts an action on the carrier.
     * On the loader carrier, the action in normal mode are motion of the carrier to go to STORAGE position
     * or HANDOFF position.
     * In engineering mode, the end user can move the carrier to a given absolute position.
     * @param action to be started
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        if (!loader.isCarrierMotionAllowedByPLC()) {
            throw new RejectedCommandException(name + " PLC does not allow carrier motion. Carrier Relay is OFF. Ckeck PLCLoaderPanel.");
        }
        carrierController.checkFault();
        checkAndChangeProfileVelocity();
        checkAndChangeProfileAcceleration();
        checkAndChangeProfileDeceleration();
        carrierController.enableAndWriteAbsolutePosition(this.absoluteTargetPosition);
    }

    /**
     * Aborts an action.
     *
     * @param action
     * @param delay
     */
    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.debug(name + " ABORTING action " + action.toString() + " within delay " + delay);
        carrierController.quickStop();
        carrierController.goToSwitchOnDisabled();
    }

    /**
     * Ends an action.
     *
     * @param action
     */
    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.debug(name + " ENDING action " + action.toString());
        /*
         */
        this.carrierController.goToSwitchOnDisabled();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is STOPPING action " + action.toString() + " within delay " + delay);
        // doesn't work : we can't do a quickStop when in PROFILE_POSITION
        // this.carrierController.quickStop();
        this.carrierController.stopPosition();
    }

    /**
     * to be executed after carrier motion. abortAction has been executed prior this
     * method.
     */
    private void afterMotion() {
        checkPositionSensors();
        loader.updateFCSStateToReady();
        this.absoluteTargetPosition = 0;
    }

    /**
     * check that position sensors confirm position at the end of a motion to
     * absoluteTargetPosition.
     */
    public void checkPositionSensors() {
        if (this.absoluteTargetPosition == this.storagePosition && !this.isAtStorage()) {
            throw new FailedCommandException(name + ": storage sensors don't confirm carrier position.");

        } else if (this.absoluteTargetPosition == this.engagedPosition && !this.isAtEngaged()) {
            throw new FailedCommandException(name + ": engaged sensors don't confirm carrier position.");

        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.isAtHandoff()) {
            throw new FailedCommandException(name + ": handoff sensors don't confirm carrier position.");
        }
    }

    /**
     * Command for ENGINEERING mode. Not used during INITIALIZATION.
     * Check that carrier controller is initialized.
     * Read sensors and update state.
     * Read carrier position on controller.
     * If position sensors and position read on carrier controller give inconsistent information
     * throw a FcsHardwareException.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if hardware is ready to be started.")
    public void initializeHardware() {
        carrierController.checkInitialized();

        loader.updateStateWithSensors();
        try {
            updatePosition();
        } catch (SDORequestException ex) {
            String msg = name + ": couldn't update position";
            FCSLOG.error(msg);
            throw new FcsHardwareException(name, ex);
        }
        if (isCarrierAlmostAtPosition(this.handoffPosition) && !this.isAtHandoff()) {
            homingDone = handoffSensors.isInError();
            throw new FcsHardwareException(
                    name + ": handoff sensors don't confirm position read on carrierController.");
        }
        if (isCarrierAlmostAtPosition(this.engagedPosition) && !this.isAtEngaged()) {
            homingDone = engagedSensors.isInError();
            throw new FcsHardwareException(
                    name + ": engaged sensors don't confirm position read on carrierController.");
        }
        if (isCarrierAlmostAtPosition(this.storagePosition) && !this.isAtStorage()) {
            homingDone = storageSensors.isInError();
            throw new FcsHardwareException(
                    name + ": storage sensors don't confirm position read on carrierController.");
        }
        this.initialized = true;
        loader.updateFCSStateToReady();
    }


    /**
     * Returns true if carrier position is in a range of 2*deltaPosition around a
     * given position.
     *
     * @param position
     * @return
     */
    private boolean isCarrierAlmostAtPosition(int position) {
        return Math.abs(this.position - position) < deltaPosition;
    }

    /**
     * return true if carrier is going to HANDOFF and is approaching handoff.
     *
     * see JIRALSSTCCSFCS-344
     *
     * @return
     */
    protected boolean isCarrierApproachingHandoff() {
        return absoluteTargetPosition > handoffPosition - 5000
                && position > handoffPosition - 5000;
    }

    /**
     * Return a printed list of hardware with initialization information. For debug
     * purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return a printed list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (this.isInitialized()) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }

    /**
     * Creates an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByLoaderCarrier createStatusDataPublishedByLoaderCarrier() {
        StatusDataPublishedByLoaderCarrier status = new StatusDataPublishedByLoaderCarrier();
        status.setPosition(position);
        status.setSpeed(speed);
        status.setProfileVelocity(profileVelocity);
        status.setProfileAcceleration(profileAcceleration);
        status.setProfileDeceleration(profileDeceleration);
        status.setCurrent(current);
        status.setAtHandoff(isAtHandoff());
        status.setAtStorage(isAtStorage());
        status.setAtEngaged(isAtEngaged());
        status.setEngagedSensorOn(engagedSensors.isOn());
        status.setControllerInError(carrierController.isInError());
        status.setHomingDone(true);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderCarrier status = this.createStatusDataPublishedByLoaderCarrier();
        KeyValueData kvd = new KeyValueData(name, status);
       subs.publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (carrierController.isBooted()) {
            carrierController.goToSwitchOnDisabled();
        }
        FCSLOG.info(name + " is shutdown.");
    }
}
