
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * This is a model for the Inclinometer used in CPPM test bench and on whole
 * size prototype.
 *
 * Because for mechanical raisons, the filter changer system X axis has to be
 * horizontal in order to be able to rotate carousel or move autochanger trucks
 * along its rails. So we have 2 inclinometers which give redondant information.
 * We must read permanently the 2 inclinometers to check that inclinaison is
 * correct. We read the inclinaison on the plutoGateway.
 *
 * @author virieux
 */
public class Inclinometer implements SensorPluggedOnDevice, AlertRaiser, HasLifecycle {

    @LookupField(strategy = TOP)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    private String name = "unset";

    @ConfigurationParameter(units = "unitless", category = "sensor")
    private volatile int byteNumero;

    @ConfigurationParameter(units = "unitless", category = "sensor")
    private volatile String deviceName;

    private double inclinaison = 0;

    @Override
    public void build() {
        dataProviderDictionaryService.registerData(new KeyValueData(name, inclinaison));

        DataProviderInfo data = new DataProviderInfo("", DataProviderInfo.Type.TRENDING, name);
        data.addAttribute(DataProviderInfo.Attribute.UNITS, "degree");
        data.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "inclinaison of the AC truck");
        data.addAttribute(DataProviderInfo.Attribute.TYPE, "double");
    }
    
    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(getName()), alwaysClear);
    }
    

    /**
     *
     */
    private double voltage;

    /**
     *
     * methods overriden from interface SensorPLuggedOnDevice
     * @return
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDeviceName() {
        return deviceName;
    }

    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    @Override
    public void updateValue(int newValue) {
        FCSLOG.fine(name + " updating inclinaison with newValue read on gateway (unit=0,1Volt): " + newValue);
        voltage = newValue;
        inclinaison = convertVoltage2Degree(voltage);
        if (newValue < 0 || newValue > 50) {
            this.raiseWarning(FcsAlert.HARDWARE_ERROR,
                    newValue + ": bad value read on inclinometer. Value read has to be between 0 and 50 (unit = 0,1Volts).",
                    getName());
        }
    }

    /**
     *
     * end of methods overriden from interface SensorPLuggedOnDevice
     */

    /**
     * Returns inclinaison value in degrees. Doesn't read again plutoGateway.
     *
     * @return inclinaison
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns inclinaison value in degrees. Doesn't read again CANopen device.")
    public double getInclinaison() {
        return inclinaison;
    }

    /**
     *
     * @return voltage read on AC PlutoGateway
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return raw voltage read on AC PlutoGateway unit is 0.1 Volt")
    public double getVoltage() {
        return voltage;
    }
    
    /**
     * Convert a voltage unit = 0.1volt to degrees.
     *
     * @param voltage
     * @return angle in degrees or NaN if voltage value is not correct.
     */
    private double convertVoltage2Degree(double voltage) {
        // voltage unit= 0.1Volt
        double voltageInVolt = voltage / 10;
        double sin_angle = (voltageInVolt - 2.5) / 11.52;
        // if |sin_angle| > 1 this returns NaN.
        return Math.toDegrees(Math.asin(sin_angle));
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

}
