package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is the Main Module for the Loader control software when the loader is in
 * standalone.
 *
 * @author virieux
 */
public class LoaderMain extends MainModule {

    @LookupField(strategy = Strategy.CHILDREN, pathFilter = "loader")  
    private Loader loader;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter = "autochanger")
    private FilterHolder autochanger;

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make LoaderGUI listen to my
         * subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.loader", "loader");
        super.init();
    }

    /**
     * Update loader state in reading the sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update loader state in reading the sensors.")
    @Override
    public void updateStateWithSensors() {
        loader.updateStateWithSensors();
        autochanger.updateStateWithSensors();
    }


    @Override
    public void initializeHardware() {
        loader.initializeHardware();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Connect the loader hardware.")
    public void connectLoader() {
        this.bridge.connectHardware();
        loader.postStart();
    }

    /**
     * In Standalone mode, we want to connect Loader when the subsystem starts.
     */
    @Override
    public void postStart() {
        connectLoader();
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Print the list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(this.bridge.printHardwareState());
        sb.append("\n ");
        sb.append(loader.printHardwareState());
        return sb.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n").append(loader.toString());
        return sb.toString();
    }

}
