package org.lsst.ccs.subsystems.fcs;

import java.util.HashMap;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.SWITCH_ON_DISABLED;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.CLAMPEDONFILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION;

import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.BrakeState;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.IN_FAULT;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.PT100Interface;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.errors.ControllerFaultException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console.
 *
 *
 * @author virieux
 *
 */
public class Carousel extends MobileItem implements FilterHolder, AlertRaiser {

    private static final long serialVersionUID = -2376279469784152348L;

    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = TREE)
    private PersistencyService persistenceService;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private int rotationTimeout;

    private long profileVelocity;
    private long profileAcceleration;
    private long profileDeceleration;

    /**
     * carousel position given by carouselControl.
     */
    protected int position = 0;
    private int relativeTargetPosition;

    /**
     * an absolute target position to reach in a rotation is modified each time a
     * rotation command has been launched.
     */
    protected int absoluteTargetPosition;
    
    
    /**
     * startPosition is the absolute position at the begining of a rotation.
     * used to compute estimated position during rotation : during rotation, the carousel controller is set 
     * in encoder type incremental so the position read on controller is an incremental position
     * not an absolute position.
     * 
     * 
     */
    protected int startPosition;

    protected int readVelocity = 0;
    protected int readCurrent = 0;

    @ConfigurationParameter(description = "below this value of deltaPosition, carousel position at STANDBY is correct", units = "Carousel encoder steps", range = "0..2181120", category = "carousel")
    private volatile int minStandbyDeltaPosition = 100;

    @ConfigurationParameter(description = "over this value of deltaPosition, carousel position at STANDBY is NOT correct", units = "Carousel encoder steps", range = "0..2181120", category = "carousel")
    private volatile int maxStandbyDeltaPosition = 130;

    /**
     * 0.1 tour = 436224 pas (2Pi = 4260 X 1024) 1 socket = 0.2 tour = 1/5 tour =
     * 872448 pas
     */
    private final int fullTurn = 4362240;
    private final int halfTurn = fullTurn / 2;

    @ConfigurationParameter(description = "velocity in slow mode", units = "Carousel velocity unit", range = "0..5000", category = "carousel")
    public volatile int slowVelocity = 500;

    @ConfigurationParameter(description = "acceleration in slow mode", units = "Carousel acceleration unit", range = "0..5000", category = "carousel")
    public volatile int slowAcceleration = 200;

    @ConfigurationParameter(description = "deceleration in slow mode", units = "Carousel acceleration unit", range = "0..5000", category = "carousel")
    public volatile int slowDeceleration = 200;

    @ConfigurationParameter(description = "in milliseconds; timeout for the rotation in slow mode", units = "milliseconds", range = "0..300000", category = "carousel")
    public volatile int slowRotationTimeout = 100000;

    @ConfigurationParameter(description = "velocity in fast mode", units = "Carousel velocity unit", range = "0..5000", category = "carousel")
    public volatile int fastVelocity = 3400;

    @ConfigurationParameter(description = "acceleration in fast mode", units = "Carousel acceleration unit", range = "0..5000", category = "carousel")
    public volatile int fastAcceleration = 2000;

    @ConfigurationParameter(description = "deceleration in fast mode", units = "Carousel acceleration unit", range = "0..5000", category = "carousel")
    public volatile int fastDeceleration = 1000;

    @ConfigurationParameter(description = "in milliseconds; timeout for the rotation in fast mode", units = "milliseconds", range = "0..300000", category = "carousel")
    public volatile int fastRotationTimeout = 20000;

    @ConfigurationParameter(description = "This is the time to wait until protection system changes its status after motions.", units = "milliseconds", range = "0..5000", category = "carousel")
    public volatile long timeToUpdateProtectionSystem = 2000;

    @ConfigurationParameter(description = "for command unlock of carousel clamp : current to send to prepare unlock", range = "-3200..3200", units = "mA", category = "carousel")
    protected volatile int currentToPrepareUnlock = -100;

    @ConfigurationParameter(description = "for command unlock of carousel clamp : time between little current to "
            + "prepare hardware and currentToLock ", range = "0..500", units = "milliseconds", category = "carousel")
    protected volatile int timeToPrepareUnlock = 200;

    @ConfigurationParameter(description = "A current to be sent to clampXminus controller during locking recovery.", units = "mV", range = "0..1000", category = "carousel")
    public volatile int recoveryLockingCurrent = 400;

    @ConfigurationParameter(description = "If the velocity in clampXminus controller goes over this value during locking recovery, "
            + "it means that the recovery process has failed.", units = "rpm", range = "0..100", category = "carousel")
    public volatile int recoveryMaxVelocity = 60;

    @ConfigurationParameter(description = "A number of steps to go back if after rotation carousel "
            + "position has exceeded standbyPosition by more than maxStandbyDeltaPosition.", units = "Carousel encoder steps", range = "0..2181120", category = "carousel")
    public volatile int recoveryBackwardStep = 10000;

    @ConfigurationParameter(description = "A number of steps to go back if after rotation carousel "
            + "position has exceeded standbyPosition by more than maxStandbyDeltaPosition.", units = "Carousel encoder steps", range = "0..2181120", category = "carousel")
    public volatile int recoveryForwardStep = 150000;
    
    @ConfigurationParameter(description = "upper brake1 limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake1Limit = 19033;

    @ConfigurationParameter(description = "upper brake2 limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake2Limit = 16959;

    @ConfigurationParameter(description = "upper brake3 limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake3Limit = 21063;

    private static int BRAKE_NO_SENSOR_LIMIT = 500;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/clampXminusController")
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/clampXplusController")
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/carouselController")
    protected EPOSControllerForCarousel carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/hyttc580")
    private TTC580Interface hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/ai814")
    protected ADCInterface ai814;

    @LookupField(strategy = TREE, pathFilter = ".*\\/pt100")
    private PT100Interface pt100;

    /* To be able to know if the autochanger holds a filter. */
    @LookupField(strategy = SIBLINGS, pathFilter = "autochanger")
    private FilterHolder autochanger;

    /**
     * A map to store the sockets by their names. The key of this map is the socket
     * name. This map is built by Toolkit during INITIALISATION.
     *
     */
    @LookupField(strategy = CHILDREN)
    protected final Map<String, CarouselSocket> socketsMap = new TreeMap<>();
    
    /**
     * map of sensors attached to this device.
     */
    @LookupField(strategy = TREE)
    protected Map<String, SensorPluggedOnTTC580> sensorsMap = new HashMap<>();

    @LookupField(strategy = SIBLINGS, pathFilter = "tcpProxy")
    protected BridgeToHardware tcpProxy;

    private boolean initialized = false;
    protected boolean clampsStateInitialized = false;

    /**
     * value of field socketAtStandbyID is read on httc580
     */
    private int socketAtStandbyID;
    protected CarouselSocket socketAtStandby;

    /**
     * Threshold that gives the 'locked' status to the clamps. The values of the
     * Baumer sensors, which are used to determine the locking status of the
     * carousel clamps, depend strongly on ambient temperature. The TTC580
     * processes temperature sensor values from the TTC30 modules and uses this
     * information to correct the threshold that gives the 'locked' status to
     * the clamps. This value (in mV) is read through the CAN bus from an SDO,
     * by sending a read request at index 0x6404, subindex 5.
     */
    protected volatile int minLockedThreshold = 9500;

    @ConfigurationParameter(description = "If lockSensor side X- returns a value < lockSensorMinLimitXminus, "
            + "sensor is in ERROR.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMinLimitXminus = 201;

    @ConfigurationParameter(description = "If lockSensor side X+ returns a value < lockSensorMinLimitXplus, "
            + "sensor is in ERROR.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMinLimitXplus = 202;

    @ConfigurationParameter(description = "If lockSensor side X- returns a value above lockSensorMaxLimitXminus, "
            + "the sensor is in error.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMaxLimitXminus = 10001;

    @ConfigurationParameter(description = "If lockSensor side X+ returns a value above lockSensorMaxLimitXplus, "
            + "the sensor is in error.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMaxLimitXplus = 11002;

    @ConfigurationParameter(range = "0..12000", units = "mV", category = "carousel")
    private volatile int filterPresenceMinNoFilter = 9200;

    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarousel.class, name);
        registerAction(ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION);
        registerAction(ROTATE_CAROUSEL_TO_RELATIVE_POSITION);
    }

    @Override
    public void init() {
	super.init();
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(HARDWARE_ERROR.getAlert(carouselController.getName()), alwaysClear);
        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(IN_FAULT.getAlert(name), alwaysClear);
    }

    
    /**
     *
     * @return true if CANopen devices are booted and initialized and homing has
     *         been done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        return this.myDevicesReady();
    }

    /**
     *
     * @return true if clamp state is initialized for all clamps
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if clamp state is initialized for all clamps.")
    public boolean isClampsStateInitialized() {
        return clampsStateInitialized;
    }

    /* used by carousel clamp */
    public int getCurrentToPrepareUnlock() {
        return currentToPrepareUnlock;
    }

    /* used by carousel clamp */
    public int getTimeToPrepareUnlock() {
        return timeToPrepareUnlock;
    }

    /* used by carousel clamp */
    public Integer getMinLockedThreshold() {
        return minLockedThreshold;
    }

    public int getLockSensorMinLimitXminus() {
        return lockSensorMinLimitXminus;
    }

    public int getLockSensorMinLimitXplus() {
        return lockSensorMinLimitXplus;
    }

    public int getLockSensorMaxLimitXminus() {
        return lockSensorMaxLimitXminus;
    }

    public int getLockSensorMaxLimitXplus() {
        return lockSensorMaxLimitXplus;
    }

    public int getFilterPresenceMinNoFilter() {
        return filterPresenceMinNoFilter;
    }

    /* used by carousel clamp */
    public int getRecoveryLockingCurrent() {
        return recoveryLockingCurrent;
    }

    /* used by carousel clamp */
    public int getRecoveryMaxVelocity() {
        return recoveryMaxVelocity;
    }

    /**
     * used by tests.
     *
     * @return
     */
    public int getFullTurn() {
        return fullTurn;
    }

    public Map<String, CarouselSocket> getSocketsMap() {
        return socketsMap;
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return carousel position.", alias = "printPosition")
    public int getPosition() {
        return position;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    /**
     * This method returns the clampX- which is at standby filterPosition. It can
     * returns null if there is no socketAtStandby halted at standby filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXminus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It can
     * returns null if there is no socketAtStandby halted at standby filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXplus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     *         otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")
    public boolean isAtStandby() {
        // TODO this is not consistent with method CarouselSocket.isAtStandby();
        return socketAtStandbyID >= 1 && socketAtStandbyID <= 5;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one. Otherwise
     * return null.
     *
     * @return
     */
    public CarouselSocket getSocketAtStandby() {
        return socketAtStandby;
    }

    /**
     *
     * @return ID of socket at STANDBY
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "return ID of socket at STANDBY, 0 if carousel is not stopped at STANDBY")
    public int getSocketAtStandbyID() {
        return socketAtStandbyID;
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns name of filter at STANDBY position  or"
            + " NO_FILTER if carousel is not at STANDBY or there is no filter at STANDBY.")
    public String getFilterAtStandbyName() {
        if (socketAtStandby == null || this.isEmptyAtStandby()) {
            return NO_FILTER;
        } else {
            return socketAtStandby.getFilterName();
        }
    }

    /**
     * Returns ID of filter which is in the socket at STANDBY position or 0 if there
     * is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns id of filter at STANDBY position  or"
            + " 0 if there is no filter at STANDBY.")
    public int getFilterIDatStandby() {
        if (socketAtStandby == null) {
            return 0;
        } else {
            return socketAtStandby.getFilterID();
        }
    }

    /**
     * return true if filterID in on carousel.
     *
     * @param filterID
     * @return
     */
    public boolean isFilterOnCarousel(int filterID) {
        return socketsMap.values().stream().anyMatch((socket) -> (socket.getFilterID() == filterID));
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To change filterID on socket which ID is given as argument.")
    public void changeFilterID(int filterID, int socketID) {
        if (socketID < 1 || socketID > 5) {
            throw new IllegalArgumentException(socketID + ": bad value - enter a digit between 1 and 5");
        }
        if (this.isFilterOnCarousel(filterID)) {
            int sockID = this.getFilterSocket(filterID).getId();
            throw new IllegalArgumentException(filterID + " filter already on carousel on socket" + sockID);
        }
        String socketName = SOCKET_NAME + socketID;
        socketsMap.get(socketName).setFilterID(filterID);
    }

    /**
     * This methods returns true if the autochanger is holding a filter at STANDBY
     * position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if Autochanger is holding filter at STANDBY.")
    public boolean isAutochangerHoldingFilter() {
        /**
         * To be done when a switch is connected : read the switch sensor and update a
         * boolean of the class like : autochangerHoldingFilter.
         */
        autochanger.updateStateWithSensors();
        return autochanger.isHoldingFilter();
    }

    /**
     * initialize carousel hardware after initialization. to be executed if during
     * boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Initialize carousel hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        tcpProxy.bootProcess();
        this.postStart();
    }

    /**
     * Executed when all components HasLifeCycle of subsystem have been checked.
     */
    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart.");
        if (carouselController.isBooted()) {
            initializeRotationController();
            profileVelocity = carouselController.readProfileVelocity();
            profileAcceleration = carouselController.readParameter(ProfileAcceleration);
            profileDeceleration = carouselController.readParameter(ProfileDeceleration);
            if (profileVelocity == slowVelocity) {
                rotationTimeout = slowRotationTimeout;
            } else {
                rotationTimeout = fastRotationTimeout;
            }
            /**
             * Try to update carousel position and if carouselController is in
             * fault catch exception to let subsystem starts nevertheless.
             */
            try {
                updatePosition();
            } catch (ControllerFaultException ex) {
                FCSLOG.error(ex.toString());
                this.raiseAlarm(IN_FAULT, ex.toString(), name);
            }
        }
        if (clampXminusController.isBooted()) {
            initializeClampController(clampXminusController);
        }
        if (clampXplusController.isBooted()) {
            initializeClampController(clampXplusController);
        }
        if (hyttc580.isBooted()) {
            initializeClampsState();
        }
        FCSLOG.info(name + " END postStart.");
    }

    private void initializeRotationController() {
        try {
            carouselController.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", carouselController.getName(), ex);
        }
    }

    /**
     * check that controller is correctly configured. This command can't be executed
     * in CarouselClamp.postStart because the 5 clamps Xminus share the same
     * controller and idem for Xplus.
     */
    private void initializeClampController(EPOSController controller) {
        try {
            /*
             * check that parameters on CPU are those on configuration
             */
            controller.initializeAndCheckHardware();
            if (!controller.isInMode(CURRENT)) {
                this.raiseAlarm(HARDWARE_ERROR, "  is not in CURRENT mode.", controller.getName());
            }

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller ", controller.getName(), ex);
        }
    }


    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Disable carousel controller, set the position sensor to TypeEncoderSSI, and check that the ssi position is correct.")
    public void setControllerPositionSensorTypeEncoderSSI() {
        carouselController.setPositionSensorTypeEncoderSSI();
        position = carouselController.readPosition();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "slow down profile velocity, acceleration and deceleration in carousel controller.")
    public void setSlowMode() {
        carouselController.changeProfileVelocity(slowVelocity);
        profileVelocity = slowVelocity;
        carouselController.writeParameter(ProfileAcceleration, slowAcceleration);
        profileAcceleration = slowAcceleration;
        carouselController.writeParameter(ProfileDeceleration, slowDeceleration);
        profileDeceleration = slowDeceleration;
        rotationTimeout = slowRotationTimeout;
        publishData();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "raise profile velocity, acceleration and deceleration in carousel controller.")
    public void setFastMode() {
        carouselController.changeProfileVelocity(fastVelocity);
        profileVelocity = fastVelocity;
        carouselController.writeParameter(ProfileAcceleration, fastAcceleration);
        profileAcceleration = fastAcceleration;
        carouselController.writeParameter(ProfileDeceleration, fastDeceleration);
        profileDeceleration = fastDeceleration;
        rotationTimeout = fastRotationTimeout;
        publishData();
    }

    public void waitForProtectionSystemUpdate() {
        waitForStateUnclampedOnFilter(timeToUpdateProtectionSystem);
    }

    /**
     * wait until carousel is unclamped and empty at standby
     *
     * @param timeout after this delay don't wait anymore.
     */
    public void waitForStateUnclampedOnFilter(long timeout) {

        FcsUtils.waitCondition(() -> isUnclampedOnFilterAtStandby(), () -> updateSocketAtStandbyWithSensors(),
                "waitForStateUnclampedOnFilter", timeout);

    }


    public void checkSensorTypeIncremental() {
        int sensorType = carouselController.readPositionSensorType();
        if (sensorType != 8) {
            throw new FcsHardwareException(
                    name + " PositionSensorType has to be set to Incremental Encoder before motion.");
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "save persist file to store which filter is in socket and clamps offsets")
    public void persistData() {
        persistenceService.persistNow();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByCarousel()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the STATUS
     * bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setPosition(position);
        status.setPositionSensorType(carouselController.getPositionSensorType());
        // during rotation positionSensorType is incremental so the position read on the controller is the incrementalPosition.
        status.setEstimatedPosition((startPosition + position) % fullTurn);
        status.setCurrent(readCurrent);
        status.setVelocity(readVelocity);
        status.setAtStandby(isAtStandby());
        status.setMoving(moving);
        status.setSocketAtStandbyID(socketAtStandbyID);
        // if carousel is empty at STANDBY getFilterAtStandbyName() returns NO_FILTER
        status.setFilterAtStandbyName(getFilterAtStandbyName());
        if (isAtStandby()) {
            status.setSocketAtStandbyName(socketAtStandby.getName());
            status.setEmptyAtStandby(this.isEmptyAtStandby());
            status.setDeltaPositionAtStandby(socketAtStandby.getDeltaPosition());
            status.setClampsStateAtStandby(socketAtStandby.getClampsState());
            status.setSlaveStatusAtStandby(socketAtStandby.getSlaveModuleStatus());
        } else if (socketAtStandbyID == 0) {
            status.setSocketAtStandbyName("NO_SOCKET_AT_STANDBY");
        } else if (socketAtStandbyID == 7) {
            status.setSocketAtStandbyName("ERROR_READING_ID");
        }
        status.setProfileAcceleration(profileAcceleration);
        status.setProfileDeceleration(profileDeceleration);
        status.setProfileVelocity(profileVelocity);
        status.setRotationTimeout(rotationTimeout);
        status.setTemperature1(pt100.getTemperature(1));
        status.setTemperature2(pt100.getTemperature(2));
        status.setTemperature3(pt100.getTemperature(3));
        status.setTemperature4(pt100.getTemperature(4));
        status.setSensor1(ai814.getInput(0));
        status.setSensor2(ai814.getInput(1));
        status.setSensor3(ai814.getInput(2));
        status.setBrakeState1(computeBrakeState(brake1Limit, ai814.getInput(0)));
        status.setBrakeState2(computeBrakeState(brake2Limit, ai814.getInput(1)));
        status.setBrakeState3(computeBrakeState(brake3Limit, ai814.getInput(2)));
        status.setMinLocked(minLockedThreshold);
        /* PLC status for PCL debug */
        status.setOUT_CS(sensorsMap.get("carousel/caCS").getValue() == 1);
        status.setOUT_CS_InError(sensorsMap.get("carousel/caCS").getValue() == sensorsMap.get("carousel/caCSb").getValue());
        status.setOUT_CFC(sensorsMap.get("carousel/caCFC").getValue() == 1);
        status.setOUT_CFC_InError(sensorsMap.get("carousel/caCFC").getValue() == sensorsMap.get("carousel/caCFCb").getValue());
        status.setOUT_CF1(sensorsMap.get("carousel/caCF1").getValue() == 1);
        status.setOUT_CF1_InError(sensorsMap.get("carousel/caCF1").getValue() == sensorsMap.get("carousel/caCF1b").getValue());
        status.setOUT_CF0(sensorsMap.get("carousel/caCF0").getValue() == 1);
        status.setOUT_CF0_InError(sensorsMap.get("carousel/caCF0").getValue() == sensorsMap.get("carousel/caCF0b").getValue());
        status.setEnableBrakes(sensorsMap.get("carousel/caEnableBrakes").getValue() == 1);
        status.setEnableRotation(sensorsMap.get("carousel/caEnableRotation").getValue() == 1);
        status.setEnableUnclamp(sensorsMap.get("carousel/caEnableUnclamp").getValue() == 1);
        status.setEnableShutter(sensorsMap.get("carousel/caEnableShutter").getValue() == 1);
        status.setAp1(sensorsMap.get("carousel/caAP1").getValue() == 1);
        status.setAp1_InError(sensorsMap.get("carousel/caAP1").getValue() == sensorsMap.get("carousel/caAP1b").getValue());
        status.setAp2(sensorsMap.get("carousel/caAP2").getValue() == 1);
        status.setAp2_InError(sensorsMap.get("carousel/caAP2").getValue() == sensorsMap.get("carousel/caAP2b").getValue());
        status.setAp3(sensorsMap.get("carousel/caAP3").getValue() == 1);
        status.setAp3_InError(sensorsMap.get("carousel/caAP3").getValue() == sensorsMap.get("carousel/caAP3b").getValue());
        status.setAf3(sensorsMap.get("carousel/caAF3").getValue() == 1);
        status.setAf3_InError(sensorsMap.get("carousel/caAF3").getValue() == sensorsMap.get("carousel/caAF3b").getValue());
        status.setTpStopUnclamp(sensorsMap.get("carousel/tpStopUnclamp").getValue() == 1);
        status.setTpStopRotation(sensorsMap.get("carousel/tpStopRotation").getValue() == 1);
        status.setTpCheckRotation(sensorsMap.get("carousel/tpCheckRotation").getValue() == 1);
        status.setCaLockout(sensorsMap.get("carousel/caLockout").getValue() == 1);
        status.setCaLockout_InError(sensorsMap.get("carousel/caLockout").getValue() == sensorsMap.get("carousel/caLockoutb").getValue());
        status.setCaEng(sensorsMap.get("carousel/caEng").getValue() == 1);
        status.setCaEng_InError(sensorsMap.get("carousel/caEng").getValue() == sensorsMap.get("carousel/caEngb").getValue());
        status.setShutterInactive(sensorsMap.get("carousel/caShutterInactive").getValue() == 1);
        status.setSleep(sensorsMap.get("carousel/caSleep").getValue() == 1);
        status.setBrakesActivated(sensorsMap.get("carousel/caBrakesActivated").getValue() == 1);
        return status;
    }

    private static BrakeState computeBrakeState(int limit, long sensorValue) {
        if (sensorValue > limit) {
            return BrakeState.CLOSED;
        } else if (sensorValue > BRAKE_NO_SENSOR_LIMIT) {
            return BrakeState.NOBRAKE;
        } else {
            return BrakeState.NO_SENSOR;
        }
    }
    
    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the filterPosition
     *         when one of its sockets is at standby filterPosition and this
     *         socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() {

        if (this.isMoving()) {
            return false;
        }
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and carousel is
     *         stopped
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() {
        if (this.isMoving()) {
            return false;
        }
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Returns true if carousel is rotating
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns false if carousel controller is SWITCH_ON_DISABLED.")
    public boolean isRotating() {
        return !carouselController.isInState(SWITCH_ON_DISABLED);
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller. Used at startup and by end user for debug.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Set position sensor type SSI and update carousel position in reading controller.")
    public void updatePosition() {
        this.setControllerPositionSensorTypeEncoderSSI();
        try {
            this.position = carouselController.readPosition();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * Updates the field readCurrent in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update field current in reading controller.")
    public void updateCurrent() {
        try {
            this.readCurrent = carouselController.readCurrent();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() {

        this.tcpProxy.updatePDOData();
        PDOData pdoStore = tcpProxy.getPDOData();

        FCSLOG.finest(name + ":pdoStore=" + pdoStore.toString());
        socketsMap.values().stream().forEach(socket -> {
            socket.updateState();
        });
    }

    /**
     * In carousel, sensors are updated from PDOs. PDOs are received 2 by 2 : 0x180
     * + ttc580 nodeID for clamp at standby. (ttc580 pdo1) 0x280 + ttc580 nodeID for
     * a clamp not at standby. (ttc580 pdo2) PDO2 contains socketID not at STANDBY
     * in turns. Exemple : if socket1 is at STANDBY - first sync returns pdo1 with
     * socket1 values and pdo2 with socket2 values - second sync returns pdo1 with
     * socket1 values and pdo2 with socket3 values - third sync returns pdo1 with
     * socket1 values and pdo2 with socket4 values - fourth sync returns pdo1 with
     * socket1 values and pdo2 with socket5 values
     *
     * socketID is coded in each PDO. After FCS start we need to send 4 sync to know
     * all clamps state.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamps state in sending 5 sync.")
    public void initializeClampsState() {
        FCSLOG.info(name + " Initializing clamps state....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        byte count = 1;
        while (count <= 5) {
            FCSLOG.info(name + " sync no " + count);
            tcpProxy.updatePDOData();
            hyttc580.updateFromPDO(tcpProxy.getPDOData());
            updateState();
            count++;
        }
        clampsStateInitialized = true;
        publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamp offsets.")
    public void initializeAndCheckClampsOffset() {
        FCSLOG.info(name + " Initializing clamps offsets....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        socketsMap.values().stream().forEach(socket -> {
            byte socketId = (byte) socket.getId();
            long offset2 = hyttc580.readOffset2SDO(socketId);
            socket.checkAndUpdateOffset2(offset2);
            long offset1 = hyttc580.readOffset1SDO(socketId);
            socket.checkAndUpdateOffset1(offset1);
            socket.getClampXminus().publishData();
            socket.getClampXplus().publishData();
        });
        persistData();
    }

    /**
     *
     * @param filterID
     * @return null if filter is not on carousel, else return socket where filter is
     *         stored.
     */
    public CarouselSocket getFilterSocket(int filterID) {
        CarouselSocket socket = null;
        for (CarouselSocket sock : socketsMap.values()) {
            if (sock.getFilterID() == filterID) {
                return sock;
            }
        }
        return socket;
    }

    /**
     * Release clamps at standby position to get ready to clamp again.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Release clamps at standby position to get ready to clamp again")
    public void releaseClamps() {
        updateStateWithSensors();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't release clamps when no socket is halted at standby position.");
        } else {
            socketAtStandby.releaseClamps();
        }
    }

    /**
     * Unlocks the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Unlock the clamps at STANDBY.")
    public void unlockClamps() {
        updateSocketAtStandbyWithSensors();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't unlock clamps while a socket is not halted at standby position.");
        } else {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        }
    }

    /**
     * This is a recovery command when storing a carousel at STANDBY : sometimes,
     * clampXplus is LOCKED but clampXminus is not LOCKED. This command sends a
     * little current in clampXminus controller to complete the locking.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Lock clampXminus when clampXplus is already locked.")
    public void recoveryLockingXminus() {
        updateStateWithSensors();
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't use command recoveryLockingXminus.");
        }
        if (socketAtStandby.getClampXplus().getClampState() == CLAMPEDONFILTER
                && socketAtStandby.getClampXminus().getClampState() != CLAMPEDONFILTER) {
            socketAtStandby.getClampXminus().recoveryLocking();
        }
    }

    /**
     * Return true if the filterID given as argument is at STANDBY position.
     *
     * @param filterID
     * @return
     */
    public boolean isAtStandby(int filterID) {
        return socketAtStandby.getFilterID() != 0 && socketAtStandby.getFilterID() == filterID;
    }

    public void releaseClampsContact() {
        // TODO;
    }

    @Override
    public boolean myDevicesReady() {
        return carouselController.isBooted() && carouselController.isInitialized();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ ROTATION COMMANDS *********************************
     */
    /**
     * ****************************************************************
     */
    /**
     * Check if carousel rotation is permitted.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation() {
        String message;
        if (!initialized) {
            throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        }

        // commented out in July 2017 because no TTC-580
        // updateState();
        if (this.isAtStandby() && socketAtStandby.isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }

        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochanger is not at HANDOFF position.");
        }
    }


    /**
     * Rotate to position newPos. Condition LPM to rotate : AP2 & AP3. This command
     * doesn't do homing but : 1) set sensor type SSI encoder and read position
     * given by SSI encoder : positionAbsolue 2) compute diffPos == the diff between
     * newPos and positionAbsolue 3) set sensor type incremental 4)set
     * absoluteTargetPosition to diffPos. 5) Then rotate to absoluteTargetPosition %
     * fullTurn.
     *
     * @param newPos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Rotate carousel to a new absolute position.", timeout = 60000)
    public void rotateToAbsolutePosition(int newPos) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("rotateToAbsolutePosition")) {
            /* read temperatures before rotation */
            if (pt100.isBooted()) {
                pt100.updateTemperatures();
            }

            /* subsystem must not be in ALARM state */
            checkReadyForAction();

            /* carouselController should be initialized */
            carouselController.checkInitialized();

            /* carouselController should not be in fault */
            carouselController.checkFault();

            /* autochanger must be at ONLINE or HANDOFF */
            if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
                throw new RejectedCommandException(
                        name + " can rotate only if autochanger is at Handoff or at Online  ");
            }

            /* read position given by SSI encoder */
            carouselController.setPositionSensorTypeEncoderSSI();

            int absolutePosition = carouselController.readPosition();
            startPosition = absolutePosition;

            /* set sensor type incremental */
            carouselController.setPositionSensorTypeSinusIncrementalEncoder();
            int incrementalPosition = carouselController.readPosition();
            // incrementalPosition should be 0 or very near 0
            FCSLOG.info(
                    name + " absolutePosition = " + absolutePosition + " incrementalPosition = " + incrementalPosition);

            int diffPos = newPos - absolutePosition;
            if (Math.abs(diffPos) <= halfTurn) {
                absoluteTargetPosition = diffPos + incrementalPosition;
            } else {
                if (diffPos < 0) {
                    absoluteTargetPosition = diffPos + incrementalPosition + fullTurn;
                } else {
                    absoluteTargetPosition = diffPos + incrementalPosition - fullTurn;
                }
            }

            FCSLOG.info(name + " is at incremental position: " + incrementalPosition
                    + "; about to rotate to incremental position" + absoluteTargetPosition);

            // rotate the carousel
            this.executeAction(ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION, rotationTimeout);
            /* in endAction a setControllerPositionSensorTypeEncoderSSI() is done */
        }
    }

    /**
     * rotate carousel within a relative position given as argument. If carousel is
     * at initial position initPos, if the argument given to this method is
     * relativePos, at the end of the motion, carousel position should be initPos +
     * relativePos.
     *
     * @param relativePos relative position.
     * @param timeout
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Rotate carousel to a relative position.", timeout = 60000)
    public void rotateToRelativePosition(int relativePos, long timeout) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("rotateToRelativePosition")) {
            /* carouselController should be initialized */
            carouselController.checkInitialized();

            /* carouselController should not be in fault */
            carouselController.checkFault();

            /* autochanger must be at ONLINE or HANDOFF */
            if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
                throw new RejectedCommandException(
                        name + " can rotate only if autochanger is at Handoff or at Online.");
            }

            /* set sensor type incremental */
            carouselController.setPositionSensorTypeSinusIncrementalEncoder();
            int incrementalPosition = carouselController.readPosition();

            relativeTargetPosition = relativePos + incrementalPosition;
            // rotate the carousel
            this.executeAction(ROTATE_CAROUSEL_TO_RELATIVE_POSITION, timeout);
            /*
             * switch back controller to PositionSensorTypeEncoderSSI is done is endAction
             */
        }
    }

    /**
     * Rotate carousel to move a socket which name is given as argument to STANDBY
     * position. This methods computes the shortest way to go to STANDBY position.
     *
     * @param socketName
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move a socket which name is given as argument to STANDBY position.", alias = "moveSocketToStandby", timeout = 50000)
    public void rotateSocketToStandby(String socketName) {
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socketToMove = this.socketsMap.get(socketName);
        if (!socketToMove.isAtStandby()) {
            int requiredPosition = socketToMove.getStandbyPosition();
            FCSLOG.info(name + " is at position: " + position + "; about to rotate to position: " + requiredPosition);
            rotateToAbsolutePosition(requiredPosition);
            checkDeltaPosition();
        }
        FCSLOG.info(name + ":" + socketName + " is at STANDBY position on carousel.");
    }

    public void checkDeltaPosition() {
        if (socketAtStandby != null) {
            socketAtStandby.updateDeltaPosition();
            long deltaPos = socketAtStandby.getDeltaPosition();
            if (Math.abs(deltaPos) > maxStandbyDeltaPosition) {
                this.raiseAlarm(HARDWARE_ERROR, " delta position at standby is over " + maxStandbyDeltaPosition, name);
            } else if (Math.abs(deltaPos) > minStandbyDeltaPosition) {
                this.raiseWarning(HARDWARE_ERROR, " delta position at standby is over " + minStandbyDeltaPosition,
                        name);
            }
        }
    }

    /**
     * Rotate carousel full turn. A number of turns can be given as argument. This
     * turns carousel in relative position.
     *
     * @param nbTurn number of turns to rotate
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Rotate carousel full turn. A number of turns can be given as argument.")
    public void rotateFullTurn(int nbTurn) {
        if (Math.abs(nbTurn) > 3) {
            throw new IllegalArgumentException("nbTurn=" + nbTurn + " should not be more than 3");
        }
        // slowRotationTimeout is the time maximum to rotate 2 sockets at slow speed;
        // for a full turn (5 sockets), 5/2 should be the right factor, but experience
        // showed that 2 is enough.
        long timeout = Math.abs(nbTurn) * 2 * slowRotationTimeout;
        int rpos = fullTurn * nbTurn;
        rotateToRelativePosition(rpos, timeout);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To be executed when at the end of rotation, carousel position"
            + "exceeded position by more than standbyPosition + deltaPositionMax (value around 200).")
    public void recoveryRotationBackward() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " no socket at STANBY method recoveryRotationBackward is not relevant.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        int delta_sign = deltaPositionAtStandby >= 0 ? 1 : -1;
        int relativePosition = delta_sign * recoveryBackwardStep;
        /* relative rotation : go back for recoveryBackwardStep steps */
        FCSLOG.info(name + " about to rotateToRelativePosition " + relativePosition);
        rotateToRelativePosition(relativePosition, 5000);
        /*absolute rotation : go to target position - deltaPositionAtStandby */
        int newPos = socketAtStandby.getStandbyPosition() - deltaPositionAtStandby;
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To be executed when at the end of rotation, carousel position"
            + "exceeded position by more than standbyPosition + deltaPositionMax (value around 200).")
    public void recoveryRotationForward() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " no socket at STANBY method recoveryRotationForward is not relevant.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        int delta_sign = deltaPositionAtStandby >= 0 ? 1 : -1;
        int relativePosition = -delta_sign * recoveryForwardStep;
        /* relative rotation - delta_sign * recoveryBackwardStep */
        FCSLOG.info(name + " about to rotateToRelativePosition " + relativePosition);
        rotateToRelativePosition(relativePosition, 5000);
        int newPos = socketAtStandby.getStandbyPosition();
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To be executed when at the end of rotation, carousel position"
            + "exceeded position by more than standbyPosition + deltaPositionMax (value around 200).")
    public void recoveryRotationStraight() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " no socket at STANBY method recoveryRotationForward is not relevant.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        int newPos = socketAtStandby.getStandbyPosition() - deltaPositionAtStandby;
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            return carouselController.isTargetReached();

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            return carouselController.isTargetReached();
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        // updateStateWithSensors updates also position, readCurrent, readVelocity,
        // etc...
        updateStateWithSensorsDuringMotion();
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or
     * ROTATE_CAROUSEL_TO_RELATIVE_POSITION. This : - enable controller, - make
     * controller go to required position previously written in field
     * absoluteTargetPosition or relativeTargetPosition, - writeControlWord "3F" or
     * "7F" on controller depending if we want to go to a relative position or to an
     * absolute position.
     *
     * @param action
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {

        main.updateAgentState(FcsEnumerations.FilterState.valueOf("ROTATING"));

        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            carouselController.enableAndWriteRelativePosition(this.relativeTargetPosition);

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            // TODO check that the command CAN bus is arrived
            carouselController.writeTargetPosition(this.absoluteTargetPosition);

            carouselController.goToOperationEnable();

            // TODO check that the command CAN bus is arrived
            carouselController.writeControlWord(0x3F);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        this.carouselController.stopPosition();
        this.carouselController.goToSwitchOnDisabled();
    }

    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        int deltaPosBeforeDisable = position - absoluteTargetPosition;
        FCSLOG.info(name + " delta position before disable = " + deltaPosBeforeDisable);
        this.carouselController.goToSwitchOnDisabled();
        main.updateAgentState(FcsEnumerations.FilterState.valueOf("CAROUSEL_STOPPED"));
        /**
         * switch back controller to PositionSensorTypeEncoderSSI LSSTCCSFCS-223 this
         * has to be done here in endAction, because it has to be executed whatever
         * happened before (even if controller is in default)
         */
        setControllerPositionSensorTypeEncoderSSI();
        /*
         * Read position can be done now because controller sensor type is EncoderSSI.
         */
        position = carouselController.readPosition();
        startPosition = 0;
        FCSLOG.info(name + " position after disable = " + position + " delta " + (position - absoluteTargetPosition));
        initializeClampsState();
        updateStateWithSensors();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.carouselController.stopPosition();
        this.carouselController.goToSwitchOnDisabled();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of ROTATION COMMANDS **************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ************ methods which override FilterHolder *** ***********
     */
    /**
     * ****************************************************************
     */
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Override
    public boolean isHoldingFilter() {
        return this.isHoldingFilterAtStandby();
    }

    // TODO find something more clever to do here.
    @Override
    public boolean isNotHoldingFilter() {
        return !this.isHoldingFilterAtStandby();
    }

    @Override
    public boolean isAtHandoff() {
        return false;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the ID of filter at STANDBY")
    @Override
    public int getFilterID() {
        return socketAtStandby.getFilterID();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read sensors and update state", timeout = 2000)
    @Override
    public synchronized void updateStateWithSensors() {
        // do not execute if last finished less than 100 ms ago.
        // we keep it at the end of actions but don't repeat it
        if (System.currentTimeMillis() - lastUpdateStateWithSensors.get() < 100 && !FcsUtils.isSimu())
            return;
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-carousel")) {
            hyttc580.checkBooted();
            if (!clampsStateInitialized) {
                throw new FcsHardwareException(
                        name + ": clamps state not initialized. " + " Please launch command initializeClampsState.");
            }
            tcpProxy.updatePDOData();
            updateState();
            updateMinLockedThreshold();
            publishData();
            lastUpdateStateWithSensors.set(System.currentTimeMillis());
        }
    }

    AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    /**
     * This is similar to updateStateWithSensors but without
     * readAndUpdateOutputInterlocks() because there is no need to refresh the
     * output interlocks (signals coming from AC when the rotation has begun).
     * updateStateWithSensorsDuringMotion updates state only from PDO. But
     * readAndUpdateOutputInterlocks reads SDO.
     */
    private void updateStateWithSensorsDuringMotion() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensorsDuringMotion-carousel")) {
            hyttc580.checkBooted();
            if (!clampsStateInitialized) {
                throw new FcsHardwareException(
                        name + ": clamps state not initialized. " + " Please launch command initializeClampsState.");
            }
            tcpProxy.updatePDOData();
            updateState();
            publishData();
        }
    }

    /**
     * read lockSensorMinLocked on hyttc580.     * don't read more than one time by minute.
     */
    public void updateMinLockedThreshold() {
        // do not execute more than one time by minute (60000ms)
        if (System.currentTimeMillis() - lastUpdateLockSensorMinLocked.get() < 60000) {
            return;
        }
        minLockedThreshold = (int) readLockSensorMinLocked();
        lastUpdateLockSensorMinLocked.set(System.currentTimeMillis());
    }

    AtomicLong lastUpdateLockSensorMinLocked = new AtomicLong(0);

    public long readLockSensorMinLocked() {
        return hyttc580.readLockSensorMinLocked();
    }

    /**
     * Overridden method from FilterHolder
     *
     * @return false
     */
    @Override
    public boolean isAtOnline() {
        // carousel is never at ONLINE
        return false;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of methods which override FilterHolder ********
     */
    /**
     * ****************************************************************
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update socket at standby state from hyttc580 data.", timeout = 2000)
    public void updateSocketAtStandbyWithSensors() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketAtStandbyWithSensors-carousel")) {
            hyttc580.checkBooted();
            hyttc580.checkInitialized();
            tcpProxy.updatePDOData();
            updateSocketAtStandbyState();
        }
    }

    public void updateSocketAtStandbyState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketAtStandbyState-carousel")) {
            this.socketAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo1());
            if (socketAtStandbyID >= 1 && socketAtStandbyID <= 5) {
                socketAtStandby = socketsMap.get("socket" + socketAtStandbyID);
                socketAtStandby.updateState();
                socketAtStandby.updateFilterID();
            }
            publishData();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update state from hyttc580 data.", timeout = 2000)
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-carousel")) {
            updateSocketAtStandbyState();
            // doesn't read again controller CPU because the following values have been
            // refreshed by PDO.
            position = carouselController.getPosition();
            readCurrent = carouselController.getCurrent();
            readVelocity = carouselController.getVelocity();
            // The use of the scheduler is required in order to leave readCurrent thread as
            // soon as possible. Since we use the scheduler and not FcsUtils async functions
            // we need to use startAsync to get proper timing indentitation/markers
            int level = FcsUtils.getTimingLevel();
            subs.getScheduler().schedule(() -> {
                FcsUtils.startAsync(level);
                updateSocketNotAtStandby();
                FcsUtils.endAsync();
            }, 0, TimeUnit.SECONDS);
        }
    }

    public void updateSocketNotAtStandby() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketNotAtStandby")) {
            int socketNotAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo2());
            if (socketNotAtStandbyID >= 1 && socketNotAtStandbyID <= 5) {
                CarouselSocket socketToUpdate = socketsMap.get("socket" + socketNotAtStandbyID);
                socketToUpdate.updateState();
            }
        }
    }

    public FilterClampState getClampsStateAtStandby() {
        if (socketAtStandby == null) {
            return FilterClampState.UNDEFINED;
        }

        return socketAtStandby.getClampsState();
    }

    public boolean isEmptyAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isEmpty();
    }

    public boolean isUnclampedOnFilterAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isUnclampedOnFilter();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read on pt100 the temperatures", timeout = 2000)
    public void readTemperatures() {
        pt100.updateTemperatures();
    }

}
