
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * Prototype online clamps have no encoder so they are opened and closed in
 * CURRENT mode. This class gathers methods and commands which are specific to
 * prototype.
 *
 * @author virieux
 */

// note using inheritance to modify behavior is breaking the meaning of
// inheritance
// better practice : factorize common behaviour in abstract common parent
public class AutochangerThreeOnlineClampsProto extends AutochangerThreeOnlineClamps {

    public AutochangerThreeOnlineClampsProto(AutochangerOnlineClamp onlineClampXminus,
            AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        super(onlineClampXminus, onlineClampXplus, onlineClampYminus);
    }

    // the modified behaviour was in startAction
    // let's modify open and close

    @Override
    public void openClamps() {
        openClampsInCurrentMode();
    }

    @Override
    public void closeClamps() {
        closeClampsInCurrentMode();
    }

    /**
     * Closes the 3 clamps in mode current. initial state = OPENED final state =
     * CLOSED. for prototype only.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Close the 3 online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsInCurrentMode() {

        updateStateAndCheckSensors();
        // TODO test conditions for closing clamps
        if (isOpened()) {
            onlineClampYminus.closeClampInCurrentMode();
            closeClampsXInCurrentMode();

        } else if (isClosed()) {
            FCSLOG.info(name + " clamps already CLOSED nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be unlocked before.");
        }
    }


    /**
     * Opens the 3 clamps with a small pressure. initial state = CLOSED final state
     * = OPENED. For prototype only. For final pruducts AC1 and AC2 see openClamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Opens the 3 online clamps with a small pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsInCurrentMode() {
        updateStateAndCheckSensors();
        // TODO test conditions for opening clamps
        if (isClosed()) {
            checkReadyForAction();
            checkParametersToOpenClampsX();
            onlineClampYminus.openClampInCurrentMode();
            openClampsXInCurrentMode();

        } else if (isOpened()) {
            FCSLOG.info(name + " clamps already OPENED. Nothing to do.");
            throw new RejectedCommandException(name + " clamps already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be closed before.");
        }
    }

    /**
     * opens the 2 clampsX in mode CURRENT. for prototype only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "open clampXs. for proto only.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsXInCurrentMode() {
        checkParametersToOpenClampsX();
        FcsUtils.parallelRun(() -> onlineClampXminus.openClampInCurrentMode(),
                () -> onlineClampXplus.openClampInCurrentMode());
    }



    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "close clamps X", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsXInCurrentMode() {
        checkParametersToCloseClampsX();
        FcsUtils.parallelRun(() -> onlineClampXminus.closeClampInCurrentMode(),
                () -> onlineClampXplus.closeClampInCurrentMode());
    }





    public void checkParametersToOpenClampsX() {
        int initialValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int finalValueXminus = onlineClampXminus.getCurrentToOpen();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToOpen();
        int initialValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int finalValueXplus = onlineClampXplus.getCurrentToOpen();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToOpen();

        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToOpenClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(
                    maxTimeToOpenClampsX + " : value too low for maxTimeToOpenClampsX. should be >= " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }

    public void checkParametersToCloseClampsX() {
        int initialValueXminus = onlineClampXminus.getInitialCurrentToClose();
        int finalValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToClose();
        int initialValueXplus = onlineClampXplus.getInitialCurrentToClose();
        int finalValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToClose();

        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToCloseClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(maxTimeToCloseClampsX + " : value too low. should be > " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }

}
