
package org.lsst.ccs.subsystems.fcs.simulation;

import java.math.BigDecimal;
import java.security.SecureRandom;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.drivers.CanCBXAI814;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSensor14bits;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * To simulate a CanCBXAI814.
 * Used in Carousel simulation.
 * @author virieux
 */
public class SimuCanOpenADC extends CanCBXAI814 {

    int value1 = FCSCst.SENSOR14BITS_MIN;
    int value2 = FCSCst.SENSOR14BITS_MAX;
    private final SecureRandom randomGenerator = new SecureRandom();

    @Override
    public void initializeAndCheckHardware() {
        super.initializeAndCheckHardware();
        for (int i = 0; i< 7; i++) {
            updateFakePDOData(i,0);
        }
    }

    /**
     * Write transmission type on the device CPU.
     * @throws FcsHardwareException
     */
    protected void writeTransmissionTypeToDevice()  {
        FCSLOG.info(name + " is initialized.") ;
    }


    /**
     * In the simulator, instead of reading a new PDO to read a new value for the sensor,
     * we set a value for the sensor and then we update the PDOData for this sensor.
     * @param numOfAnalogInput
     * @param value
     */
    public void updateFakePDOData(int numOfAnalogInput, int value) {
        if (numOfAnalogInput >= 8) {
            throw new IllegalArgumentException(numOfAnalogInput + " bad value; should be < 8");
        }
        int cobid;
        long newPDOValue;
        inputs[numOfAnalogInput] = value;
        if (numOfAnalogInput < 5) {
            cobid = cobid1;
            newPDOValue = inputs[0] + (inputs[1] << 16)
                    + ((long) inputs[2] << 32) + ((long) inputs[3] << 48);
        } else {
            cobid = cobid2;
            newPDOValue = inputs[4] + (inputs[5] << 16)
                    + ((long) inputs[6] << 32) + ((long) inputs[7] << 48);
        }
        ((SimuCanOpenInterface) tcpProxy.getCanInterface()).simulatePDOData(cobid, newPDOValue);
    }


    /**
     * simulate that value of sensor, which is at input no numOfAnalogInput, is between min value and max value.
     * @param numOfAnalogInput
     * @param minValue
     * @param maxValue
     */
    public void simulateNewValue(int numOfAnalogInput, int minValue, int maxValue) {
        checkValues(minValue,maxValue);
        int value = computeNewValue(minValue,maxValue);
        updateFakePDOData(numOfAnalogInput,value);
    }

    void checkValues(int aValue1, int aValue2) {
        if (aValue2 > FCSCst.SENSOR14BITS_MAX) {
            throw new IllegalArgumentException(name +
                "aValue2="+aValue2 +" Can't be >"+FCSCst.SENSOR14BITS_MAX);
        }
        if (aValue1 < FCSCst.SENSOR14BITS_MIN) {
            throw new IllegalArgumentException(name +
                " aValue1="+aValue1 +" Can't be <"+FCSCst.SENSOR14BITS_MIN);
        }

        if (aValue2 < aValue1) {
            throw new IllegalArgumentException(name +
                " aValue1="+aValue1 + " aValue2="+aValue2 + "we must have: aValue1<aValue2");
        }
    }

    /**
     * The simulated sensor doesn't read its new value on a CANopen device but the new value is
     * generated by a random generator between a min and a max value.
     * @return a random between value1 and value2
     */
    int computeNewValue(int value1, int value2) {
        if (value2 > 0) {
            int n = randomGenerator.nextInt(value2);

            if (n > value1) {
                return n;
            } else {
                BigDecimal bd = new BigDecimal(value1 + n * (value2 - value1) / value2);
                return bd.intValue();
            }
        } else {
            String msg = getName() + ": ERROR in readNewValue: value2 should be positive";
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * simulates that a CanOpenSensor14bits plugged on this device returns a value between minValue and maxValue.
     * @param sensor
     * @param minValue
     * @param maxValue
     */
    public void simulateSensorValue(CanOpenSensor14bits sensor, int minValue, int maxValue) {
        simulateNewValue(sensor.getNumOfAnalogInput(),minValue,maxValue);
        sensor.updateValue();
    }

}
