package org.lsst.ccs.subsystems.fcs.utils;

import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This class gathers static final variables and static methods.
 *
 * By example, to creates Objects for GUI and Status Bus.
 *
 * @author virieux
 */
public final class FcsUtils {

    protected static final Logger FCSLOG = FCSCst.FCSLOG;

    /**
     * The maximum value which a number coded on 2 bytes can reach.
     */
    public static final int MAX_VALUE_2BYTES = 65535;

    /**
     * The maximum number of CANopen devices which can be on a CANbus. Also the
     * maximum CANopen node ID a device can have.
     */
    public static final int MAX_NODE_ID = 127;

    /**
     * A value for all fcs Modules tickmillis.
     */
    public static final int TICKMILLIS = 3000;

    private static final String[] CAN_OPEN_COMMANDS = new String[] { "sync", "scan", "rsdo", "wsdo", "info", "quit",
            "srtr", "reset" };

    /**
     *
     * @param name
     */
    public static void checkSocketName(String name) {
        String errorMsg = ": invalid socket name - Socket name must be \"" + SOCKET_NAME + 'X' + "\" where 0 " + '<'
                + "= X " + '<' + "= 5";

        /* Socket Name can be an empty String or */
        /* Socket Name can be "AC" if filter is on AUTOCHANGER. */
        if (name.isEmpty() || name.contentEquals("AC"))
            return;

        /* Socket Name must start with "socket". */
        if (!name.startsWith(SOCKET_NAME))
            throw new IllegalArgumentException(name + errorMsg);

        /* Socket Name must end with 1, 2, 3, 4 or 5. */
        String end = name.substring(6);
        try {
            int ix = Integer.parseInt(end);
            if (ix < 1 || ix > 5)
                throw new IllegalArgumentException(name + errorMsg);

        } catch (NumberFormatException ex) {
            throw new IllegalArgumentException(name + errorMsg, ex);
        }
    }

    /**
     * Computes a new value given as argument in replacing bit which is at position
     * given as argument by 0.
     *
     * @param val value to transform
     * @param bit bit numero to be replaced
     * @return new value
     * @throws IllegalArgumentException if bit <0 or bit >15
     */
    public static int force2zero(int val, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        return val & ~(1 << bit);
    }

    /**
     * Return a new value where the digit (bit number given as argument) has been
     * replaced by 1.
     *
     * @param val
     * @param bit
     * @return
     */
    public static int force2one(int val, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        return val | (1 << bit);
    }

    /**
     * Sets the nth bit
     *
     * @param val     the value to modifiy
     * @param bit     the bit number to set
     * @param zeroOne if zero forces to zero. Forces to one otherwise
     * @return
     */
    public static int forceBit(int val, int bit, int zeroOne) {
        if (zeroOne == 0) {
            return force2zero(val, bit);
        } else {
            return force2one(val, bit);
        }
    }

    /**
     * build a Can Open wsdo Command that can understand the C wrapper.
     *
     * exemple : wsdo,2,6411,01,2,3000
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @param size
     * @param data
     * @return
     */
    public static String buildWsdoCommand(int nodeID, int index, int subindex, int size, int data) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("wsdo").append(sep);
        sb.append(Integer.toHexString(nodeID)).append(sep);
        sb.append(Integer.toHexString(index)).append(sep);
        sb.append(Integer.toHexString(subindex)).append(sep);
        sb.append(Integer.toHexString(size)).append(sep);
        sb.append(Integer.toHexString(data));
        return sb.toString();
    }

    /**
     * build a Can Open rsdo Command that can understand the C wrapper. exemple:
     * rsdo,1,1018,0
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return
     */
    public static String buildRsdoCommand(int nodeID, int index, int subindex) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("rsdo").append(sep);
        sb.append(Integer.toHexString(nodeID)).append(sep);
        sb.append(Integer.toHexString(index)).append(sep);
        sb.append(Integer.toHexString(subindex));
        return sb.toString();
    }

    /**
     * returns true if the command given as argument is an available CANopen
     * command.
     *
     * @param command
     * @return
     */
    public static boolean isValidCommandWord(String command) {
        for (String s : CAN_OPEN_COMMANDS) {
            if (s.equals(command))
                return true;
        }
        return false;
    }

    /**
     * @param command
     * @throws IllegalArgumentException if command is not a valid command.
     */
    public static void checkCommand(String command) {
        if (command == null)
            throw new IllegalArgumentException(" null command");
        String[] words = command.split(",");
        if (words.length == 0 || !isValidCommandWord(words[0])) {
            throw new IllegalArgumentException(command + " invalid command");
        }
    }

    /**
     * Convert a number coded on 8 bits which represents a signed number (INTEGER8)
     * to a byte.
     *
     * @param integer8
     * @return
     */
    public static byte convertInteger8(long integer8) {
        if (integer8 >= Math.pow(2, 8)) {
            throw new IllegalArgumentException(integer8 + " too big. Should be coded on 8 bits, so < Math.pow(2,8)");
        }
        return (byte) integer8;
    }

    /**
     * Convert a number coded on 16 bits which represents a signed number
     * (INTEGER16) to a short. Used to interprete numbers coming from CANopen
     * devices where we can have signed or unsigned numbers depending of which
     * parameter we read.
     *
     * INTEGER16=0 => value=0 INTEGER16=2356 => value=2356 INTEGER16=65342 =>
     * value=-194 INTEGER16=32767 => value=32767 INTEGER16=32768 => value=-32768
     * INTEGER16=65535 => value=-1
     *
     * @param integer16 a long because readSDO command returns a long.
     * @return
     */
    public static short convertInteger16(long integer16) {
        if (integer16 >= Math.pow(2, 16)) {
            throw new IllegalArgumentException(integer16 + " too big. Should be coded on 16 bits, so < Math.pow(2,16)");
        }
        return (short) integer16;
    }

    /**
     * Convert a number coded on 32 bits which represents a signed number
     * (INTEGER32) to an int.
     *
     * @param integer32
     * @return
     */
    public static int convertInteger32(long integer32) {
        if (integer32 >= Math.pow(2, 32)) {
            throw new IllegalArgumentException(integer32 + " too big. Should be coded on 32 bits, so < Math.pow(2,32)");
        }
        return (int) integer32;
    }

    /**
     * Displays the list of parameters that we have to define for a given mode.
     *
     * @param modeInString
     * @return
     */
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: ");
        sb.append(modeInString);
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.EposMode.valueOf(modeInString).getParameters();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Displays the list of parameters.
     *
     * @return
     */
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : ");
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.Parameter.values();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * For online clamps.
     *
     * @param initialValue
     * @param finalValue
     * @param increment
     * @return
     */
    public static int getSignedStepHeight(int initialValue, int finalValue, int increment) {
        if (finalValue - initialValue > 0) {
            return increment;
        } else {
            return -increment;
        }
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        System.out.println(FcsUtils.displayListParameters("HOMING"));
        System.out.println(FcsUtils.displayListParameters("CURRENT"));
        System.out.println(FcsUtils.displayListParameters("PROFILE_POSITION"));
        System.out.println(FcsUtils.displayListParameters());
    }

    /**
     * Returns true if the final value is reached.
     *
     * @param stepHeight
     * @param newCurrentValue
     * @param finalValue
     * @return
     */
    public static boolean finalValueReached(int stepHeight, int newCurrentValue, int finalValue) {
        // TODO why not >= and <= if "reached" ?
         boolean goingUp = stepHeight > 0 && newCurrentValue > finalValue;
        boolean goingDown = stepHeight < 0 && newCurrentValue < finalValue;
        return goingUp || goingDown;
    }

    public static int computeNewCurrentValue(int stepHeight, int currentValue, int finalValue) {
        if (stepHeight > 0) {
            return Math.min(currentValue + stepHeight, finalValue);
        } else {
            return Math.max(currentValue + stepHeight, finalValue);
        }
    }

    public static void sleep(int duration, String itemName) {
        try {
            FCSLOG.finest(itemName + " BEGIN SLEEP for " + duration + " milliseconds");
            Thread.sleep(duration);
            FCSLOG.finest(itemName + " END SLEEP for " + duration + " milliseconds");
        } catch (InterruptedException ex) {
            String msg = itemName + " interrupted while sleeping";
            FCSLOG.error(itemName + msg, ex);
        }
    }

    public static void checkPositive(int arg) {
        if (arg < 0) {
            throw new IllegalArgumentException(arg + ":illegal value. Must be > 0");
        }
    }

    /**
     * Compute index to read a byte on a CanOpenPlutoGateway
     *
     * @param byteNumero
     * @return
     */
    public static int returnIndex(int byteNumero) {
        int index = 0x6000 + byteNumero / 4;
        return index;
    }

    /**
     * Compute subindex to read a byte on a CanOpenPlutoGateway
     *
     * @param byteNumero
     * @return
     */
    public static int returnSubindex(int byteNumero) {
        int subindex;
        subindex = 1 + byteNumero % 4;
        return subindex;
    }

}
