package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import static org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel.extractCurrentAverage;
import static org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel.extractFollowingError;
import static org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel.extractPosition;
import static org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel.extractStatusWord;
import static org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel.extractVelocity;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This class is the driver of the EPOS controller EPOS24 which controls motion
 * of Autochanger Trucks on the linear rails.
 *
 * @author virieux
 */
public class CanOpenEPOSLinearRailTruck extends CanOpenEPOS implements EPOSControllerForLinearRail {

    @ConfigurationParameter(description = "minimal position of the encoder ribbon.", category = "controller")
    private volatile int encoderRibbonMinValue;

    private boolean homingDone;

    /**
     * A boolean to be set each time brake is activated or released. This
     * boolean avoid to send a command to read on the controller each time we
     * publish controller status.
     */
    private boolean brakeActivatedPub = false;

    private boolean configDigitalInputOK = true;
    private boolean configDigitalOutputOK = true;
    private StringBuilder configDigitalInputSb;
    private StringBuilder configDigitalOutputSb;

    /**
     * define cobid to be able to read values from PDO data.
     */
    protected int cobid1 = this.nodeID + 0x180;
    protected int cobid2 = this.nodeID + 0x280;

    /**
     * PDO1 represents:
     * current: 2 bytes
     * positionActualValue: 4 bytes
     * statusWord: 2 bytes
     */
    private long pdo1 = 0L;

    /**
     * PDO2 represents:
     * FollowingError: 2 bytes
     * VelocityActualValue 4 bytes
     * StatusWord: 2 bytes
     */
    private long pdo2 = 0L;

    /* for tests at SLAC AC linear rail controllers are not configured to send PDOs */
    private final boolean updateByPDO = false;

    @Override
    public boolean isBrakeActivatedPub() {
        return brakeActivatedPub;
    }

    @Override
    public void setBrakeActivatedPub(boolean brakeActivatedToPublish) {
        this.brakeActivatedPub = brakeActivatedToPublish;
    }

    /**
     * return true if homing is done: the controller knows its absolute position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description = "return true if homing is done: the controller knows its absolute position.")
    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Do homing of the controller.")
    @Override
    public void homing() {
        checkInitialized();
        homingDone = false;
        int ssiPosition = readSSIPosition();
        int pos = ssiPosition - encoderRibbonMinValue;
        FCSLOG.debug(getName() + " ssiPosition=" + ssiPosition + ", position=" + pos);

        if (Math.abs(pos - readPosition()) < 10) {
            FCSLOG.info(getName() + " no homing to be done.");
            homingDone = true;
        } else {
            doHoming(pos);
            publishData();
        }
    }

    private void doHoming(int pos) {
        FCSLOG.debug(getName() + " ==> BEGIN homing");
        EposMode myMode = readMode();
        goToOperationEnable();
        defineAbsolutePosition(pos);
        checkHomingDone(pos);
        homingDone = true;
        changeMode(myMode);
        activateBrakeAndDisable();
        FCSLOG.debug(getName() + " ==> END homing");
    }

    private void checkHomingDone(int pos) {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " homing target not yet reached. duration=" + duration);
            duration = System.currentTimeMillis() - timeStart;
        }
        if (isTargetReached() && pos == readPosition()) {
            writeControlWord(0xF);

        } else {
            activateBrakeAndDisable();
            throw new FcsHardwareException(getName() + " couldn't do homing :" + " target is not reached.");
        }
    }

    /**
     * Configure this controller to be used as an autochanger linear rail truck
     * controller. This method is to be executed during initialization phase.
     *
     * This configures the digital inputs of the EPOS controller to work with a
     * holding brake.
     *
     * Digital Inputs activated for Linear Rails :
     * **************************************************************************
     * Drive Enable Input=01 Mask=Enabled Polarity=HighActive
     * QuickStop Input=02 Mask=Enabled Polarity=HighActive
     * **************************************************************************
     * 1 - Configure the Digital Input Functionnality index 0x2070 subindex 01 to 04
     * Functionnality tab Value = 15 General Purpose A Value = 14 General Purpose B
     * Value = 13 General Purpose C Value = 12 General Purpose D Value = 11 General
     * Purpose E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop 1 Value = 4 Drive enable 1 Value = 3 Position marker 0
     * Value = 2 Home switch 0 Value = 1 Positive Limit switch 0 Value = 0 Negative
     * Limit switch 0
     * **************************************************************************
     * 2 - Configure the Digital Input Polarity index 0x2071 subindex 03 Value=0
     * HighActive Value=1 Low Active
     * **************************************************************************
     * 3 - Configure the Digital Input Mask index 0x2071 subindex 02 Value=0
     * HighActive Value=1 Low Active
     * **************************************************************************
     * 4 - Configure the Digital Input Execution Mask index 0x2071 subindex 04
     * Used to suppress execution of the digital input functionalities.
     * The function (Negative or Positive Limit Switch Error Routine) will be
     * executed when the associated bit in functionalities state register goes high
     * and the bit in this execution mask is set.
     * Functionnality tab Value = 15 General Purpose A 0 Value = 14
     * General Purpose B 0 Value = 13 General Purpose C 0 Value = 12 General Purpose
     * D 0 Value = 11 General Purpose E 0 Value = 10 General Purpose F 0 Value = 9
     * General Purpose G 0 Value = 8 General Purpose H 0 Value = 7 General Purpose I
     * 0 Value = 6 General Purpose J 0 Value = 5 QuickStop 1 Value = 4 Drive enable
     * 1 Value = 3 Position marker 0 Value = 2 Home switch 0 Value = 1 Positive
     * Limit switch 0 Value = 0 Negative Limit switch 0
     *
     *
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To configure the autochanger linear rails controllers.")
    // tested on CPPM testbench in september 2015
    @Override
    public void configureDigitalInputOfLinearRails() {
        // Drive Enable
        writeParameter(Parameter.ConfigurationOfDigitalInput1, 0x4);

        // QuickStop
        writeParameter(Parameter.ConfigurationOfDigitalInput2, 0x5);

        // DigitalInputPolarity
        // Value=0 HighActive
        // Value=1 Low Active
        writeParameter(Parameter.DigitalInputFonctionnalityPolarity, 0x0);

        // DigitalInputFonctionnalitiesMask
        // Used to display the state of the digital input functionalities.
        // If a bit is set to “1”, the functionality state will be displayed.
        writeParameter(Parameter.DigitalInputFonctionnalityMask, 0x30);

        // DigitalInputFonctionnalitiesExecutionMask
        writeParameter(Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "check configuration of digital input of linear rail controller.")
    public void checkConfigDigitalInputOfLinearRails() {
        configDigitalInputSb = new StringBuilder();
        configDigitalInputSb.append(" Configuration of Digital Input:");
        checkParamInput(Parameter.ConfigurationOfDigitalInput1, 4);
        checkParamInput(Parameter.ConfigurationOfDigitalInput2, 5);
        checkParamInput(Parameter.DigitalInputFonctionnalityPolarity, 0);
        checkParamInput(Parameter.DigitalInputFonctionnalityMask, 0x30);
        checkParamInput(Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
        if (configDigitalInputOK) {
            FCSLOG.info(configDigitalInputSb.toString());
        } else {
            FCSLOG.error(configDigitalInputSb.toString());
        }
    }

    private void checkParamInput(Parameter param, int configValue) {
        int paramValue = (int) this.readParameter(param);
        if (configValue != paramValue) {
            configDigitalInputOK = false;
        }
        configDigitalInputSb.append(param).append("=");
        configDigitalInputSb.append(paramValue).append(";");
    }

    /**
     * Configuration of an EPOS controller to work with a holding brake. This is
     * used for controllers of the Autochanger Linear Rails.
     *
     * Digital Outputsactivated for Linear Rails are: General_A
     * Output=01 Mask=Enabled Polarity=HighActive Ready/Fault
     * Output=02 Mask=Enabled Polarity=HighActive Holding Brake
     * Output=03 Mask=Enabled Polarity=LowActive
     * ********************************************************************************
     * 1- First of all, mask all outputs index 0x2079 subindex 01
     * ********************************************************************************
     * 2- Configure the Digital Output Fonctionnalities index 0x2079 subindex 01 to 03
     * Functionnality tab Value = 15 General Purpose Out_A Value = 14 General
     * Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General Purpose
     * Out_D Value = 11 General Purpose Out_E Value = 10..8 not used Value = 7..3
     * reserved Value = 2 Holding Brake Value = 1 Position compare Value = 0
     * Ready/Fault
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value = 1
     * associated output inverted => LowActive Tab Digital Output Functionnality
     * Mask Value Bit_15 General Purpose Out_A 0 Bit_14 General Purpose Out_B 0
     * Bit_13 General Purpose Out_C 0 Bit_12 General Purpose Out_D 0 Bit_11 General
     * Purpose Out_E 0 Bit_10 General Purpose Out_F 0 Bit_9 General Purpose Out_G 0
     * Bit_8 General Purpose Out_H 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5
     * reserved 0 Bit_4 reserved 0 Bit_3 reserved 0 Bit_2 Holding Brake 1 (read
     * only) Bit_1 Position Compare 0 (read only) Bit_0 Ready/Fault 0 (read only)
     * ********************************************************************************
     * 4 - Configure the Digital Output Functionnality Mask index 0x2078 subindex
     * 0x02 Value = 0 functionnality not activatd Value = 1 functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General
     * Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7 reserved 0
     * Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0 Bit_3 reserved 0 Bit_2
     * Holding Brake 1 Bit_1 Position Compare 0 Bit_0 Ready/Fault 1
     * ********************************************************************************
     * To configure the autochanger linear rails controllers. This method is to be
     * executed during initialization phase.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To configure the autochanger linear rails controllers.")
    // tested on CPPM testbench in september 2015
    @Override
    public void configureDigitalOutputOfLinearRails() {
        // 1-Mask all outputs
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0);

        // 2-Configuration of Digital Output Fonctionnalities
        // General_A
        writeParameter(Parameter.ConfigurationOfDigitalOutput1, 0xF);
        // Ready/Fault
        writeParameter(Parameter.ConfigurationOfDigitalOutput2, 0);
        // Holding Brake
        writeParameter(Parameter.ConfigurationOfDigitalOutput3, 0x2);

        // 3-Configuration of Digital Output Fonctionnality Polarity
        writeParameter(Parameter.DigitalOutputFonctionnalityPolarity, 0x4);

        // 4-Configuration of Digital Output Fonctionnality Mask
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0x8005);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "check configuration of digital output of linear rail controller.")
    public void checkConfigDigitalOutputOfLinearRails() {
        configDigitalOutputSb = new StringBuilder();
        configDigitalOutputSb.append(" Configuration of Digital Output:");
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput1, 0xf);
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput2, 0);
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput3, 2);
        checkParamOutput(Parameter.DigitalOutputFonctionnalityPolarity, 4);
        checkParamOutput(Parameter.DigitalOutputFonctionnalityMask, 0x8005);
        if (configDigitalOutputOK) {
            FCSLOG.info(configDigitalOutputSb.toString());
        } else {
            FCSLOG.error(configDigitalOutputSb.toString());
        }
    }

    private void checkParamOutput(Parameter param, int configValue) {
        int paramValue = (int) this.readParameter(param);
        if (configValue != paramValue) {
            configDigitalOutputOK = false;
        }
        configDigitalOutputSb.append(param).append("=");
        configDigitalOutputSb.append(paramValue).append(";");
    }

    @Override
    public void shutdown() {
        // shutdown of linear rail controllers is done in AutochangerTwoTrucks
    }

    /***
     * methods for managing PDOs
     *
     * @throws DriverException
     */

    @Override
    public void doInitializePDOs() throws DriverException {
        if (updateByPDO) {
            tcpProxy.addReceivedPDO(cobid1);
            tcpProxy.addReceivedPDO(cobid2);
        }
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        if (updateByPDO) {
            FCSLOG.finest(name + " updatingFromPDO = " + pdo);
            boolean updated = false;
            if (pdo.getPDOs().containsKey(cobid1)) {
                pdo1 = (long) pdo.getPDOs().get(cobid1);
                updated = true;
                /* update from PDO1 */
                FCSLOG.finest(name + " updatingFromPDO1 = " + pdo1 + " binaire:" + Long.toBinaryString(pdo1));
                statusWord = extractStatusWord(pdo1);
                current = (short) extractCurrentAverage(pdo1);
                position = (int) extractPosition(pdo1);
            }
            if (pdo.getPDOs().containsKey(cobid2)) {
                pdo2 = (long) pdo.getPDOs().get(cobid2);
                updated = true;
                /* update from PDO2 */
                FCSLOG.finest(name + " updatingFromPDO2 = " + pdo2 + " binaire:" + Long.toBinaryString(pdo2));
                followingError = (short) extractFollowingError(pdo2);
                velocity = (int) extractVelocity(pdo2);
            }
            if (updated) {
                this.publishData();
            }
        }
    }

    /**
     * For tests and debug
     *
     * @return values updated by PDOs
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print values updated by PDOs to debug and test")
    public String printValuesUpdatedByPDOs() {
        StringBuilder sb = new StringBuilder("=> pdo1 value = ");
        sb.append(pdo1);
        sb.append(" / pdo1 = 0x");
        sb.append(Long.toHexString(pdo1));
        sb.append(" / status word = 0x");
        sb.append(Long.toHexString(statusWord));
        sb.append("\n status word = 0b");
        sb.append(Long.toBinaryString(statusWord));
        sb.append("\n current average = ");
        sb.append(current);
        sb.append(" / current average = 0x");
        sb.append(Long.toHexString(current));
        sb.append("\n position = ");
        sb.append(position);
        sb.append(" / position = 0x");
        sb.append(Long.toHexString(position));
        sb.append("\n => pdo2 value = ");
        sb.append(pdo2);
        sb.append(" / pdo2 = 0x");
        sb.append(Long.toHexString(pdo2));
        sb.append("\n followingError = ");
        sb.append(followingError);
        sb.append(" / followingError = 0x");
        sb.append(Long.toHexString(followingError));
        sb.append(followingError);
        sb.append("\n Velocity = ");
        sb.append(velocity);
        sb.append(" / velocity = 0x");
        sb.append(Long.toHexString(velocity));
        return sb.toString();
    }
}
