
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;

/**
 * This is a Model for the system which identifies filters on the autochanger trucks.
 * On the autochanger trucks, we have hall effect sensors to identify filters.
 * 3 sensors are on the truck X- and 3 sensors are on the truck X+.
 * Each sensors gives a digital signal depending if there is a magnet in front of the sensors.
 * Following the magnet combination, we can define 64 ID numbers (from 0 to 63).
 * The location are named ID0, ID1 and ID2 on the X- side,
 * and they are named ID3, ID4, and ID5 on the X+ side.
 * The ID0 is the Least Significant Bit (LSB)
 * The ID5 is the Most Significant Bit (MSB)
 *
 * cf document LCA-13492
 *
 * @author virieux
 */
public class FilterIdentificator {

    private static final int SENSORS_NUMBER = 6;

    /**
     * An array of 6 filter ID digital sensors.
     */
    @LookupField(strategy = CHILDREN)
    private final ArrayList<DigitalSensor> filterIDSensors = new ArrayList<>();

    private int filterId;

    /**
     * return filterId. Don't read sensors again.
     *
     * @return filterId
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return filterId.")
    public int getFilterId() {
        return filterId;
    }

    /**
     * Return the list of filterID sensors. Used in simulation and for tests.
     *
     * @return
     */
    public List<DigitalSensor> getFilterIDSensors() {
        @SuppressWarnings("unchecked")
        List<DigitalSensor> filterSensors = (List<DigitalSensor>) filterIDSensors.clone();
        return filterSensors;
    }

    /**
     * Update filterId from sensors values. If a sensor is ON, bit=1 and if sensor
     * is off, bit = 0.
     */
    public void updateFilterId() {
        int id = 0;
        for (int i = 0; i < SENSORS_NUMBER; i++) {
            if (!filterIDSensors.get(i).isOn()) {
                id = id + (int) Math.pow(2, i);
            }
        }
        filterId = id;
    }
}
