package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_PLUTOGATEWAY_NAME;

/**
 * A class to represent the CANopen pluto gateway which is a device where we can read sensors.
 * Used for the autochanger and the loader sensors.
 * There is 2 devices plutoGateway : one for the autochanger, another one for the loader.
 *
 * @author virieux
 */
public class CanOpenPlutoGateway extends PlutoGateway {
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    
    @Override
    public void init() {
        super.init();
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(SDO_ERROR.getAlert(name), alwaysClear);
    }
    
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Initialize the Gateway.")
    public void initializeAndCheckHardware() {
        checkBooted();
        try {
            FCSLOG.info(name + " BEGIN initializeHardware");
            this.configurePlutoGateway();
            this.initialized = true;
            FCSLOG.info(name + " is INITIALIZED.");
            FCSLOG.info(name + " END initializeHardware");
            // publishData for the LoaderHardwareGUI
            this.publishData();

        } catch (SDORequestException | FailedCommandException ex) {
            throw new FcsHardwareException(name, ex);
        }
    }

    /**
     * Configures the gateway. Configuration depends on specific needs for each
     * subsystems. This methods switches to the right configuration in reading the
     * name of the gateway.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Configures this gateway for subsystem needs. "
            + "If gateway's name contains loader it executes configureLoaderPlutoGateway"
            + "else it executes configureAutochangerPlutoGateway.")
    public void configurePlutoGateway() {
        // TODO this is ugly.
        if (name.contains("loader")) {
            configureLoaderPlutoGateway();
        } else if (getName().startsWith("acSensors")) {
            configureAutochangerPlutoGateway();
        }
    }

    /**
     * Configures this gateway to receive 3 data blocs from Pluto. Used for
     * autochanger pluto gateway. Assuming Pluto has 3 fonctions Gateway_to_User_C
     * First one => Fonction_ID=01 Second one => Fonction_ID=02 Third one =>
     * Fonction_ID=03
     *
     * @throws SDORequestException
     */
    private void configureAutochangerPlutoGateway() {

        FCSLOG.info(name + " configureAutochangerGateway");

        // Clear config
        writeSDO(0x2010, 1, 0x1, 0x0);

        // Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        writeSDO(0x2011, 1, 0x2, 0x17);

        // Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        writeSDO(0x2011, 2, 0x2, 0x4);

        // Set Additionnal data TPDO2 => define Aera_2 (doc p:96)
        writeSDO(0x2012, 0x1, 0x2, 0x117);

        // Set Additionnal data TPDO2 => define Aera_3 (doc p:96)
        writeSDO(0x2012, 0x2, 0x02, 0x143);
    }

    /**
     * Configures plutoGateway for loader needs.
     */
    private void configureLoaderPlutoGateway() {
        FCSLOG.info(name + " configureLoaderPlutoGateway");

        // Clear additional data configuration.
        writeSDO(0x2010, 1, 1, 0x1);

        // Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        writeSDO(0x2011, 1, 2, 0x1);

        // Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        writeSDO(0x2011, 2, 2, 0x2);

        // Set Additionnal data TPDO2 => define Aera_2 (doc p:96)
        writeSDO(0x2012, 1, 2, 0x3);

        // Activate additional data for Area_0, Area_1 and Area_2 (Enable Data To Pluto
        // Areas 0-3)
        writeSDO(0x2002, 1, 1, 0x7);
    }

    /**
     * Write a transmission type in the CPU of the pluto gateway.
     *
     * @param transmissionType
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    public void writeTransmissionType(int transmissionType) {
        writeSDO(0x2005, 1, 1, transmissionType);
    }

    /**
     * return an array of 16 values read on the device. It reads 16 PDO (rsdo) with
     * index from 0x6000 to 0x6003, and subindex from 1 to 4.
     *
     * @return readValues
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return values read on the device.")
    @Override
    public int[] readNewValues() {
        // TODO remove calls to reReadValueInError if sleep resolves the issue of
        // CanOpenPLutoGateway stuttering

        checkBooted();
        try {
            /*
             * Values returned by Object 0x6000..0x6003 subindex 1..4 are UINT8. Could be a
             * represented by a byte.
             */
            /* Index 0x6000 Subindex 1 = byteNumero:0 */
            /* Index 0x6000 Subindex 2 = byteNumero:1 */
            /* Index 0x6000 Subindex 3 = byteNumero:2 */
            /* Index 0x6000 Subindex 4 = byteNumero:3 */
            /* Index 0x6001 Subindex 1 = byteNumero 4 */
            /* Index 0x6001 Subindex 2 = byteNumero 5 */
            /* Index 0x6001 Subindex 3 = byteNumero 6 */
            /* Index 0x6001 Subindex 4 = byteNumero 7 */
            /* Index 0x6002 Subindex 1 = byteNumero 8 */
            /* Index 0x6002 Subindex 2 = byteNumero 9 */
            /* Index 0x6002 Subindex 3 = byteNumero 10 */
            /* Index 0x6002 Subindex 4 = byteNumero 11 */
            /* Index 0x6003 Subindex 1 = byteNumero 12 */
            /* Index 0x6003 Subindex 2 = byteNumero 13 */
            /* Index 0x6003 Subindex 3 = byteNumero 14 */
            /* Index 0x6003 Subindex 4 = byteNumero 15 */
            
            int lastBytePlusOne = 16;
            if (LOADER_PLUTOGATEWAY_NAME.equals(this.getName())) {
                //the last 4 bytes on loaderPlutoGateway are always = 0.
                //no need to read them.
                lastBytePlusOne = 12;
            }

            for (int byteNumero = 0; byteNumero < lastBytePlusOne; byteNumero++) {
                int index = FcsUtils.returnIndex(byteNumero);
                int subindex = FcsUtils.returnSubindex(byteNumero);
                readValues[byteNumero] = (int) readSDO(index, subindex);
                if (byteNumero > 0) {
                    reReadValueInError(byteNumero, index, subindex);
                }
            }

        } catch (SDORequestException ex) {
            String message = name + "=> ERROR IN READING SENSOR:";
            this.raiseWarning(SDO_ERROR, message, name, ex);
        }
        return readValues;
    }

    /**
     * Resolves the issue of CanOpenPlutoGateway stuttering. Very ugly. If the
     * new value read on plutoGateway is the same than the previous one, it's an
     * error, so we have to read it again.
     *
     * @param i Value to be read again
     * @param index
     * @param subindex
     */
    private void reReadValueInError(int i, int index, int subindex) {
        if (readValues[i] == readValues[i - 1]) {
            FCSLOG.fine(name + " error in reading values for byte=" + i + ",index=0x" + Integer.toHexString(index)
                    + ", subindex=" + subindex);
            FcsUtils.sleep(2, name);
            readValues[i] = (int) readSDO(index, subindex);
        }
    }
}
