package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the CCS
 * Master Control Module (MCM) in normal operation mode or from a Console
 * (org.lsst.ccs.subsystem.console.TestConsole).
 *
 * <P/>
 * Example of commands that can be sent to the FcsMain : <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>
 * The main goals of the FcsMain is :
 * <UL>
 * <LI>Receive commands from MCM for the whole subsystem.</LI>
 * <LI>Execute commands received from MCM in invoking methods on the Carousel
 * Module on the Autochanger Module and on Loader Module.</LI>
 * <LI>Handle the logic of the whole subsystem.</LI>
 * <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 *
 * @author virieux
 *
 */
public class FcsMain extends MainModule {

    /**
     *
     */
    private static final long serialVersionUID = 7669526660659959402L;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;

    @LookupField(strategy = CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = CHILDREN)
    private Autochanger autochanger;

    @LookupField(strategy = CHILDREN)
    private Loader loader;

    @LookupField(strategy = CHILDREN)
    private FilterManager filterManager;

    private long setFilterDuration = 0;

    @Override
    public void build() {
        if (bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (bridgeToLoader.equals(bridge)) {
            throw new RuntimeException(
                    "In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();
    }

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make FcsGUI listen to my subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs",
                "fcs");

        // register setFilterDuration without any other information
        // dataProviderDictionaryService.registerData(new
        // KeyValueData("setFilterDuration", setFilterDuration));
        DataProviderInfo data = new DataProviderInfo("", DataProviderInfo.Type.TRENDING, "setFilterDuration");
        data.addAttribute(DataProviderInfo.Attribute.UNITS, "milliseconds");
        data.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "duration of command setFilter");
        data.addAttribute(DataProviderInfo.Attribute.TYPE, "long");
        // register setFilterDuration with units, description, type.
        dataProviderDictionaryService.addDataProviderInfoToDictionary(data);

        super.init();
    }

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY);
            updateAgentState(FilterReadinessState.READY);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        // because of some sensors still missing, autochanger is never initialized
        // but we need the Filter to be ready for filter exchanges
        // TODO remove after tests
        updateAgentState(FilterState.READY);
        updateAgentState(FilterReadinessState.READY);
    }

    /**
     * Return the name of the filter which is at ONLINE. Return null if no filter is
     * at ONLINE.
     *
     * @return
     */
    public String getOnlineFilterName() {
        if (autochanger.isHoldingFilter()) {
            return autochanger.getFilterOnTrucksName();

        } else {
            return null;
        }
    }

    /**
     * For end user. Return the name of the filter which is at ONLINE. Return null
     * if no filter is at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns the name of the filter which is at ONLINE.")
    public String printFilterONLINEName() {
        if (getOnlineFilterName() != null) {
            return getOnlineFilterName();
        } else {
            return "NONE";
        }
    }

    /**
     * This methods returns the filter which has for name the String given as
     * argument.
     *
     * @param filterName
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "returns the filter which has for name the String given as argument.")
    public Filter getFilterByName(String filterName) {
        return filterManager.getFilterByName(filterName);
    }

    /**
     * Return true if the changer is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    /**
     * Return true if the loader is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return autochanger.isLoaderConnected();
    }

    /**
     * Return true if the hardware of the changer is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        loader.disconnectLoaderCANbus();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Connect the loader hardware.")
    public void connectLoaderCANbus() {
        loader.connectLoaderCANbus();
    }

    /**
     * Check if a filter name given by an end user at the console is a valid Filter
     * Name. Delegate to filterManager.
     *
     * @param aName
     */
    public void checkFilterName(String aName) {
        filterManager.checkFilterName(aName);
    }

    /**
     * For Whole FCS GUI. Has to be overriden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return carousel.isFilterOnCarousel(filterID) || autochanger.isFilterOnAC(filterID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Store the filter in the carousel and move the empty autochanger to HANDOFF position."
            + " Initial State for AC: a filter at HANDOFF or ONLINE. Final State for AC : empty at HANDOFF.", autoAck = false, timeout = 60000)
    public void storeFilterOnCarousel() {
        try (AutoTimed at = new AutoTimed("storeFilterOnCarousel")) {
            updateStateWithSensors();
            FCSLOG.info(name + " === About to store filter on carousel ===");

            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }

            // carousel must be empty at STANDBY
            if (!carousel.isReadyToGrabAFilterAtStandby()) {
                throw new RejectedCommandException(name + " carousel should be at standby with an empty socket");
            }
            // autochanger should be at HANDOFF or ONLINE
            if (!(autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                throw new RejectedCommandException(name + " autochanger must be at HANDOFF or at ONLINE");
            }

            if (!autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(name + " autochanger must hold the filter");
            }
            // Make sure the AC is free to move
            if (autochanger.isAtOnline()) {
                if (autochanger.getOnlineClamps().isLocked()) {
                    autochanger.getOnlineClamps().unlockClamps();
                }
                // TODO throw an Exception if not Closed ?
                if (autochanger.getOnlineClamps().isClosed()) {
                    if (autochanger.getOnlineClamps().isHomingDone()) {
                        autochanger.getOnlineClamps().openClamps();
                    } else {
                        autochanger.getOnlineClamps().homing();
                    }
                }
            }
            updateStateWithSensors();

            if (autochanger.getOnlineClamps().isOpened()) {
                FCSLOG.info(name + " === autochanger is free to move ===");
            } else {
                throw new FcsHardwareException(name + " autochanger online clamps should be opened");
            }
            carousel.checkDeltaPosition();
            FCSLOG.info(name + " === carousel is ready to receive a filter at standby ===");
            autochanger.getAutochangerTrucks().moveFilterToStandby();
            /* because both subsystems sensors have changed we have to update again */
            updateStateWithSensors();

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on AC = " + autochanger.getFilterID());
                FCSLOG.info("==================================");
                carousel.persistData();

            } else {
                recoveryLockingProcess();
                if (carousel.isHoldingFilterAtStandby()) {
                    carousel.getSocketAtStandby().updateFilterID();
                    carousel.persistData();
                } else {
                    throw new FcsHardwareException(name + ": carousel should be CLAMPED_ON_FILTER when autochanger is "
                            + "at STANDBY with a filter; recovery process didn't work");
                }
            }
            if (autochanger.isAtStandby()) {
                // wait update local protection system
                autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff();
            } else {
                throw new FcsHardwareException(
                        name + ": autochanger should be at STANDBY after " + "moveFilterToStandby() command");
            }
        }
    }

    private void recoveryLockingProcess() {
        try (AutoTimed at = new AutoTimed("recoveryLockingProcess")) {
            if (carousel.getClampXplus().isLocked() && !carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, name);
                carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, autoAck = false, description = "Unclamp filter from carousel, move autochanger to approachStandby position and release the carousel clamps", timeout = 15000)
    public void disengageFilterFromCarousel() {
        try (AutoTimed at = new AutoTimed("disengageFilterFromCarousel")) {
            updateStateWithSensors();
            if (!FilterReadinessState.READY.equals(getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }
            if (!carousel.isAtStandby()) {
                throw new RejectedCommandException(name + "Carousel should be at Standby");
            }
            if (!(autochanger.isAtStandby())) {
                throw new RejectedCommandException("AC should be at Standby Position");
            }
            // Unclamp the filter from the carousel
            carousel.unlockClamps();
            carousel.updateStateWithSensors();
            carousel.waitForProtectionSystemUpdate();

            if (!carousel.isUnclampedOnFilterAtStandby()) {
                String msg = name + "Carousel clamps still locked. Aborting autochanger movement because "
                        + "carousel is still holding the filter. bad state for socketAtStandby: "
                        + carousel.getClampsStateAtStandby() + " should be UNCLAMPED_ON_FILTER";
                raiseAlarm(FcsAlert.HARDWARE_ERROR, msg, name);
                throw new FcsHardwareException(name + msg);
            }
            autochanger.waitForProtectionSystemUpdate();
            // AC begins to leave standby position : Move AC to approach standby position
            autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();

            updateStateWithSensors();
            if (!carousel.isEmptyAtStandby()) {
                raiseAlarm(FcsAlert.HARDWARE_ERROR, "Carousel is still seeing the filter", name);
                throw new FcsHardwareException(
                        name + " aborting autochanger movement because carousel is still seeing filter.");
            }

            // Release the clamps
            // TODO do in parallel? => activate a warning if it fails.

            FcsUtils.asyncRun(() -> {
                try {
                    carousel.releaseClamps();
                } catch (Exception e) {
                    this.raiseWarning(FcsAlert.HARDWARE_ERROR, "cannot release carousel clamps", "carousel");
                }
            });
            updateStateWithSensors();
        }
    }

    // SET FILTER PRECONDITIONS

    public boolean autochangerNotInTravel() {
        return autochanger.isAtHandoff() || autochanger.isAtOnline() || autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return autochanger.getLatches().isClosed() || autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return autochanger.getOnlineClamps().isLocked() || autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !autochanger.isAtOnline() || autochanger.isEmpty() || autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !autochanger.isAtStandby() || autochanger.isEmpty() || autochanger.isHoldingFilter()
                || carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return carousel.isHoldingFilterAtStandby() || carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return autochanger.isAtStandby() || autochanger.isEmpty() || carousel.isReadyToGrabAFilterAtStandby();
    }

    /**
     * move filter to online. command to be executed by mcm. can be executed only
     * when FilterReadinessState is READY, loader not connected, no controller is
     * error, carousel stopped at STANDBY, autochanger stopped at STANDBY, ONLINE or
     * HANDOFF, no sensors in error, if a filter is on AC at HANDOFF or ONLINE =>
     * carousel socket at STANDBY must be READY_TO_CLAMP && autochanger latches have
     * to be CLOSED, if a filter is on AC at STANDBY => carousel or autochanger must
     * hold the filter, if a filter is on AC at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilter(int filterID) {
        setFilterAtHandoffOrOnline(filterID, true);
    }

    /**
     * move filter to online. command to be executed by mcm. can be executed only
     * when FilterReadinessState is READY, loader not connected, no controller is
     * error, carousel stopped at STANDBY, autochanger stopped at STANDBY, ONLINE or
     * HANDOFF, no sensors in error, if a filter is on AC at HANDOFF or ONLINE =>
     * carousel socket at STANDBY must be READY_TO_CLAMP && autochanger latches have
     * to be CLOSED, if a filter is on AC at STANDBY => carousel or autochanger must
     * hold the filter, if a filter is on AC at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoff(int filterID) {
        setFilterAtHandoffOrOnline(filterID, false);
    }

    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter")) {

            updateStateWithSensors();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            FCSLOG.info(name + ": filter to set online:" + filterID);
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    // CHECK it should be valid now
                    // loaderPresenceSensor is invalid
                    // .precondition(!isLoaderConnected(), "loader is connected - can't continue
                    // setFilter")
                    .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name,
                            filterID)
                    .precondition(isFilterInCamera(filterID), "%s: filter %s in out of camera", name, filterID)
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by AC or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY).action(() -> {
                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                            FCSLOG.info(name + ": AC is holding a filter, filter name= "
                                    + autochanger.getFilterOnTrucksName());
                        } else {
                            FCSLOG.info(name + ": AC is not holding filter, filter name on AC"
                                    + autochanger.getFilterOnTrucksName());
                            FCSLOG.info(name + ": filterID: " + filterID + " is on socket:"
                                    + carousel.getFilterSocket(filterID).getId());
                        }
                        /**
                         * /* If the filter we want to move ONLINE is NOT on AC : /* - if AC is at
                         * STANDBY it has to be moved empty to HANDOFF /* - if AC is at HANDOFF or
                         * ONLINE, and another filter is on AC this filter has to be store on carousel,
                         * /* then AC can be moved empty at HANDOFF.
                         */
                        carousel.setControllerPositionSensorTypeEncoderSSI();
                        if (!autochanger.isFilterOnAC(filterID)) {
                            FCSLOG.info(name + ": filterID: " + filterID + " is NOT on AC");
                            if (autochanger.isAtStandby()) {
                                FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                                autochanger.moveEmptyFromStandbyToHandoff();
                            } else {
                                /* autochanger can be at HANDOFF or ONLINE with a filter or empty */
                                /* if a filter is on AC */
                                if (autochanger.isHoldingFilter()) {
                                    FCSLOG.info(
                                            name + ": AC is holding filter: " + autochanger.getFilterOnTrucksName());
                                    FCSLOG.info(name + ": is going to store a filter on carousel");
                                    storeFilterOnCarousel();
                                }
                            }

                            /* Now autochanger is empty at Handoff */
                            if (!(autochanger.isEmpty() && autochanger.isAtHandoff())) {
                                throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                            }
                            // Rotate desired filter to standby
                            carousel.rotateSocketToStandby(carousel.getFilterSocket(filterID).getName());
                            FcsUtils.sleep(100, name);
                            updateStateWithSensors();
                            if (carousel.isAtStandby()) {
                                FCSLOG.info(name + ": carousel is at standby");
                            } else {
                                throw new FcsHardwareException(name + ": carousel should be at standby after "
                                        + "rotateSocketToStandby command.");
                            }

                            /* At this point filterID should be on carousel at STANDBY */
                            if (carousel.getFilterIDatStandby() == filterID) {
                                // Go and grab filter on AC
                                autochanger.waitForProtectionSystemUpdate();
                                // Carousel could have continued rotation due to unbalanced state
                                carousel.checkDeltaPosition();
                                autochanger.grabFilterAtStandby();

                            } else {
                                throw new FcsHardwareException(name + " filterID at standby is "
                                        + carousel.getFilterIDatStandby() + " should be: " + filterID);
                            }
                            updateStateWithSensors();
                            if (!autochanger.isFilterOnAC(filterID)) {
                                throw new FcsHardwareException(name + " filter: " + filterID + " should be now on AC");
                            }
                        }
                        /* now filterID is on AC. AC is at STANDBY or at HANDOFF or ONLINE. */
                        if (autochanger.isAtStandby()) {
                            try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {

                                // Unclamp filter from carousel and move to approach position
                                disengageFilterFromCarousel();
                                updateStateWithSensors();

                                if (toOnline) {
                                    // Move the filter to online position
                                    autochanger.moveAndClampFilterOnline();
                                } else {
                                    // Move the filter to handoff position
                                    autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
                                }
                            }
                        } else if (toOnline) {
                            if (autochanger.isAtHandoff()) {
                                autochanger.moveAndClampFilterOnline();

                            } else if (autochanger.isAtOnline()) {
                                autochanger.lockFilterAtOnline();
                            }
                        } else {
                            if (autochanger.isAtOnline() && autochanger.getOnlineClamps().isOpened()) {
                                try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {
                                    autochanger.getAutochangerTrucks().moveToApproachOnlinePositionWithLowVelocity();
                                    autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
                                }
                            }
                        }
                String filterName = filterManager.getFilterNameByID(filterID);
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
                setFilterDuration = System.currentTimeMillis() - beginTime;
                FCSLOG.info("filter " + filterID + " is " + (toOnline ? "ONLINE" : "HANDOFF")
                        + ". setFilterDuration = " + setFilterDuration);
                this.publishData();
                    }); // END of action
        }
    }

    /**
     * just for mcm testing
     *
     * @param state
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "change subsystem state.")
    public void changeState(String state) {
        updateAgentState(FilterState.valueOf(state));
        FCSLOG.fine("SUBSYSTEM STATE=" + isInState(FilterState.valueOf(state)));
    }

    /**
     * Load a filter from the loader to the camera. The loader must hold a filter at
     * STORAGE position. The autochanger must be empty at HANDOFF, latches open. At
     * the end of this command the filter is held by autochanger at Handoff
     * position, and the loader carrier is empty at STORAGE.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.")
    public void loadFilter() {
        // This command has been tested in January 2020 with Francis
        //
        // TODO check that the camera is in the position "horizontal".
        // to test if carousel is empty at standby is over engineering : it has to be
        // tested by AC before command StoreFilterOnCarousel
        updateStateWithSensors();
        // autochanger trucks must be empty at "Hand-Off" position
        if (!(autochanger.isAtHandoff() && autochanger.isEmpty())) {
            throw new RejectedCommandException(
                    name + " autochanger is not empty at HANDOFF position; can't load a filter.");
        }

        // autochanger latches must be open.
        if (!(autochanger.getLatches().isOpened())) {
            throw new RejectedCommandException("Autochanger latches must be open before loadFilter command.");
        }

        // The Loader carrier is retracted at "Storage" position and the filter is
        // clamped
        if (!(loader.isClampedOnFilter() && loader.isAtStorage())) {
            throw new RejectedCommandException(
                    name + " can't load filter because loader is not holding a filter at storage position.");
        }

        loader.moveFilterFromStorageToHandoff();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();

        autochanger.closeLatches();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();

        loader.openClampAndMoveEmptyToStorage();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();
    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Autochanger at HANDOFF. Loader
     * carrier must be empty at STORAGE. At the end of this command, autochanger is
     * empty at HANDOFF, latches are open, and loader carrier is holding the filter
     * at STORAGE (hooks are clamped).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unload a filter from the camera to the loader. The camera has to be at horizontal position. "
            + "The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. "
            + "\nAt the end of this command, autochanger is empty at HANDOFF, latches are open, "
            + "and loader carrier is holding the filter at STORAGE (hooks are clamped)")
    public void unloadFilter() {
        // This command has been tested in January 2020 with Francis
        // TODO check that the camera is in the position "horizontal"
        updateStateWithSensors();
        // the auto changer trucks must hold a filter at "Hand-Off" position
        if (!(autochanger.isAtHandoff() && autochanger.isHoldingFilter())) {
            throw new RejectedCommandException(
                    name + " autochanger is not holding a filter at STANDOFF position; can't unload a filter.");
        }

        // The Loader carrier must be empty, in storage position, and the loader clamps
        // must be open.
        if (!(loader.isEmpty() && loader.isOpened() && loader.isAtStorage())) {

        }

        loader.moveEmptyToHandoffAndClose();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();

        autochanger.openLatches();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();

        loader.moveFilterFromHandoffToStorage();
        FcsUtils.sleep(50, name);
        updateStateWithSensors();

        // this.disconnectLoaderCANbus();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "check if controllers are in fault")
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer :
     * carousel, autochanger and loader. Read also AC trucks and ONLINE clamps
     * position.
     *
     * @throws FcsHardwareException
     */
    @Override
    public synchronized void updateStateWithSensors() {
        // do not execute if last finished less than 100 ms ago.
        // we keep it at the end of actions but don't repeat it
        // synchronized : parallel operations should do it just once and n-1 will just
        // wait.
        if (System.currentTimeMillis() - lastUpdateStateWithSensors.get() < 100 && !FcsUtils.isSimu()) {
            FCSLOG.info(name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
            return;
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs")) {
            // Carousel
            carousel.updateStateWithSensors();
            if (carousel.isAtStandby()) {
                carousel.getSocketAtStandby().updateFilterID();
            }
            // Autochanger
            autochanger.updateStateWithSensors();
            if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.isHoldingFilter()
                    && autochanger.getOnlineClamps().isLocked()) {
                String filterName = filterManager.getFilterNameByID(autochanger.getFilterID());
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
            } else {
                updateAgentState(FilterState.valueOf("ONLINE_NONE"));
            }
            // update loader sensors is loader is connected on camera
            if (autochanger.isLoaderConnected() && loader.isCANbusConnected()) {
                loader.updateStateWithSensors();
            }
        }
    }

    AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
        autochanger.initializeHardware();
        loader.initializeHardware();
        postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        bridgeToLoader.publishData();
    }

    @Override
    public void postStart() {
        // TODO if we initializeClampsState here => httc580 is not booted, so fcs
        // doesn't start
        // check why
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(name + " is shutting down.");
        bridge.doShutdown();
        if (loader.isCANbusConnected()) {
            bridgeToLoader.doShutdown();
        }
        FCSLOG.info(name + " is shutdown.");
    }
}
