
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutochangerOnlineClamp;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForOnlineClamp;

/**
 * To simulate a controller which opens and closes a ONLINE clamp on the autochanger.
 * @author virieux
 */
public class SimuAutochangerOnlineClampController extends SimuEPOSController implements EPOSControllerForOnlineClamp {

    @ConfigurationParameter(isFinal = true)
    private volatile String onlineClampName;

    private final SimuAutochangerPlutoGateway plutoGateway;

    private AutochangerOnlineClamp onlineClamp;

    private boolean brakeActivatedPub;

    // in the simulation, homing is always done.
    private boolean homingDone = true;

    /**
     * Build a new SimuAutochangerOnlineClampController.
     * @param nodeID
     * @param serialNB
     * @param plutoGateway
     * @param onlineClampName
     */
    public SimuAutochangerOnlineClampController(
            int nodeID, String serialNB,
            SimuAutochangerPlutoGateway plutoGateway,
            String onlineClampName) {
        super(nodeID, serialNB);
        this.onlineClampName = onlineClampName;
        this.plutoGateway = plutoGateway;
    }

    @Override
    public boolean isBrakeActivatedPub() {
        return brakeActivatedPub;
    }

    @Override
    public void setBrakeActivatedPub(boolean brakeActivatedToPublish) {
        this.brakeActivatedPub = brakeActivatedToPublish;
    }

    @Override
    public void init() {
        ComponentLookup lookup = subs.getComponentLookup();
        this.onlineClamp = (AutochangerOnlineClamp) lookup.getComponentByPath(onlineClampName);
    }

    /**
     * Quick and dirty simulation.
     * This will have to be refactorized after tests on the real hardware.
     *
     * @param current
     */
    @Override
    public void writeCurrent(int current) {
        this.targetCurrent = current;
        this.actualCurrent = current;
        if (targetCurrent == onlineClamp.getFinalCurrentToClose()) {

            plutoGateway.simulateAutochangerOnlineClampIsClosed(onlineClamp);

        } else if(targetCurrent == onlineClamp.getCurrentToOpen()) {

            plutoGateway.simulateAutochangerOnlineClampIsOpened(onlineClamp);

        } else if(targetCurrent == onlineClamp.getCurrentToClamp()) {

            plutoGateway.simulateAutochangerOnlineClampIsLocked(onlineClamp);
        }
    }

    @Override
    public void writeTargetPosition(int positionToReach) {
        targetPosition = positionToReach;
        position = positionToReach;
        onlineClamp.updatePosition();

        if (targetPosition == onlineClamp.getTargetPositionToOpen()) {
            plutoGateway.simulateAutochangerOnlineClampIsOpened(onlineClamp);

        } else if (targetPosition == onlineClamp.getTargetPositionToClose()) {
            plutoGateway.simulateAutochangerOnlineClampIsClosed(onlineClamp);
        }
    }

    @Override
    public void activateBrakeAndDisable() {
        brakeActivatedPub = true;
    }

    @Override
    public void enableAndReleaseBrake() {
        brakeActivatedPub = false;
    }

    @Override
    public void doReleaseBrake() {
        brakeActivatedPub = false;
    }

    @Override
    public boolean isBrakeActivated() {
        return brakeActivatedPub;
    }

    @Override
    public void activateBrake() {
        brakeActivatedPub = true;
    }

    @Override
    public void homing() {
    }

    @Override
    public boolean isHomingDone() {
        return homingDone;
    }


}
