package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutochangerLatch;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * To simulate a controller which can open or close a latch in the autochanger.
 * @author virieux
 */
public class SimuAutochangerLatchController extends SimuEPOSController {

    @ConfigurationParameter(isFinal = true)
    private volatile String latchName;

    private AutochangerLatch latch;
    private final SimuAutochangerPlutoGateway plutoGateway;

    /**
     * Build a SimuAutochangerLatchController.
     * @param nodeID
     * @param serialNB
     * @param latchName
     * @param plutoGateway
     */
    public SimuAutochangerLatchController(
            int nodeID, String serialNB,
            String latchName,
            SimuAutochangerPlutoGateway plutoGateway) {
        super(nodeID, serialNB);
        this.mode = CURRENT;
        this.latchName = latchName;
        this.plutoGateway = plutoGateway;
    }

    @Override
    public void init() {
        FCSLOG.debug(name + ":latchName=" + latchName);
        ComponentLookup lookup = subs.getComponentLookup();
        latch = (AutochangerLatch) lookup.getComponentByPath(latchName);
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.targetCurrent = aValue;
        int currentInitial = this.actualCurrent;
        FCSLOG.debug("=>initial current=" + actualCurrent);
        int stepsNB = 10;
        int step = (targetCurrent - this.actualCurrent) / stepsNB;

        for (int i = 1; i <= stepsNB; i++) {
            if (i < stepsNB) {
                this.actualCurrent = (short) (currentInitial + (i * step));
            } //because in the last iteration the actual current could be
            //different from the target value.
            else {
                this.actualCurrent = aValue;
            }
            latch.updateCurrent();
            if (actualCurrent == latch.getCurrentToClose()) {
                FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
                this.plutoGateway.simulateAutochangerLatchIsLocked(latchName);

            } else if (actualCurrent == -latch.getCurrentToClose()) {
                this.plutoGateway.simulateAutochangerLatchIsUnlocked(latchName);
            }
            latch.publishData();
            FCSLOG.debug("i=" + i + ",actualCurrent=" + actualCurrent);
            try {
                Thread.sleep(10);
                if (latch.getHaltRequired().get()) {
                    FCSLOG.debug(name + " STOP simulated latch motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                throw new FcsHardwareException(name + " sleep was interrupted.",ex);
            }

        }
    }

    @Override
    public void stopAction()  {
        writeCurrent((short)0);
    }


}
