package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This represents a Mechanical Item which can be moved by a motor. Generaly we
 * want to execute an action on this mechanical item:
 * - start the action,
 * - read sensors to know if the action is completed,
 * - wait for the end of the action
 *
 * This module provides a general way to start an action and wait until the
 * action is completed. It factorises some code that otherwise would have to be
 * duplicated in each Module that needs such fonctionnalities.
 *
 * This class raises alerts :
 * - FCS005 in halt, quickStop
 *
 * @author virieux
 */
public abstract class MobileItem implements HardwareController, AlertRaiser, SignalHandler, HasLifecycle {

    protected static final Logger FCSLOG = FCSCst.FCSLOG;

    @LookupName
    protected String name;

    @LookupField(strategy = TOP)
    protected Subsystem subs;

    @LookupField(strategy = TREE)
    protected AgentStateService agentStateService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    // TODO verify the need for that boolean
    // (a refactoring within cancelReadingSensors could be helpful)
    protected volatile boolean hasToWaitForEndOfAction = false;
    protected volatile MobileItemAction currentAction;
    protected long commandDuration;
    protected long beginTime;

    protected boolean timeoutExceeded;

    private volatile boolean moving = false;
    // Used because we have to wait for the update from the sensors to know if the
    // action is completed
    protected final Lock lock = new ReentrantLock();
    private final Condition motionCompleted = lock.newCondition();

    @ConfigurationParameter(description = "define intervale time between 2 read of sensors")
    private volatile int readSensorsRate = 250;

    protected ScheduledFuture<?> readSensorsHandle;
    protected final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(2);
    // prendre un Scheduler qui peut me permettre de donner un nom au Thread.

    private final AtomicBoolean haltRequired = new AtomicBoolean(false);
    private final AtomicBoolean stopRequired = new AtomicBoolean(false);

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    /**
     * Return true if the MobileItemMobile is moving.
     *
     * That means that we are waiting for an action to be completed.
     *
     * @return
     */
    public boolean isMoving() {
        return moving;
    }

    // for the simulation
    public AtomicBoolean getHaltRequired() {
        return haltRequired;
    }

    /**
     * This stops the thread reading sensors.
     *
     */
    protected void cancelReadingSensors() {
        lock.lock();
        try {
            FCSLOG.debug(name + " => stop reading sensors");
            motionCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        this.readSensorsHandle.cancel(true);
        FCSLOG.debug(name + " => readingSensors canceled");
    }

    /**
     * Return True if all CAN open devices useful for this MobileItemMobile are
     * ready for an action. As an example, if this mobil item is moved by a
     * controller and sensors are read through a sensors gateway, this method will
     * return True when the controller and the gateway are ready to be used (booted,
     * configured, homing done if needed).
     *
     * @return
     */
    public abstract boolean myDevicesReady();

    /*
     * This method must returns true if the action is completed, false otherwise.
     * It's abstract because it'up to the sub-classes to know when the action is
     * completed.
     */
    public abstract boolean isActionCompleted(MobileItemAction action);

    /**
     * reads sensors or any device and updates state from sensors
     *
     * It can be also reading a position or a current on a motor controller.
     * It's abstract because it'up to the sub-classes to know what to do.
     */
    public abstract void updateStateWithSensorsToCheckIfActionIsCompleted();

    /**
     * starts the action given as argument.
     *
     * It has to be specified in sub-classes. Usually it consist in sending a
     * command to a controller : give a value of current or give a target position.
     *
     * @param action
     */
    public abstract void startAction(MobileItemAction action);

    /**
     * aborts action actually running.
     *
     * It has to be specified in sub-classes. It can be : write a zero value of
     * current or send a stop command to a controller. Must not be waiting for a
     * lock or a condition otherwise executeAction could never complete.
     *
     * @param action
     * @param delay
     */
    public abstract void abortAction(MobileItemAction action, long delay);


    /**
     * end current action.
     *
     * This command is executed systematically at the end of any action
     * in the `finally` clause of executeAction().
     * It has to be specified in sub-classes.
     * It should leave the controller in a safe state for the hardware.
     * Must not be waiting for a lock or a condition otherwise executeAction
     * could never complete.
     *
     * @param action
     */
    public abstract void endAction(MobileItemAction action);

    public abstract void quickStopAction(MobileItemAction action, long delay);

    public abstract void publishData();

    protected void checkReadyForAction() {
        if (!myDevicesReady()) {
            throw new FcsHardwareException(name + ": hardware is not ready to execute ACTION commands.");
        }

        if (agentStateService.isInState(AlertState.ALARM)) {
            throw new FcsHardwareException(name + ": can't execute ACTION commands when subsystem is in ALARM state.");
        }
    }

    /**
     * This executes an action which moves the MobileItem and waits for the end od
     * this action. The MobileItemAction is an ENUM from the fcs.bus package.
     *
     * @param                  action, timeout
     * @param timeoutForAction
     * @throws FailedCommandException if the action can't be completed
     * @throws                        org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected void executeAction(MobileItemAction action, long timeoutForAction) {

        checkReadyForAction();
        beginTime = System.currentTimeMillis();

        lock.lock();

        try {
            this.hasToWaitForEndOfAction = true;

            this.currentAction = action;

            commandDuration = 0;

            timeoutExceeded = false;

            this.moving = true;

            FCSLOG.info(name + ": ===> STARTING ACTION:" + action.toString());

            startAction(action);

            readSensorsUntilActionIsCompleted(action, timeoutForAction);

            waitForEndOfAction(action);

            // At this stage, the action can be completed with success or a
            // HALT signal has been received.
            if (haltRequired.get()) {
                abortAction(action, 0);
                FCSLOG.info(name + "=> Received an ABORT command for action :" + action.toString());

            } else if (stopRequired.get()) {
                quickStopAction(action, 0);
                FCSLOG.info(name + "=> Received a STOP command for action :" + action.toString());

            } else if (isActionCompleted(action)) {
                FCSLOG.info(name + " ===> ACTION COMPLETED : " + action.doneString());
            }

            if (timeoutExceeded) {
                FCSLOG.info(name + " ===> ACTION NOT COMPLETED : " + action.doneString());
                String msg = name + ":" + action.toString()
                        + " exceeded timeout for this task: duration=" + commandDuration + ",timeout=" + timeoutForAction;
                FCSLOG.error(msg);
                throw new FailedCommandException(msg);
            }

        } catch (Exception ex) {
            String msg = name + " ===> ERROR during ACTION : " + action.toString() + ex.getMessage();
            FCSLOG.error(msg + ex.getMessage());
            FCSLOG.error(action.getFailureMsg());
            //TODO do a quickStop
            throw new FailedCommandException(msg, ex);


        } finally {

            FCSLOG.debug(name + ": finally in executeAction.");

            this.moving = false;
            this.hasToWaitForEndOfAction = false;
            this.motionCompleted.signalAll();
            haltRequired.set(false);
            stopRequired.set(false);
            lock.unlock();

            endAction(action);
            commandDuration = System.currentTimeMillis() - beginTime;

            publishData();
            FCSLOG.info(name + " ===> END ACTION:" + action.toString()
                    + " duration = " + commandDuration);
            subs.publishSubsystemDataOnStatusBus(
                    new KeyValueData(name + "_" + action + "_duration", commandDuration));
        }
    }

    /**
     * Start reading the sensors at a fixed rate (scheduler.scheduleAtFixedRate)
     * until the action is completed or we received a halt signal or the timeout for
     * this action is past.
     *
     * @param action
     *
     * @param timeout
     */
    public void readSensorsUntilActionIsCompleted(final MobileItemAction action,
            final long timeout) {

        final Runnable readSensors;
        readSensors = () -> {
            try {
                commandDuration = System.currentTimeMillis() - beginTime;
                updateStateWithSensorsToCheckIfActionIsCompleted();

                boolean actionCompleted = isActionCompleted(action);

                // we got a HALT signal, we have to stop reading the sensors.
                if (haltRequired.get() || stopRequired.get()) {
                    hasToWaitForEndOfAction = false;
                    FCSLOG.info(name + ":" + action.toString() + " ABORT or STOP REQUESTED FOR ACTION "
                            + "BY ABORT or STOP COMMAND");
                    cancelReadingSensors();

                    // action is completed with success.
                } else if (actionCompleted) {
                    hasToWaitForEndOfAction = false;
                    FCSLOG.info(name + ":" + action.toString() + " ACTION COMPLETED");
                    cancelReadingSensors();

                    // action is not completed but timeout is over.
                } else if (commandDuration >= timeout) {
                    hasToWaitForEndOfAction = false;
                    timeoutExceeded = true;
                    FCSLOG.info(name + ":" + action.toString() + " ACTION NOT COMPLETED during allocated time = " + timeout);
                    cancelReadingSensors();

                    // action is not completed, we have to continue reading sensors.
                } else {
                    FCSLOG.info(name + ":" + action.name() + " not completed... / duration=" + commandDuration);
                }

            } catch (Exception ex) {
                hasToWaitForEndOfAction = false;
                cancelReadingSensors();
                FCSLOG.error(name + ": ERROR in reading sensors during action " + action.name(), ex);
                throw new FcsHardwareException(name + ": ERROR in reading sensors during action " + action.name(), ex);
            }
        };
        this.readSensorsHandle = scheduler.scheduleAtFixedRate(readSensors, readSensorsRate, readSensorsRate, TimeUnit.MILLISECONDS);
    }

    /**
     * This method waits until the action is completed. This methods is called by
     * executeAction and has already acquired the lock.
     *
     * @param action
     */
    protected void waitForEndOfAction(MobileItemAction action) {
        while (hasToWaitForEndOfAction) {
            try {
                FCSLOG.info(name + " waiting for end of " + action.toString());
                motionCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex.toString());
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF ACTION");
    }

    private void halt(MobileItemAction action, long delay) {
        if (action == null) {
            FCSLOG.warning(name + ": no current action running => nothing to abort.");
            return;
        }
        FCSLOG.debug(name + ": ABORTING ACTION " + action.toString() + "within delay=" + delay);
        abortAction(action, delay);
        moving = false;
    }

    private void quickStop(MobileItemAction action, long delay) {
        if (action == null) {
            FCSLOG.warning(name + ": no current action running => nothing to stop.");
            return;
        }
        FCSLOG.debug(name + ": QUICKSTOP for" + action.toString() + "within delay=" + delay);
        quickStopAction(action, delay);
        moving = false;
    }

    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when the
     * MobileItemMobile is stopped. cf checkStopped
     */
    @Override
    public void shutdown() {
        FCSLOG.info(name + " is shutting down.");
        scheduler.shutdown();
    }

    /**
     *
     * @param signal
     * @return
     */
    // TODO refactorize to get rid of boolean moving with is redondant with
    // haltRequired
    // and perhaps Command.State
    @Override
    public TreeWalkerDiag signal(Signal signal) {
        SignalLevel sl = signal.getLevel();
        FCSLOG.info(sl.toString());

        if (!this.moving) {
            FCSLOG.warning(name + " is not moving; nothing to stop.");

        } else {

            switch (signal.getLevel()) {
            case HALT:
                FCSLOG.info(name + " HALT required");
                this.haltRequired.set(true);
                halt(currentAction, signal.getTimeHint());
                break;
            case STOP:
                FCSLOG.info(name + " STOP required");
                this.stopRequired.set(true);
                quickStop(currentAction, signal.getTimeHint());
                break;
            default:
                assert false;

            }
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public void checkStopped() throws HardwareException {
        if (moving) {
            throw new HardwareException(false, name + " is moving, has to be stopped before a shutdown.");
        }
    }

}
