package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.framework.HardwareController;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION;

import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.common.PT100Interface;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console.
 *
 *
 * @author virieux
 *
 */
public class Carousel extends MobileItem implements HardwareController, FilterHolder, AlertRaiser {

    private static final long serialVersionUID = -2376279469784152348L;

    @LookupField(strategy = TOP)
    protected Subsystem subs;

    @LookupField(strategy = TREE)
    private MainModule main;


    private int rotationTimeout;

    private long profileVelocity;
    private long profileAcceleration;
    private long profileDeceleration;

    /**
     * carousel position given by carouselControl.
     */
    protected int position = 0;
    private int relativeTargetPosition;

    /**
     * an absolute target position to reach in a rotation is modified each time
     * a rotation command has been launched.
     */
    protected int absoluteTargetPosition;

    /**
     * a difference between a target position and the actual position of the
     * carousel this is used to know if the rotation is completed or not.
     */
    @ConfigurationParameter(description = "", units = "Carousel encoder steps", range = "0..2181120")
    protected volatile int deltaPosition = 1000;

    protected int readVelocity = 0;
    protected int readCurrent = 0;

    @ConfigurationParameter(description = "below this value of deltaPosition, carousel position at STANDBY is correct", units = "Carousel encoder steps", range = "0..2181120")
    private volatile int minStandbyDeltaPosition = 100;

    @ConfigurationParameter(description = "over this value of deltaPosition, carousel position at STANDBY is NOT correct", units = "Carousel encoder steps", range = "0..2181120")
    private volatile int maxStandbyDeltaPosition = 130;

    /**
     * 0.1 tour = 436224 pas (2Pi = 4260 X 1024) 1 socket = 0.2 tour = 1/5 tour
     * = 872448 pas
     */
    private final int fullTurn = 4362240;
    private final int halfTurn = fullTurn / 2;

    @ConfigurationParameter(description = "velocity in slow mode", units = "Carousel velocity unit", range = "0..5000")
    public volatile int slowVelocity = 500;

    @ConfigurationParameter(description = "acceleration in slow mode", units = "Carousel acceleration unit", range = "0..5000")
    public volatile int slowAcceleration = 200;

    @ConfigurationParameter(description = "deceleration in slow mode", units = "Carousel acceleration unit", range = "0..5000")
    public volatile int slowDeceleration = 200;

    @ConfigurationParameter(
            description = "in milliseconds; timeout for the rotation in slow mode", units = "milliseconds", range = "0..300000")
    public volatile int slowRotationTimeout = 100000;

    @ConfigurationParameter(description = "velocity in fast mode", units = "Carousel velocity unit", range = "0..5000")
    public volatile int fastVelocity = 3400;

    @ConfigurationParameter(description = "acceleration in fast mode", units = "Carousel acceleration unit", range = "0..5000")
    public volatile int fastAcceleration = 2000;

    @ConfigurationParameter(description = "deceleration in fast mode", units = "Carousel acceleration unit", range = "0..5000")
    public volatile int fastDeceleration = 1000;

    @ConfigurationParameter(
            description = "in milliseconds; timeout for the rotation in fast mode", units = "milliseconds", range = "0..300000")
    public volatile int fastRotationTimeout = 20000;

    @ConfigurationParameter(units = "milliseconds", range = "0..5000")
    public volatile long timeToUpdateProtectionSystem = 2000;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*clampXminusController")
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*clampXplusController")
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    @LookupField(strategy = TREE, pathFilter = ".*carouselController")
    protected EPOSControllerForCarousel carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = TREE, pathFilter = ".*hyttc580")
    private TTC580Interface hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    @LookupField(strategy = TREE, pathFilter = ".*ai814")
    protected ADCInterface ai814;

    @LookupField(strategy = TREE, pathFilter = "pt100")
    private PT100Interface pt100;

    /*To be able to know if the autochanger holds a filter. */
    @LookupField(strategy = SIBLINGS, pathFilter = "autochanger")
    private FilterHolder autochanger;

    /**
     * A map to store the sockets by their names. The key of this map is the
     * socket name. This map is built by Toolkit during INITIALISATION.
     *
     */
    @LookupField(strategy = CHILDREN)
    protected final Map<String, CarouselSocket> socketsMap = new TreeMap<>();

    @LookupField(strategy = SIBLINGS, pathFilter = "tcpProxy")
    protected BridgeToHardware tcpProxy;

    private boolean initialized = false;
    protected boolean clampsStateInitialized = false;
    protected boolean homingDone = false;

    /**
     * value of field socketAtStandbyID is read on httc580
     */
    private int socketAtStandbyID;
    protected CarouselSocket socketAtStandby;

    /**
     * Outputs to Auto-Changer interlocks to be read on httc580 and publish on
     * the bus.
     */
    private boolean OUT_CS;
    private boolean OUT_CFC;
    private boolean OUT_CF0;
    private boolean OUT_CF1;
    private boolean OUT_CS_InError;
    private boolean OUT_CFC_InError;
    private boolean OUT_CF0_InError;
    private boolean OUT_CF1_InError;

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarousel.class, name);
    }

    /**
     *
     * @return true if CANopen devices are booted and initialized and homing has
     * been done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        // TODO test also if homing has been done.
        return this.myDevicesReady();
    }

    /**
     *
     * @return true if clamp state is initialized for all clamps
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if clamp state is initialized for all clamps.")
    public boolean isClampsStateInitialized() {
        return clampsStateInitialized;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if carouselController homing is done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     * used by tests.
     *
     * @return
     */
    public int getFullTurn() {
        return fullTurn;
    }

    public Map<String, CarouselSocket> getSocketsMap() {
        return socketsMap;
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return carousel position.", alias = "printPosition")
    public int getPosition() {
        return position;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    /**
     * This method returns the clampX- which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXminus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXplus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     * otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")
    public boolean isAtStandby() {
        // TODO this is not consistent with method CarouselSocket.isAtStandby();
        return socketAtStandbyID >= 1 && socketAtStandbyID <= 5;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one.
     * Otherwise return null.
     *
     * @return
     */
    public CarouselSocket getSocketAtStandby() {
        return socketAtStandby;
    }

    /**
     *
     * @return ID of socket at STANDBY
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "return ID of socket at STANDBY")
    public int getSocketAtStandbyID() {
        return socketAtStandbyID;
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns name of filter at STANDBY position  or"
            + " NO_FILTER if there is no filter at STANDBY.")
    public String getFilterAtStandbyName() {
        if (socketAtStandby == null) {
            return NO_FILTER;
        } else {
            return socketAtStandby.getFilterName();
        }
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns id of filter at STANDBY position  or"
            + " NO_FILTER if there is no filter at STANDBY.")
    public int getFilterIDatStandby() {
        if (socketAtStandby == null) {
            return 0;
        } else {
            return socketAtStandby.getFilterID();
        }
    }

    /**
     * return true if filterID in on carousel.
     *
     * @param filterID
     * @return
     */
    public boolean isFilterOnCarousel(int filterID) {
        return socketsMap.values().stream().anyMatch((socket) -> (socket.getFilterID() == filterID));
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To change filterID on socket which ID is given as argument.")
    public void changeFilterID(int filterID, int socketID) {
        if (socketID < 1 || socketID > 5) {
            throw new IllegalArgumentException(socketID + ": bad value - enter a digit between 1 and 5");
        }
        if (this.isFilterOnCarousel(filterID)) {
            int sockID = this.getFilterSocket(filterID).getId();
            throw new IllegalArgumentException(filterID + " filter already on carousel on socket" + sockID);
        }
        String socketName = SOCKET_NAME + socketID;
        socketsMap.get(socketName).setFilterID(filterID);
    }

    /**
     * This methods returns true if the autochanger is holding a filter at
     * STANDBY position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if Autochanger is holding filter at STANDBY.")
    public boolean isAutochangerHoldingFilter() {
        /**
         * To be done when a switch is connected : read the switch sensor and
         * update a boolean of the class like : autochangerHoldingFilter.
         */
        autochanger.updateStateWithSensors();
        return autochanger.isHoldingFilter();
    }

    /**
     * initialize carousel hardware after initialization. to be executed if
     * during boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Initialize carousel hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        tcpProxy.bootProcess();
        this.postStart();
    }

    /**
     * Executed when all components HasLifeCycle of subsystem have been checked.
     */
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (carouselController.isBooted()) {
            initializeRotationController();
            profileVelocity = carouselController.readProfileVelocity();
            profileAcceleration = carouselController.readParameter(ProfileAcceleration);
            profileDeceleration = carouselController.readParameter(ProfileDeceleration);
            if (profileVelocity == slowVelocity) {
                rotationTimeout = slowRotationTimeout;
            } else {
                rotationTimeout = fastRotationTimeout;
            }
        }
        if (clampXminusController.isBooted()) {
            initializeClampController(clampXminusController);
        }
        if (clampXplusController.isBooted()) {
            initializeClampController(clampXplusController);
        }
        this.initializeClampsState();
        FCSLOG.fine(name + " END postStart.");
    }

    private void initializeRotationController() {
        try {
            carouselController.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", carouselController.getName(), ex);
        }
    }

    /**
     * check that controller is correctly configured. This command can't be
     * executed in CarouselClamp.postStart because the 5 clamps Xminus share the
     * same controller and idem for Xplus.
     */
    private void initializeClampController(EPOSController controller) {
        try {
            /*
             * check that parameters on CPU are those on configuration
             */
            controller.initializeAndCheckHardware();
            if (!controller.isInMode(CURRENT)) {
                this.raiseAlarm(HARDWARE_ERROR, "  is not in CURRENT mode.",
                        controller.getName());
            }

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller ",
                    controller.getName(), ex);
        }
    }

    /**
     * homing of rotation controller to be done before each rotation
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "CarouselHoming")
    public void homingold() {
        homingDone = false;
        setControllerPositionSensorTypeEncoderSSI();
        int ssiPosition = carouselController.readSSIPosition();
        carouselController.setPositionSensorTypeSinusIncrementalEncoder();
        try {
            carouselController.defineAbsolutePosition(ssiPosition);
            carouselController.changeMode(PROFILE_POSITION);
        } finally {
            /* whatever happends during defineAbsolutePosition, controller must be disabled */
            carouselController.disableOperation();
        }
        position = carouselController.readPosition();
        if (position == 0) {
            throw new FcsHardwareException(name
                    + " carousel position is 0. This should not happen after homing.");
        }
        homingDone = true;
        publishData();
    }

    /**
     * homing of rotation controller to be done before each rotation
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "CarouselHoming")
    public void homing() {
        setControllerPositionSensorTypeEncoderSSI();
        int ssiPosition = carouselController.readSSIPosition();
        FCSLOG.info(name + " ssiPosition = " + ssiPosition);
        carouselController.doSetSinusIncrementalEncoder();
        try {
            carouselController.defineAbsolutePosition(ssiPosition);
            carouselController.changeMode(PROFILE_POSITION);
        } finally {
            /* whatever happends during defineAbsolutePosition, controller must be disabled */
            carouselController.goToSwitchOnDisabled();
        }
        position = carouselController.readPosition();
        if (Math.abs(position) < 1) {
            throw new FcsHardwareException(name
                    + " |position| < 1. This should not happen after homing.");
        }
        homingDone = true;
        publishData();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Disable carousel controller, set the position sensor to TypeEncoderSSI, and check that the ssi position is correct.")
    public void setControllerPositionSensorTypeEncoderSSI() {
        carouselController.setPositionSensorTypeEncoderSSI();
        position = carouselController.readPosition();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "slow down profile velocity, acceleration and deceleration in carousel controller.")
    public void setSlowMode() {
        carouselController.changeProfileVelocity(slowVelocity);
        profileVelocity = slowVelocity;
        carouselController.writeParameter(ProfileAcceleration, slowAcceleration);
        profileAcceleration = slowAcceleration;
        carouselController.writeParameter(ProfileDeceleration, slowDeceleration);
        profileDeceleration = slowDeceleration;
        rotationTimeout = slowRotationTimeout;
        publishData();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "raise profile velocity, acceleration and deceleration in carousel controller.")
    public void setFastMode() {
        carouselController.changeProfileVelocity(fastVelocity);
        profileVelocity = fastVelocity;
        carouselController.writeParameter(ProfileAcceleration, fastAcceleration);
        profileAcceleration = fastAcceleration;
        carouselController.writeParameter(ProfileDeceleration, fastDeceleration);
        profileDeceleration = fastDeceleration;
        rotationTimeout = fastRotationTimeout;
        publishData();
    }

    public void waitForProtectionSystemUpdate() {
        waitForStateUnclampedOnFilter(timeToUpdateProtectionSystem);
    }

    /**
     * wait until carousel is unclamped and empty at standby
     *
     * @param timeout after this delay don't wait anymore.
     */
    public void waitForStateUnclampedOnFilter(long timeout) {
        long cmdBeginTime = System.currentTimeMillis();
        long waitTime = 0;
        while (!isUnclampedOnFilterAtStandby() && waitTime < timeout) {
            FCSLOG.info(name + " waiting for local protection system update. waitTime = " + waitTime);
            FcsUtils.sleep(100, name);
            waitTime = System.currentTimeMillis() - cmdBeginTime;
            updateSocketAtStandbyWithSensors();
        }
    }

    private void checkHomingDone() {
        //TODO move that to homing ?
        if (!isHomingDone()) {
            throw new FcsHardwareException(name + " homing has to be done before motion.");
        }
    }

    public void checkSensorTypeIncremental() {
        int sensorType = carouselController.readPositionSensorType();
        if (sensorType != 8) {
            throw new FcsHardwareException(
                    name + " PositionSensorType has to be set to Incremental Encoder before motion.");
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData("carousel", createStatusDataPublishedByCarousel()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the
     * STATUS bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setPosition(position);
        status.setPositionSensorType(carouselController.getPositionSensorType());
        status.setCurrent(readCurrent);
        status.setVelocity(readVelocity);
        status.setHomingDone(homingDone);
        status.setAtStandby(isAtStandby());
        status.setSocketAtStandbyID(socketAtStandbyID);
        if (this.isEmptyAtStandby()) {
            status.setFilterAtStandbyName(NO_FILTER);
        } else {
            status.setFilterAtStandbyName(getFilterAtStandbyName());
        }
        if (isAtStandby()) {
            status.setSocketAtStandbyName(socketAtStandby.getName());
            status.setEmptyAtStandby(this.isEmptyAtStandby());
        } else if (socketAtStandbyID == 0) {
            status.setSocketAtStandbyName("NO_SOCKET_AT_STANDBY");
        } else if (socketAtStandbyID == 7) {
            status.setSocketAtStandbyName("ERROR_READING_ID");
        }
        status.setProfileAcceleration(profileAcceleration);
        status.setProfileDeceleration(profileDeceleration);
        status.setProfileVelocity(profileVelocity);
        status.setRotationTimeout(rotationTimeout);
        status.setOUT_CF0(OUT_CF0);
        status.setOUT_CF1(OUT_CF1);
        status.setOUT_CFC(OUT_CFC);
        status.setOUT_CS(OUT_CS);
        status.setOUT_CF0_InError(OUT_CF0_InError);
        status.setOUT_CF1_InError(OUT_CF1_InError);
        status.setOUT_CFC_InError(OUT_CFC_InError);
        status.setOUT_CS_InError(OUT_CS_InError);
        status.setTemperature1(pt100.getTemperature(1));
        status.setTemperature2(pt100.getTemperature(2));
        status.setTemperature3(pt100.getTemperature(3));
        status.setTemperature4(pt100.getTemperature(4));
        status.setSensor1(ai814.getInput(0));
        status.setSensor2(ai814.getInput(1));
        status.setSensor3(ai814.getInput(2));
        status.setSensor4(ai814.getInput(3));
        status.setSensor5(ai814.getInput(4));
        status.setSensor6(ai814.getInput(5));
        status.setSensor7(ai814.getInput(6));
        status.setSensor8(ai814.getInput(7));
        return status;
    }

    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the
     * filterPosition when one of its sockets is at standby filterPosition and
     * this socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() {

        if (this.isRotating()) {
            return false;
        }
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and carousel is
     * stopped
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() {
        if (this.isRotating()) {
            return false;
        }
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Returns true if carousel is rotating
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if carousel is rotating.")
    public boolean isRotating() {
        return isMoving();
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update carousel position in reading controller.")
    public void updatePosition() {
        if (this.homingDone) {
            try {
                this.position = carouselController.readPosition();
            } catch (SDORequestException ex) {
                FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
            }
            this.publishData();
        } else {
            throw new RejectedCommandException(
                    "Carousel homing not done : " + "do carousel homing before updatePosition.");
        }
    }

    /**
     * Updates the field readCurrent in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update field current in reading controller.")
    public void updateCurrent() {
        try {
            this.readCurrent = carouselController.readCurrent();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() {

        this.tcpProxy.updatePDOData();
        PDOData pdoStore = tcpProxy.getPDOData();

        FCSLOG.finest(name + ":pdoStore=" + pdoStore.toString());
        socketsMap.values().stream().forEach(socket -> {
            socket.updateState();
        });
    }

    /**
     * In carousel, sensors are updated from PDOs. PDOs are received 2 by 2 :
     * 0x180 + ttc580 nodeID for clamp at standby. (ttc580 pdo1) 0x280 + ttc580
     * nodeID for a clamp not at standby. (ttc580 pdo2) PDO2 contains socketID
     * not at STANDBY in turns. Exemple : if socket1 is at STANDBY - first sync
     * returns pdo1 with socket1 values and pdo2 with socket2 values - second
     * sync returns pdo1 with socket1 values and pdo2 with socket3 values -
     * third sync returns pdo1 with socket1 values and pdo2 with socket4 values
     * - fourth sync returns pdo1 with socket1 values and pdo2 with socket5
     * values
     *
     * socketID is coded in each PDO. After FCS start we need to send 4 sync to
     * know all clamps state.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamps state in sending 5 sync.")
    public void initializeClampsState() {
        FCSLOG.info(name + " Initializing clamps state....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        byte count = 1;
        while (count <= 5) {
            FCSLOG.info(name + " sync no " + count);
            tcpProxy.updatePDOData();
            hyttc580.updateFromPDO(tcpProxy.getPDOData());
            updateState();
            count++;
        }
        clampsStateInitialized = true;
        publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamp offsets.")
    public void initializeAndCheckClampsOffset() {
        FCSLOG.info(name + " Initializing clamps offsets....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        socketsMap.values().stream()
                .forEach(socket -> {
                    byte socketId = (byte) socket.getId();
                    long offset2 = hyttc580.readOffset2SDO(socketId);
                    socket.checkAndUpdateOffset2(offset2);
                });
        publishData();
    }

    /**
     *
     * @param filterID
     * @return null if filter is not on carousel, else return socket where
     * filter is stored.
     */
    public CarouselSocket getFilterSocket(int filterID) {
        CarouselSocket socket = null;
        for (CarouselSocket sock : socketsMap.values()) {
            if (sock.getFilterID() == filterID) {
                return sock;
            }
        }
        return socket;
    }

    /**
     * Release clamps at standby position to get ready to clamp again.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Release clamps at standby position to get ready to clamp again")
    public void releaseClamps() {
        updateStateWithSensors();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't release clamps when no socket is halted at standby position.");
        } else {
            socketAtStandby.releaseClamps();
        }
    }

    /**
     * Unlocks the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Unlock the clamps at STANDBY.")
    public void unlockClamps() {
        updateSocketAtStandbyWithSensors();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + " can't unlock clamps while a socket is not halted at standby position.");
        } else {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        }
    }

    /**
     * Return true if the filterID given as argument is at STANDBY position.
     *
     * @param filterID
     * @return
     */
    public boolean isAtStandby(int filterID) {
        return socketAtStandby.getFilterID() != 0 && socketAtStandby.getFilterID() == filterID;
    }

    public void releaseClampsContact() {
        // TODO;
    }

    @Override
    public boolean myDevicesReady() {
        return carouselController.isBooted() && carouselController.isInitialized();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ ROTATION COMMANDS *********************************
     */
    /**
     * ****************************************************************
     */
    /**
     * Check if carousel rotation is permitted.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     * (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation() {
        String message;
        if (!initialized) {
            throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        }

        // commented out in July 2017 because no TTC-580
        // updateState();
        if (this.isAtStandby() && socketAtStandby.isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }

        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochanger is not at HANDOFF position.");
        }
    }

    /**
     * Rotate to position newPos. Condition LPM to rotate : AP2 & AP3
     *
     * @param newPos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Rotate carousel to a new absolute position.", timeout = 60000)
    public void rotateToAbsolutePosition(int newPos) {
        /* read temperatures before rotation */
        if (pt100.isBooted()) {
            pt100.updateTemperatures();
        }
        boolean atPosition = (Math.abs(newPos - position)) < deltaPosition;
        if (!atPosition) {
            /* subsystem must not be in ALARM state*/
            checkReadyForAction();

            /* carouselController should be initialized */
            carouselController.checkInitialized();

            /* carouselController should not be in fault */
            carouselController.checkFault();

            /* autochanger must be at ONLINE or HANDOFF */
            if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
                throw new RejectedCommandException(name
                        + " can rotate only if autochanger is at Handoff or at Online  " + newPos);
            }

            /* a homing must be done before every motion */
            homing();
            //TODO a checkHomingDone is already done is homing. 
            checkHomingDone();

            int diffPos = newPos - position;
            if (Math.abs(newPos - position) <= halfTurn) {
                absoluteTargetPosition = newPos;
            } else {
                if (diffPos < 0) {
                    absoluteTargetPosition = newPos + fullTurn;
                } else {
                    absoluteTargetPosition = newPos - fullTurn;
                }
            }

            FCSLOG.info(name + " is at position: " + position + "; about to rotate to absolute position: "
                    + absoluteTargetPosition);

            // rotate the carousel
            this.executeAction(ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION, rotationTimeout);
            main.updateAgentState(FcsEnumerations.FilterState.valueOf("CAROUSEL_STOPPED"));
            /* switch back controller to PositionSensorTypeEncoderSSI LSSTCCSFCS-223 */
            setControllerPositionSensorTypeEncoderSSI();
            /* Read position can be done now because controller sensor type is EncoderSSI. */
            position = carouselController.readPosition();
            initializeClampsState();
            updateStateWithSensors();
        }
    }

    /**
     * rotate carousel within a relative position given as argument. If carousel
     * is at initial position initPos, if the argument given to this method is
     * relativePos, at the end of the motion, carousel position should be
     * initPos + relativePos.
     *
     * @param relativePos relative position.
     * @param timeout
     */
    public void rotateToRelativePosition(int relativePos, long timeout) {
        /* carouselController should be initialized */
        carouselController.checkInitialized();

        /* carouselController should not be in fault */
        carouselController.checkFault();

        /* autochanger must be at ONLINE or HANDOFF */
        if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
            throw new RejectedCommandException(name
                    + " can rotate only if autochanger is at Handoff or at Online.");
        }

        /* a homing must be done before every motion */
        homing();
        checkHomingDone();

        relativeTargetPosition = relativePos;
        // rotate the carousel
        this.executeAction(ROTATE_CAROUSEL_TO_RELATIVE_POSITION, timeout);
        /* switch back controller to PositionSensorTypeEncoderSSI LSSTCCSFCS-223 */
        setControllerPositionSensorTypeEncoderSSI();
        /* Read position can be done now because controller sensor type is EncoderSSI. */
        position = carouselController.readPosition();
        initializeClampsState();
        updateStateWithSensors();
    }

    /**
     * Rotate carousel to move a socket which name is given as argument to
     * STANDBY position. This methods computes the shortest way to go to STANDBY
     * position.
     *
     * @param socketName
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move a socket which name is given as argument to STANDBY position.", alias = "moveSocketToStandby", timeout = 50000)
    public void rotateSocketToStandby(String socketName) {
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socketToMove = this.socketsMap.get(socketName);
        if (!socketToMove.isAtStandby()) {
            int requiredPosition = socketToMove.getStandbyPosition();
            if (position != requiredPosition) {
                FCSLOG.info(name + " is at position: " + position + "; about to rotate to position: " + requiredPosition);
                rotateToAbsolutePosition(requiredPosition);
            }
            checkDeltaPosition();
        }
        FCSLOG.info(name + ":" + socketName + " is at STANDBY position on carousel.");
    }

    public void checkDeltaPosition() {
        if (socketAtStandby != null) {
            socketAtStandby.updateDeltaPosition();
            long deltaPos = socketAtStandby.getDeltaPosition();
            if (Math.abs(deltaPos) > maxStandbyDeltaPosition) {
                this.raiseAlarm(HARDWARE_ERROR, " delta position at standby is over " + maxStandbyDeltaPosition, name);
            } else if (Math.abs(deltaPos) > minStandbyDeltaPosition) {
                this.raiseWarning(HARDWARE_ERROR, " delta position at standby is over " + minStandbyDeltaPosition, name);
            }
        }
    }

    /**
     * Rotate carousel full turn. A number of turns can be given as argument.
     * This turns carousel in relative position.
     *
     * @param nbTurn number of turns to rotate
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel full turn. A number of turns can be given as argument.")
    public void rotateFullTurn(double nbTurn) {
        if (Math.abs(nbTurn) > 3) {
            throw new IllegalArgumentException("nbTurn=" + nbTurn + " should not be more than 3");
        }
        //slowRotationTimeout is the time maximum to rotate 2 sockets at slow speed; 
        //for a full turn (5 sockets), 5/2 should be the right factor, but experience showed that 2 is enough.
        long timeout = (int) Math.abs(nbTurn) * 2 * slowRotationTimeout;
        int rpos = (int) (fullTurn * nbTurn);
        rotateToRelativePosition(rpos, timeout);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            return carouselController.isTargetReached();

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            // FCSLOG.info(name + " => Checking for carousel position to complete action - "
            //         + " CurrentCarousel Position = " + this.position + " Target Position = "
            //         + this.absoluteTargetPosition + " Delta Position expected = " + deltaPosition);
            // return Math.abs(this.position - this.absoluteTargetPosition) < deltaPosition;
            // TODO validate change once the hardware tests are done ?
            return carouselController.isTargetReached();
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        // updateStateWithSensors updates also position, readCurrent, readVelocity, etc...
        FCSLOG.debug(name + " position=" + this.position);
        updateStateWithSensors();
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or
     * ROTATE_CAROUSEL_TO_RELATIVE_POSITION. This : - enable controller, - make
     * controller go to required position previously written in field
     * absoluteTargetPosition or relativeTargetPosition, - writeControlWord "3F"
     * or "7F" on controller depending if we want to go to a relative position
     * or to an absolute position.
     *
     * @param action
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {

        main.updateAgentState(FcsEnumerations.FilterState.valueOf("ROTATING"));

        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            carouselController.enableAndWriteRelativePosition(this.relativeTargetPosition);

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            carouselController.enable();
            FcsUtils.sleep(250, name);
            carouselController.writeTargetPosition(this.absoluteTargetPosition);
            carouselController.writeControlWord(0x3F);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        this.carouselController.stopPosition();
        /* a tempo to let time to the motion command to complete. */
        FcsUtils.sleep(500, name);
        this.carouselController.disableOperation();
    }

    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        this.carouselController.stopPosition();
        /* a tempo to let time to the motion command to complete. */
        FcsUtils.sleep(250, name);
        this.carouselController.disableOperation();
        /* do not read position now because Position Encoder Type is still Incremental */
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.carouselController.stopPosition();
        this.carouselController.disableOperation();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of ROTATION COMMANDS **************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ************ methods which override FilterHolder *** ***********
     */
    /**
     * ****************************************************************
     */
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Override
    public boolean isHoldingFilter() {
        return this.isHoldingFilterAtStandby();
    }

    // TODO find something more clever to do here.
    @Override
    public boolean isNotHoldingFilter() {
        return !this.isHoldingFilterAtStandby();
    }

    @Override
    public boolean isAtHandoff() {
        return false;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the ID of filter at STANDBY")
    @Override
    public int getFilterID() {
        return socketAtStandby.getFilterID();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read sensors and update state", timeout = 2000)
    @Override
    public void updateStateWithSensors() {
        long cmdBeginTime = System.currentTimeMillis();
        if (!clampsStateInitialized) {
            throw new FcsHardwareException(
                    name + ": clamps state not initialized. " + " Please launch command initializeClampsState.");
        }
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        tcpProxy.updatePDOData();
        updateState();
//        readAndUpdateOutputInterlocks();
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateStateWithSensors duration = " + duration);
        publishData();
    }

    /**
     * Overridden method from FilterHolder
     *
     * @return false
     */
    @Override
    public boolean isAtOnline() {
        //carousel is never at ONLINE
        return false;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of methods which override FilterHolder ********
     */
    /**
     * ****************************************************************
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update socket at standby state from hyttc580 data.", timeout = 2000)
    public void updateSocketAtStandbyWithSensors() {
        long cmdBeginTime = System.currentTimeMillis();
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        tcpProxy.updatePDOData();
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " hyttc580.updateFromPDO(tcpProxy.updatePDOData()) duration = " + duration);
        updateSocketAtStandbyState();
        duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateStateWithSensorsAtStandby duration = " + duration);
    }

    public void updateSocketAtStandbyState() {
        long cmdBeginTime = System.currentTimeMillis();
        this.socketAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo1());
        if (socketAtStandbyID >= 1 && socketAtStandbyID <= 5) {
            socketAtStandby = socketsMap.get("socket" + socketAtStandbyID);
            socketAtStandby.updateState();
            socketAtStandby.updateFilterID();
        }
        publishData();
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateSocketAtStandbyState duration = " + duration);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update state from hyttc580 data.", timeout = 2000)
    public void updateState() {
        long cmdBeginTime = System.currentTimeMillis();
        updateSocketAtStandbyState();
        //doesn't read again controller CPU because the following values have been refreshed by PDO.
        position = carouselController.getPosition();
        readCurrent = carouselController.getCurrent();
        readVelocity = carouselController.getVelocity();
        // The use of the scheduler is required in order to leave readCurrent thread as soon as possible.
        subs.getScheduler().schedule(() -> {
            updateSocketNotAtStandby();
        }, 0, TimeUnit.SECONDS);
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateState duration = " + duration);
    }

    public void updateSocketNotAtStandby() {
        long cmdBeginTime = System.currentTimeMillis();
        int socketNotAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo2());
        if (socketNotAtStandbyID >= 1 && socketNotAtStandbyID <= 5) {
            CarouselSocket socketToUpdate = socketsMap.get("socket" + socketNotAtStandbyID);
            socketToUpdate.updateState();
        }
        long duration = System.currentTimeMillis() - cmdBeginTime;
        FCSLOG.info(name + " updateSocketNotAtStandby duration = " + duration);
    }

    public FilterClampState getClampsStateAtStandby() {
        if (socketAtStandby == null) {
            return FilterClampState.UNDEFINED;
        }

        return socketAtStandby.getClampsState();
    }

    public boolean isEmptyAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isEmpty();
    }

    public boolean isUnclampedOnFilterAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isUnclampedOnFilter();
    }

    /**
     * Read on hyttc580 by SDO the outputs to autochanger interlocks and update
     * output signals CS, CFC, CF0, CF1. (see
     * https://confluence.slac.stanford.edu/pages/viewpage.action?spaceKey=LSSTCAM&title=CAN+Open+communication+with+carousel+modules)
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read on hyttc580 by SDO the outputs to autochanger interlocks "
            + "CS, CFC, CF0, CF1.", timeout = 2000)
    public void readAndUpdateOutputInterlocks() {
        long sdo = hyttc580.readInterlocksSDO();
        //sdo returned is on 2 bytes, signals CS, CFC, CF0, CF1 are on the first byte
        byte signals = (byte) (sdo >> 8 & 0xFF);

        byte val = (byte) ((signals >> 7) & 1);
        byte nval = (byte) ((signals >> 6) & 1);
        OUT_CS = val == 1;
        OUT_CS_InError = val == nval;

        val = (byte) ((signals >> 5) & 1);
        nval = (byte) ((signals >> 4) & 1);
        OUT_CFC = val == 1;
        OUT_CFC_InError = val == nval;

        val = (byte) ((signals >> 3) & 1);
        nval = (byte) ((signals >> 2) & 1);
        OUT_CF1 = val == 1;
        OUT_CF1_InError = val == nval;

        val = (byte) ((signals >> 1) & 1);
        nval = (byte) (signals & 1);
        OUT_CF0 = val == 1;
        OUT_CF0_InError = val == nval;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read on pt100 the temperatures", timeout = 2000)
    public void readTemperatures() {
        pt100.updateTemperatures();
    }

}
