package org.lsst.ccs.subsystems.fcs;


import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is used with the final hardware and prototype in which we have
 * online clamps to hold the filter at online position.
 * It's not used for Single Filter Test.
 *
 * CAMERA PROTECTION SYSTEM
 * Signal comming from carousel :
 * CS : Carousel stopped on socket
 * CF0: No filter in carousel socket
 * CF1: Filer in carousel socket
 * CFC: Filter clamped on carousel socket
 *
 * Signal coming from loader :
 * LPS: Loader at Storage position (loader is connected)
 * LRH: Loader at Handoff position, closed but not clamped
 *
 * Local Protection Module
 * Signals emitted by AC:
 * enableRailLin1 enable master linear rail (X+)
 * enableRailLin2 enable slave linear rails (X-)
 * enableClamps
 * enableLatches
 *
 * Sensors in AC1 and AC2 and not in PROTO :
 *
 * loaderPresenceSensors * lockOutSensors
 *
 *
 * @author virieux
 */
public class Autochanger implements HardwareController, FilterHolder, AlertRaiser, HasLifecycle {

    @LookupField(strategy = TOP)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    private ConfigurationService configurationService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    protected String name;

    @LookupField(strategy = SIBLINGS, pathFilter = "tcpProxy")
    private BridgeToHardware tcpProxy;

    private final PlutoGatewayInterface plutoGateway;

    @LookupField(strategy = SIBLINGS, pathFilter = "filterManager")
    private FilterManager filterManager;

    @LookupField(strategy = SIBLINGS, pathFilter = "filterIdentificator")
    protected FilterIdentificator filterIdentificator;

    /**
     * CPS signal : loader_LPS - loader_LPS is a signal coming from loader
     * loaderPresenceSensors are complementary sensors on autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "loaderConnectedSensors")
    private ComplementarySensors loaderConnectedSensors;

    /**
     * CPS signal : loader_LRH
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "loaderHoldingFilterSensors")
    private ComplementarySensors loaderHoldingFilterSensors;

    // carouselHoldingFilterSensors
    /**
     * CPS signal : carousel_CFC Carousel stop on socket
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "carouselHoldingFilterSensors")
    private ComplementarySensors carouselHoldingFilterSensors;

    /**
     * CPS signal : carousel_CS Carousel stop on socket
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "carouselStoppedAtStandbySensors")
    private ComplementarySensors carouselStoppedAtStandbySensors;

    /**
     * CPS signal : carousel_CF0 No filter in carousel socket
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "carousel_CF0Sensors")
    private ComplementarySensors carousel_CF0Sensors;

    /**
     * CPS signal : carousel_CF1 Filter in carousel socket
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "carousel_CF1Sensors")
    private ComplementarySensors carousel_CF1Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AF0_Sensors")
    private ComplementarySensors OUT_AF0_Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AF1_Sensors")
    private ComplementarySensors OUT_AF1_Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AF3_Sensors")
    private ComplementarySensors OUT_AF3_Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AP1_Sensors")
    private ComplementarySensors OUT_AP1_Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AP2_Sensors")
    private ComplementarySensors OUT_AP2_Sensors;

    /**
     * CPS signal emitted by autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "OUT_AP3_Sensors")
    private ComplementarySensors OUT_AP3_Sensors;


    /**
     * Local Protection Module signals
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "lpmLinearRail1Status")
    private DigitalSensor lpmLinearRail1Status;

    @LookupField(strategy = SIBLINGS, pathFilter = "lpmLinearRail2Status")
    private DigitalSensor lpmLinearRail2Status;

    @LookupField(strategy = SIBLINGS, pathFilter = "lpmOnlineClampsStatus")
    private DigitalSensor lpmOnlineClampsStatus;

    @LookupField(strategy = SIBLINGS, pathFilter = "lpmLatchesStatus")
    private DigitalSensor lpmLatchesStatus;

    @LookupField(strategy = CHILDREN, pathFilter = "autochangerTrucks")
    private AutochangerTwoTrucks autochangerTrucks;

    @LookupField(strategy = CHILDREN, pathFilter = "latches")
    protected AutochangerTwoLatches latches;

    @LookupField(strategy = CHILDREN, pathFilter = "onlineClamps")
    private AutochangerThreeOnlineClamps onlineClamps;

    /**
     * Loader Presence Sensor : true if loader is connected on autochanger
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "loaderPresenceSensors")
    private ComplementarySensors loaderPresenceSensors;

    /**
     * lockOut Sensor : (or keyLock) if the key is turned out the lock sensors
     * return true. If lockOut is true, motion is forbidden.
     */
    @LookupField(strategy = SIBLINGS, pathFilter = "lockOutSensors")
    private ComplementarySensors lockOutSensors;

    // plage de valeurs acceptables pour l'horizontalité
    // max acceptable pour la différence entre les 2 inclinometres
    // si on lit la même valeur sur les 2 c'est OK
    @LookupField(strategy = CHILDREN, pathFilter = "inclinometerXminus")
    private Inclinometer inclinometerXminus;

    @LookupField(strategy = CHILDREN, pathFilter = "inclinometerXplus")
    private Inclinometer inclinometerXplus;

    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = SIBLINGS, pathFilter = "carousel")
    private FilterHolder carousel;

    @LookupField(strategy = SIBLINGS, pathFilter = "loader")
    private FilterHolder loader;

    @ConfigurationParameter(description = "UNDEFINED", units = "milliseconds")
    public volatile long timeToUpdateProtectionSystem = 2000;

    @ConfigurationParameter(description = "time to wait in ms between activateBrake and disableOperation for online clamps", units = "milliseconds")
    private volatile int waitTimeForBrakeOC = 20;

    @ConfigurationParameter(description = "time to wait in ms between activateBrake and disableOperation for linear rails", units = "milliseconds")
    private volatile int waitTimeForBrakeLR = 20;

    @Persist
    protected int filterOnTrucksID;

    public Autochanger(PlutoGatewayInterface plutoGateway) {
        this.plutoGateway = plutoGateway;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    public int getWaitTimeForBrakeOC() {
        return waitTimeForBrakeOC;
    }

    public int getWaitTimeForBrakeLR() {
        return waitTimeForBrakeLR;
    }


    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutoChanger.class, "autochangerGeneral");
    }

    /**
     * Return true if an error has been detected on sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }

    /**
     * returns trucks to be able to send commands to trucks in fcs subsystem.
     *
     * @return trucks
     */
    public AutochangerTwoTrucks getAutochangerTrucks() {
        return autochangerTrucks;
    }

    /**
     * returns onlineClamps to be able to send commands to onlineClamps in fcs
     * subsystem.
     *
     * @return onlineClamps
     */
    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    /**
     * return latches
     *
     * @return
     */
    public AutochangerTwoLatches getLatches() {
        return latches;
    }

    /**
     * If autochanger holds a filter, returns filterID, else returns 0.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "If autochanger holds a filter, return filterID, else return 0.")
    @Override
    public int getFilterID() {
        return filterOnTrucksID;
    }

    /**
     * Used for simulator only.
     *
     * @param filterOnTrucksID
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        this.filterOnTrucksID = filterOnTrucksID;
    }

    /**
     *
     * @param filterID
     * @return return true if filter with filterID is on AC
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return true if filter with filterID is on AC")
    public boolean isFilterOnAC(int filterID) {
        return filterOnTrucksID == filterID && filterOnTrucksID != 0;
    }

    /**
     *
     * @param filterID
     * @return true if filterID is on Autochanger and trucks are ONLINE and latches
     *         are closed.
     */
    public boolean isFilterONLINE(int filterID) {
        return isFilterOnAC(filterID) && this.isAtOnline() && this.isHoldingFilter();
    }

    /**
     * If autochanger holds a filter, return filter name else return null.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "If a filter is in autochanger, return filter name else return NO_FILTER.")
    public String getFilterOnTrucksName() {
        if (filterOnTrucksID != 0) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return NO_FILTER;
        }
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean is
     * being updated and waits for a response from a sensor, this methods waits
     * until loaderConnected is updated. If the field loaderConnected is not being
     * updated, it returns immediatly the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        return loaderPresenceSensors.isOn() && !loaderPresenceSensors.isInError();
    }

    /**
     * TODO when autochanger is in standalone mode this method has to answer the
     * value of the fake carousel sensor.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby() {
        return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }

    /**
     * Return true if loader is connected and holding a filter. This command doesn't
     * read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader is connected and holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff() {
        // TODO change that with reading of new loaderPresence Sensors
        // return isLoaderConnected() && loader.isHoldingFilter();
        return loader.isHoldingFilter();
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is
     *         hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return tcpProxy.allDevicesBooted();
    }

    /**
     * @return if all autochanger CANopen devices are booted, identified and
     *         nitialized and homing of the controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and "
            + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return tcpProxy.allDevicesBooted() && isInitialized();
    }

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are
     * initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized() && latches.isInitialized()
                && onlineClamps.isInitialized();
    }

    public boolean isLinearRailMotionAllowed() {
        return this.lpmLinearRail1Status.isOn() && this.lpmLinearRail2Status.isOn();
    }

    public void waitForProtectionSystemUpdate() {
        waitForRailMotionAllowed(timeToUpdateProtectionSystem);
    }

    /**
     * wait until linear rail motion is authorized by local protection system.
     * because there are temporisation in local protection system, there is a
     * delay between sensors authorize us to move and local protection system
     * authorizes.
     * @param timeout after this delay don't wait anymore.
     */
    private void waitForRailMotionAllowed(long timeout) {
        long beginTime = System.currentTimeMillis();
        long waitTime = 0;
        while (!isLinearRailMotionAllowed() && waitTime < timeout) {
            FCSLOG.info(name + " waiting for local protection system update. waitTime = " + waitTime);
            FcsUtils.sleep(100, name);
            waitTime = System.currentTimeMillis() - beginTime;
            this.plutoGateway.updateValues();
        }
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (plutoGateway.isBooted()) {
            initializeGateway();
            try {
                // In postStart if a RuntimeException is thrown, fcs cannot starts. To avoid
                // this we have to catch the exception and raise an ALERT of level ALARM.
                updateStateWithSensors();
            } catch (Exception ex) {
                this.raiseAlarm(HARDWARE_ERROR, name + " couldn't updateStateWithSensors in postStart ", ex);
            }
        } else {
            plutoGateway.raiseAlarmIfMissing();
        }
        FCSLOG.fine(name + " END postStart.");
    }

    private void initializeGateway() {
        try {
            this.plutoGateway.initializeAndCheckHardware();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize gateway", plutoGateway.getName(), ex);
        }
    }

    /**
     * For end users, for tests and in engineering mode. This command can be used to
     * recover after a missing hardware during fcs startup. For example when fcs was
     * started before hardware power up. Check all hardware and publish data.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For end users, for tests and in engineering mode. "
            + "This command can be used to recover after a missing hardware "
            + "during fcs startup. For example, when fcs was started before hardware "
            + "power up. Check all hardware and publish data.", timeout = 5000)
    public void initializeHardware() {
        FCSLOG.info(name + " BEGIN initializeHardware");

        // check that all hardware is booted and identified.
        tcpProxy.bootProcess();

        try {
            this.postStart();
            autochangerTrucks.postStart();
            onlineClamps.postStart();
            latches.postStart();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " couldn't initialize autochanger", ex);
        }
        FCSLOG.info(name + " END initializeHardware");
    }

    /**
     * Check if Local Protection Module allows linear rails motions.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if Local Protection Module allows linear rails motion.")
    public void checkLinearRailMotionAllowed() {
        if (lpmLinearRail1Status.isOn() && lpmLinearRail2Status.isOn()) {
            FCSLOG.fine(name + ":linear rails motion allowed by Local Protection Module");
        } else {
            throw new RejectedCommandException(name + ": linear rails motion NOT allowed by Local Protection Module.");
        }
    }

    /**
     * Check if Local Protection Module allows latches motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if Local Protection Module allows latches motion.")
    public void checkLatchMotionAllowed() {
        if (lpmLatchesStatus.isOn()) {
            FCSLOG.fine(name + ":latches open or close allowed by Local Protection Module");
        } else {
            throw new RejectedCommandException(
                    name + ": latches open or close NOT allowed by Local Protection Module.");
        }
    }

    /**
     * Check if Local Protection Module allows online clamps motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if Local Protection Module allows online clamps motion.")
    public void checkOnlineClampMotionAllowed() {
        if (lpmOnlineClampsStatus.isOn()) {
            FCSLOG.fine(name + ":Online clamps open or close allowed by Local Protection Module");
        } else {
            throw new RejectedCommandException(
                    name + ": Online clamps open or close NOT allowed by Local Protection Module.");
        }
    }

    /**
     * Check if latches can be opened. If latches can't be openened because the
     * conditions for opening are not filled, this command throws a
     * RejectedCommandException. If latches are in error, it throws a
     * FcsHardwareException.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches() {
        FCSLOG.info(name + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new FcsHardwareException(name + ": latches are in ERROR state - can't unlockClamps latches.");

        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't open latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");

        } else if (autochangerTrucks.isAtStandby()) {
            checkConditionsForOpeningLatchesAtStandby();

        } else if (autochangerTrucks.isAtHandoff()) {
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }

    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(
                    name + ": latches are not intialized. " + "Please initialize hardware first.");
        }
    }

    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position */
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(name + " carousel is holding filter at STANDBY => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter " + "- can't open latches.");
        }
    }

    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position */
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(name + " loader is holding filter at HANDOFF => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter " + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches() {

        FCSLOG.info(name + " checking conditions for closing latches.");

        updateStateWithSensors();
        String message;
        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new RejectedCommandException(name + ": latches are in ERROR state - can't close latches.");

        } else if (isEmpty()) {
            message = name + ": no filter in autochanger - can't close latches.";
            throw new RejectedCommandException(message);

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + ": autochanger is not " + "at handoff position neither at standby - can't close latches.");
        }
    }

    /**
     * log if actions on online clamps are allowed, throws an exception otherwise.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "log if actions on online clamps are allowed,"
            + "throws an exception otherwise.")
    public void checkConditionsForActioningOnlineClamps() {
        if (!isAtOnline()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when trucks are not at ONLINE.");
        }
        if (isEmpty()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when no filter on trucks.");
        }
        FCSLOG.info(name + " autochangerTrucks are at ONLINE, a filter is there : actions are allowed.");
    }

    /**
     * This methods checks that filter can be moved by trucks. Returns immediately
     * if autochanger trucks are empty. Throws RejectedCommandException if a filter
     * is in trucks AND held both by autochanger and (loader or carousel).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion() {

        String msg = name + " can't move trucks because ";
        this.updateStateWithSensors();

        if (this.isSensorsInError()) {
            handleSensorsError();

        } else if (isEmpty()) {
            FCSLOG.info(name + " trucks are empty - can move");

        } else if (!this.onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    name + " : a filter " + "is in trucks and ONLINE clamps are NOT OPENED.");

        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);

        } else if (autochangerTrucks.isAtHandoff()) {
            FCSLOG.info(name + " : trucks are AT HANDOFF ");
            checkFilterSafetyAtHandoff(msg);

        } else if (this.autochangerTrucks.isAtOnline()) {
            FCSLOG.info(name + " : trucks are at ONLINE ");
            checkFilterSafetyAtOnline(msg);
            // TODO the tests below could be removed because they have been already done in
            // checkFilterSafetyAtStandby
        } else if (latches.isHoldingFilter() && carousel.isHoldingFilter()) {
            throw new RejectedCommandException(msg + " carousel is holding filter too.");

        } else if (!latches.isHoldingFilter() && !carousel.isHoldingFilter()) {
            throw new RejectedCommandException(msg + " neither autochanger neither carousel "
                    + "is holding filter. Close latches or carousel clamps.");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtStandby(String message) {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message + "both carousel and autochanger are holding filter at STANDBY");

        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "neither carousel nor autochanger are holding filter at STANDBY");

        } else if (!(latches.isClosed() || latches.isOpened())) {
            throw new RejectedCommandException(message + " latches must be opened or closed.");

        } else {
            FCSLOG.info(name + " filter safe at STANDBY - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtHandoff(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(
                    message + "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(name + " filter safe at HANDOFF - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE. It can be
     * move : - if no filter is at STANDBY (otherwise risk of collision) - AND
     * ONLINE clamps are opened, - AND latches are CLOSED.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtOnline(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                    + "- can't move trucks");
        }

        if (isHoldingFilter() && !onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    message + "onlineClamps have to be opened on filter at ONLINE - can't move trucks.");
        }

        if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(
                    message + "neither latches nor onlineClamps are holding filter at ONLINE");
        } else {
            FCSLOG.info(name + " filter safe at ONLINE - can move");
        }
    }

    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when error
     * is detected in sensors.
     *
     * @throws FcsHardwareException
     */
    private void handleSensorsError() {
        boolean transientError = false;
        String msg = name + " error detected in sensors :";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
            transientError = autochangerTrucks.isPositionSensorErrorsTransient();
        }
        if (this.latches.isFilterEngagedInError()) {
            transientError = transientError || isLatchesErrorTransient();
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        if (transientError) {
            this.raiseWarning(AC_SENSOR_ERROR, msg + " - can be a transient error. ", name);
        } else {
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg);
        }
    }

    private boolean isLatchesErrorTransient() {
        return Math.abs(autochangerTrucks.getPosition() - 984000) < 5000;
    }

    /**
     * This method reads all the sensors and updates the trucks, latches and online
     * clamps state.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {
        plutoGateway.checkInitialized();
        this.plutoGateway.updateValues();
        updateState();
        updateFilterOnTrucksID();
        this.publishData();
    }

    /**
     * update filterOnTrucksID
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "update filterOnTrucksID.")
    public void updateFilterOnTrucksID() {
        if (!latches.isEmpty()) {
            filterOnTrucksID = filterIdentificator.getFilterId();
        } else {
            filterOnTrucksID = 0;
        }
    }

    private void updateState() {
        this.autochangerTrucks.updateState();
        this.latches.updateState();
        this.onlineClamps.updateState();
        this.filterIdentificator.updateFilterId();
        if (this.isAtStandby() && carouselStoppedAtStandbySensors.isOn()) {
            carousel.updateStateWithSensors();
        }
    }

    /**
     * Update FCS state and FCS readyness state and publish on the status bus. Check
     * that Autochanger hardware is ready to be operated and moved. This means that
     * : - all CAN open devices are booted, identified and initialized, -
     * initialization has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSStateToReady() {
        if (this.isCANDevicesReady() && latches.isInitialized() && autochangerTrucks.isInitialized()
                && onlineClamps.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /****************************************************************************
     * Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return true if a filter is in trucks and latches are CLOSED.")
    @Override
    public boolean isHoldingFilter() {
        return latches.isHoldingFilter();
    }

    @Override
    public boolean isNotHoldingFilter() {
        return latches.isOpened();
    }

    /**
     * Return true if autochanger trucks are at HANDOFF. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoff();
    }

    /**
     * Return true if autochanger trucks are at STANDBY. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandby();
    }

    /***************************************************************************
     * end of Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * return true if autochanger trucks are at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnline();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks position is around approachStandbyPosition. "
            + "This command doesn't read again the sensors.")
    public boolean isAtApproachStandbyPosition() {
        return Math.abs(autochangerTrucks.getPosition() - autochangerTrucks.getApproachStandbyPosition()) < 1000;
    }

    /****************************************************************************
     * AutochangerTrucks ENGINEERING1 ACTION commands
     ***************************************************************************
     */

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks are
     * already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Handoff position.")
    public void goToHandOff() {
        autochangerTrucks.goToHandOff();
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        autochangerTrucks.goToOnline();
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks are
     * already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        autochangerTrucks.goToStandby();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "goToOnline with filter then adjust position then close and finally lock clamps", autoAck = false, timeout = 20000)
    public void moveAndClampFilterOnline() {
        autochangerTrucks.moveAndClampFilterOnline();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "go slowly to appproachStandbyPosition ")
    public void moveToApproachStandbyPositionWithLowVelocity() {
        autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "go fast to appproachStandbyPosition ")
    public void moveToApproachStandbyPositionWithHighVelocity() {
        autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "go fast to appproachOnlinePosition ")
    public void moveToApproachOnlinePositionWithHighVelocity() {
        autochangerTrucks.moveToApproachOnlinePositionWithHighVelocity();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "go slowly to appproachOnlinePosition ")
    public void moveToApproachOnlinePositionWithLowVelocity() {
        autochangerTrucks.moveToApproachOnlinePositionWithLowVelocity();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "align slave and move empty from approachStandby to Handoff")
    public void alignSlaveAndMoveEmptyFromApproachToHandoff() {
        autochangerTrucks.alignSlaveAndMoveEmptyFromApproachToHandoff();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "move slowly to standby position")
    public void moveToStandbyWithLowVelocity() {
        autochangerTrucks.moveToStandbyWithLowVelocity();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "move fast to handoff position")
    public void moveToHandoffWithHighVelocity() {
        autochangerTrucks.moveToHandoffWithHighVelocity();
    }


    /**
     * Move a filter to standby position step by step. A first step with a high
     * speed to the approachStandbyPosition. a second step to standby position with
     * low speed.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "move filter to approachPosition with high speed and move to STANDBY with lowSpeed.", autoAck = false, timeout = 30000)
    public void moveFilterToStandby() {
        autochangerTrucks.moveFilterToStandby();
    }

    /**
     * move slave truk to masterPosition. When moving along the rails with slave
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignement. This consists in : setting slaveController mode to
     * PROFILE_POSITION, go to absolute masterPosition pos, and then set
     * slaveController mode back to MASTER_ENCODER mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Align slave controller position to master controller position.", timeout = 20000)
    public void alignSlave() {
        autochangerTrucks.alignSlave();
    }

    /**
     * change speed & acceleration profile to slow
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "change ProfileVelocity parameter to lowSpeed and slowProfileAcceleration and slowProfileDeceleration")
    public void slowTrucksProfile() {
        autochangerTrucks.slowProfile();
    }

    /**
     * change speed and acceleration profile to fast
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "change ProfileVelocity and ProfileAcceleration and ProfileDeceleration parameters to highSpeed")
    public void fastTrucksProfile() {
        autochangerTrucks.fastProfile();
    }

    /**
     * Do homing of both trucks controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Do homing for both  controllers.")
    public void homingTrucks() {
        autochangerTrucks.homing();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "move to approach position with high velocity then move to STANDBY with low velocity")
    public void dockingAtStandbyPositionWithoutFilter() {
        autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        autochangerTrucks.moveToStandbyWithLowVelocity();
    }

    /**
     * just for tests //TODO get rid of this method after tests
     */
    public void disableYminus() {
        onlineClamps.disableYminus();
    }

    /****************************************************************************
     * END OF AutochangerTrucks ENGINEERING1 commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerLatches ENGINEERING1 ACTION commands
     ***************************************************************************
     */

    /**
     * Open latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Open latches.")
    public void openLatches() {
        latches.open();
    }

    /**
     * Close latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Close latches.")
    public void closeLatches() {
        latches.close();
    }

    /****************************************************************************
     * END OF AutochangerLatches ENGINEERING1 commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerOnlineClamps ENGINEERING1 ACTION commands
     ***************************************************************************
     */

    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For
     * final pruducts AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Opens the 3 online clamps in mode PROFILE_POSITION.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClamps() {
        onlineClamps.openClamps();
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = " close clamps in mode PROFILE_POSITION for AC1 and AC2",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClamps() {
        onlineClamps.closeClamps();
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps
     * have to be CLOSED. At the end of this action, the clamps are CLOSED but a
     * strong pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Locks clamps : closed with a strong pressure (high current).",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void lockClamps() {
        onlineClamps.lockClamps();
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to
     * decrease pressure on the clamps. The clamps have to be LOCKED. At the end
     * of this action, the clamps are CLOSED with a small pressure of the clamp
     * hardware on the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "unlocks clamps : decreases current in controller to decrease pressure.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockClamps() {
        onlineClamps.unlockClamps();
    }

    /**
     * do homing of the 3 clamps
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller",
            timeout = 6000)
    public void homingClamps() {
        onlineClamps.homing();
    }

    /****************************************************************************
     * END OF AutochangerOnlineClamps ENGINEERING1 commands
     ***************************************************************************
     */


    /**
     * Return true if there is no filter in the autochanger. This command
     * doesn't read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }




    /**
     * Moves empty to standby position and close latches on filter at standby on
     * carousel.
     *
     * @throws RejectedCommandException if latches are not OPEN.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, alias = "grabFilter",
            description = "Move autochanger trucks to STANDBY position, close latches and stay at STANDBY")
    public void grabFilterAtStandby() {
        updateState();
        if (!latches.isOpened()) {
            throw new RejectedCommandException(name + " latches must be open to execute grabFilterAtStandby");
        }
        FCSLOG.info(name + " === About to move empty to standby position ===");
        autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        autochangerTrucks.moveToStandbyEmptyWithLowVelocity();
        FCSLOG.info(name + " ===> filter on AC after moveToStandbyEmptyWithLowVelocity =" + filterOnTrucksID + " should be != 0");
        updateFilterOnTrucksID();
        latches.close();
        FCSLOG.info(name + ": filter " + filterOnTrucksID + " is now on autochanger");
    }



    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "open latches then go to approachStandbyPosition with low speed "
            + "and move to HANDOFF with highSpeed.", timeout = 20000, autoAck = false)
    public void moveEmptyFromStandbyToHandoff() {
        if (!this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(name + "%s : can't open latches if carousel is not holding it.");
        }
        if (latches.isClosed()) {
            // TODO this is already done in latches.open(). Is it useful to do it here ?
            checkConditionsForOpeningLatches();
            latches.open();
        }
        waitForProtectionSystemUpdate();
        autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
        autochangerTrucks.alignSlaveAndMoveEmptyFromApproachToHandoff();
        updateFilterOnTrucksID();
    }

    /**
     * set a new value for the acquisition frequency of linear rail controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "set a new value for the acquisition "
            + "frequency of linear rail controllers current in milliseconds.")
    public void setFastAcqRateLinearRails(int rate) {
        configurationService.change("acTruckXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("acTruckXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategoriesAs("timers:fastRails");
    }

    /**
     * set a new value for the acquisition frequency of ONLINE clamps controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "set a new value for the acquisition "
            + "frequency of ONLINE clamps controllers current, in milliseconds.")
    public void setFastAcqRateOnlineClamps(int rate) {
        configurationService.change("onlineClampXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampYminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategoriesAs("timers:fast");
    }

    /**
     * speed up online clamps current monitoring for the 3 clamps
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "speed up online clamps current monitoring for the 3 clamps")
    public void increaseCurrentMonitoring() {
        configurationService.loadCategories("timers:fast");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "slow down online clamps current monitoring for the 3 clamps")
    public void decreaseCurrentMonitoring() {
        configurationService.loadCategories("timers:slow");
    }

    /**
     * increase current monitoring for the 2 trucks
     */
    public void increaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:fastRails");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    public void decreaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:slowRails");
    }

    /**
     * Read the filterID sensors at STANDBY to check filterID.
     *
     * @return
     */
    private int identifyFilterAtStandby() {
        return filterIdentificator.getFilterId();
    }

    /**
     * Creates an Object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChanger createStatusDataPublishedByAutoChanger() {
        StatusDataPublishedByAutoChanger status = new StatusDataPublishedByAutoChanger();
        status.setLoaderConnectedSensorValue(this.loaderConnectedSensors.isOn()); // loader_LPS
        status.setLoaderConnectedSensorsInError(this.loaderConnectedSensors.isInError());// loader_LPS
        status.setLoaderHoldingFilterAtHandoff(loaderHoldingFilterSensors.isOn()); // loader_LRH
        status.setLoaderHoldingFilterAtHandoffInError(loaderHoldingFilterSensors.isInError());
        status.setCarouselHoldingFilter(carouselHoldingFilterSensors.isOn()); // carousel_CFC
        status.setCarouselHoldingFilterInError(carouselHoldingFilterSensors.isInError());
        status.setCarousel_CS(carouselStoppedAtStandbySensors.isOn()); // carousel_CS
        status.setCarousel_CS_InError(carouselStoppedAtStandbySensors.isInError());// carousel_CS
        status.setCarousel_CF0(carousel_CF0Sensors.isOn());
        status.setCarousel_CF0_InError(carousel_CF0Sensors.isInError());
        status.setCarousel_CF1(carousel_CF1Sensors.isOn());
        status.setCarousel_CF1_InError(carousel_CF1Sensors.isInError());
        status.setEnableRailLin1(lpmLinearRail1Status.isOn());
        status.setEnableRailLin2(lpmLinearRail2Status.isOn());
        status.setEnableClamps(lpmOnlineClampsStatus.isOn());
        status.setEnableLatches(lpmLatchesStatus.isOn());
        status.setLoaderPresence(loaderPresenceSensors.isOn());
        status.setLoaderPresenceInError(loaderPresenceSensors.isInError());
        status.setInclinaisonXminus(inclinometerXminus.getInclinaison());
        status.setInclinaisonXplus(inclinometerXplus.getInclinaison());
        status.setOUT_AF0(OUT_AF0_Sensors.isOn());
        status.setOUT_AF1(OUT_AF1_Sensors.isOn());
        status.setOUT_AF3(OUT_AF3_Sensors.isOn());
        status.setOUT_AP1(OUT_AP1_Sensors.isOn());
        status.setOUT_AP2(OUT_AP2_Sensors.isOn());
        status.setOUT_AP3(OUT_AP3_Sensors.isOn());
        status.setOUT_AF0_InError(OUT_AF0_Sensors.isInError());
        status.setOUT_AF1_InError(OUT_AF1_Sensors.isInError());
        status.setOUT_AF3_InError(OUT_AF3_Sensors.isInError());
        status.setOUT_AP1_InError(OUT_AP1_Sensors.isInError());
        status.setOUT_AP2_InError(OUT_AP2_Sensors.isInError());
        status.setOUT_AP3_InError(OUT_AP3_Sensors.isInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(
                new KeyValueData("autochangerGeneral", createStatusDataPublishedByAutoChanger()));
    }

    /**
     * to be used in setFilter
     */
    public void lockFilterAtOnline() {
        if (!this.isAtOnline()) {
            throw new FcsHardwareException(name + "is not at ONLINE");
        }
        if (this.isEmpty()) {
            throw new FcsHardwareException(name + " must be loaded with a filter");
        }
        onlineClamps.lockFilterAtOnline();
    }

}
