
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;

/**
 *
 * @author virieux
 */
public class CanOpenEPOSCarousel extends CanOpenEPOS implements EPOSControllerForCarousel {

    /**
     * define cobid to be able to read values from PDO data.
     */
    protected int cobid1 = this.nodeID + 0x180;
    protected int cobid2 = this.nodeID + 0x280;

    /**
     * PDO1 represents : current: 2 bytes positionActualValue: 4 bytes statusWord: 2
     * bytes
     */
    private long pdo1 = 0L;

    /**
     * PDO2 represents : FollowingError: 2 bytes VelocityActualValue 4 bytes
     * StatusWord: 2 bytes
     */
    private long pdo2 = 0L;

    public CanOpenEPOSCarousel(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }




    /**
     * set a pdo1 for simulation
     * @param pdo1
     */
    public void setPdo1(long pdo1) {
        this.pdo1 = pdo1;
    }

    /**
     * set a pdo2 for simulation
     * @param pdo2
     */
    public void setPdo2(long pdo2) {
        this.pdo2 = pdo2;
    }

    /**
     * for simulation
     * @return pdo1
     */
    public long getPdo1() {
        return pdo1;
    }

    /**
     * for simulation
     * @return pdo2
     */
    public long getPdo2() {
        return pdo2;
    }

    /**
     * for simulation only
     * @return cobid1
     */
    public int getCobid1() {
        return cobid1;
    }

    /**
     * for simulation only
     * @return cobid2
     */
    public int getCobid2() {
        return cobid2;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
    }

    /**
     * process PDOData to retrieve data from this device.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        FCSLOG.info(name + " updatingFromPDO = " + pdo);
        boolean updated = false;
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
            updated = true;
            /* update from PDO1 */
            FCSLOG.info(name + " updatingFromPDO1 = " + pdo1 + " binaire:" + Long.toBinaryString(pdo1));
            statusWord = extractStatusWord(pdo1);
            current = (short) extractCurrentAverage(pdo1);
            position = (int) extractPosition(pdo1);
            FCSLOG.info(name + " position = " + position + " pdo1 = " + " binaire:" + Long.toBinaryString(pdo1));
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
            updated = true;
            /* update from PDO2 */
            FCSLOG.info(name + " updatingFromPDO2 = " + pdo2 + " binaire:" + Long.toBinaryString(pdo2));
            followingError = (short) extractFollowingError(pdo2);
            velocity = (int) extractVelocity(pdo2);
        }
        if (updated) {
            this.publishData();
        }
    }

    /**
     * @param pdo
     * @return statusWord from pdo
     */
    public static int extractStatusWord(long pdo) {
        return (int) pdo & 0xFFFF;
    }

    /**
     * @param pdo
     * @return current from pdo
     */
    public static int extractCurrentAverage(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return current from pdo
     */
    public static int extractPosition(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * @param pdo
     * @return following from pdo
     */
    public static int extractFollowingError(long pdo) {
        return (int) (pdo >> 16) & 0xFFFF;
    }

    /**
     * @param pdo
     * @return velocity from pdo
     */
    public static int extractVelocity(long pdo) {
        return (int) (pdo >> 32) & 0xFFFFFFFF;
    }

    /**
     * For tests and debug
     *
     * @return values updated by PDOs
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print values updated by PDOs to debug and test")
    public String printValuesUpdatedByPDOs() {
        StringBuilder sb = new StringBuilder("=> pdo1 value = ");
        sb.append(pdo1);
        sb.append(" / pdo1 = 0x");
        sb.append(Long.toHexString(pdo1));
        sb.append(" / status word = 0x");
        sb.append(Long.toHexString(statusWord));
        sb.append("\n status word = 0b");
        sb.append(Long.toBinaryString(statusWord));
        sb.append("\n current average = ");
        sb.append(current);
        sb.append(" / current average = 0x");
        sb.append(Long.toHexString(current));
        sb.append("\n position = ");
        sb.append(position);
        sb.append(" / position = 0x");
        sb.append(Long.toHexString(position));
        sb.append("\n => pdo2 value = ");
        sb.append(pdo2);
        sb.append(" / pdo2 = 0x");
        sb.append(Long.toHexString(pdo2));
        sb.append("\n followingError = ");
        sb.append(followingError);
        sb.append(" / followingError = 0x");
        sb.append(Long.toHexString(followingError));
        sb.append(followingError);
        sb.append("\n Velocity = ");
        sb.append(velocity);
        sb.append(" / velocity = 0x");
        sb.append(Long.toHexString(velocity));
        return sb.toString();
    }

}
