package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import java.util.List;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the CCS
 * Master Control Module (MCM) in normal operation mode or from a Console
 * (org.lsst.ccs.subsystem.console.TestConsole).
 *
 * <P/>
 * Example of commands that can be sent to the FcsMain : <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>
 * The main goals of the FcsMain is :
 * <UL>
 * <LI>Receive commands from MCM for the whole subsystem.</LI>
 * <LI>Execute commands received from MCM in invoking methods on the Carousel
 * Module on the Autochanger Module and on Loader Module.</LI>
 * <LI>Handle the logic of the whole subsystem.</LI>
 * <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 *
 * @author virieux
 *
 */
public class FcsMain extends MainModule {

    /**
     *
     */
    private static final long serialVersionUID = 7669526660659959402L;

    /**
     * this bridge is to communicate with the canbus1 on the PC104
     */
    private final BridgeToHardware bridgeToLoader;

    @LookupField(strategy = Strategy.CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = Strategy.CHILDREN)
    private Autochanger autochanger;

    @LookupField(strategy = Strategy.CHILDREN)
    private Loader loader;

    @LookupField(strategy = Strategy.CHILDREN)
    private FilterManager filterManager;

    private long setFilterDuration = 0;

    /**
     * Build a FcsMainModule
     *
     * @param bridge
     * @param bridgeToLoader
     */
    public FcsMain(BridgeToHardware bridge, BridgeToHardware bridgeToLoader) {
        super(bridge);
        this.bridgeToLoader = bridgeToLoader;
    }

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make FcsGUI listen to my subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs", "fcs");
    }

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY, FilterReadinessState.READY);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        // because of some sensors still missing, autochanger is never initialized
        // but we need the Filter to be ready for filter exchanges
        // TODO remove after tests
        updateAgentState(FilterState.READY, FilterReadinessState.READY);
    }

    /**
     * Return the name of the filter which is at ONLINE. Return null if no filter is
     * at ONLINE.
     *
     * @return
     */
    public String getOnlineFilterName() {
        if (autochanger.isHoldingFilter()) {
            return autochanger.getFilterOnTrucksName();

        } else {
            return null;
        }
    }

    /**
     * For end user. Return the name of the filter which is at ONLINE. Return null
     * if no filter is at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns the name of the filter which is at ONLINE.")
    public String printFilterONLINEName() {
        if (getOnlineFilterName() != null) {
            return getOnlineFilterName();
        } else {
            return "NONE";
        }
    }

    /**
     * This methods returns the filter which has for name the String given as
     * argument.
     *
     * @param filterName
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "returns the filter which has for name the String given as argument.")
    public Filter getFilterByName(String filterName) {
        return filterManager.getFilterByName(filterName);
    }

    /**
     * Return true if the changer is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    /**
     * Return true if the loader is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return this.bridgeToLoader.isReady();
    }

    /**
     * Return true if the hardware of the changer is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Disconnect the loader hardware.")
    public void disconnectLoader() {
        this.bridgeToLoader.disconnectHardware();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Connect the loader hardware.")
    public void connectLoader() {
        this.bridgeToLoader.connectHardware();
        loader.postStart();
    }

    /**
     * Check if a filter name given by an end user at the console is a valid Filter
     * Name. Delegate to filterManager.
     *
     * @param aName
     */
    public void checkFilterName(String aName) {
        filterManager.checkFilterName(aName);
    }

    /**
     * For Whole FCS GUI. Has to be overriden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }


    public boolean isFilterInCamera (int filterID) {
        return carousel.isFilterOnCarousel(filterID) || autochanger.isFilterOnAC(filterID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Store the filter in the carousel and move the empty autochanger to HANDOFF position.",
            autoAck = false, timeout = 60000)
    public void storeFilterOnCarousel() {
        updateStateWithSensors();
        FCSLOG.info(name + " about to store filter on carousel");
        subs.helper()
                .precondition(FilterReadinessState.READY)
                .precondition(autochanger.isHoldingFilter())
                .precondition(autochanger.isAtHandoff() || autochanger.isAtOnline())
                .precondition(carousel.isReadyToGrabAFilterAtStandby(),
                        " carousel should be at standby with an empty socket")
                .action(() -> {

            updateStateWithSensors();
            // TODO remove this test : the Exception should never be thrown
            if (carousel.isAtStandby() && carousel.isEmptyAtStandby()) {
                FCSLOG.info(name + " carousel is at standby with an empty socket ");
            } else {
                throw new FcsHardwareException(name + " carousel should be at standby with an empty socket");

            }
            // Make sure the AC is free to move
            if (autochanger.isAtOnline()) {
                if (autochanger.getOnlineClamps().isLocked()) {
                    autochanger.getOnlineClamps().unlockClamps();
                }
                // TODO throw an Exception if not Closed ?
                if (autochanger.getOnlineClamps().isClosed()) {
                    if (autochanger.getOnlineClamps().isHomingDone()) {
                        autochanger.getOnlineClamps().openClamps();
                    } else {
                        autochanger.getOnlineClamps().homing();
                    }
                }
            }
            updateStateWithSensors();

            if (autochanger.getOnlineClamps().isOpened()) {
                FCSLOG.info(name + " autochanger is free to move");
            } else {
                throw new FcsHardwareException(name + " autochanger online clamps should be opened");
            }

            FCSLOG.info(name + " carousel is ready to receive a filter at standby");
            autochanger.getAutochangerTrucks().moveFilterToStandby();
            /* because both subsystems sensors have changed we have to update again*/
            updateStateWithSensors();

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info(": carousel is CLAMPED_ON_FILTER");
            } else {
                throw new FcsHardwareException(name
                        + ": carousel should be CLAMPED_ON_FILTER when autochanger is at STANDBY with a filter");
            }
            if (autochanger.isAtStandby()) {
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff();
            } else {
                throw new FcsHardwareException(name + ": autochanger should be at STANDBY after "
                        + "moveFilterToStandby() command");
            }

                });
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            autoAck = false,
            description = "Unclamp filter from carousel, move autochanger to approachStandby position and release the carousel clamps",
            timeout = 15000)
    public void disengageFilterFromCarousel() {
        updateStateWithSensors();
        checkControllers();
        subs.helper()
                .precondition(FilterReadinessState.READY)
                .precondition(carousel.isAtStandby(), "Carousel should be at Standby")
                .precondition(autochanger.isAtStandby(), "AC should be at Standby or Handoff position")
                .action(() -> {

            // Unclamp the filter from the carousel
            carousel.unlockClamps();
            FcsUtils.sleep(100, name);
            carousel.updateStateWithSensors();

            if (!carousel.isUnclampedOnFilterAtStandby()) {
                raiseAlarm(FcsAlert.HARDWARE_ERROR, "Carousel clamps still locked", name);
                throw new FcsHardwareException(name + " aborting autochanger movement because "
                        + "carousel is still holding the filter. bad state for socketAtStandby: "
                        + carousel.getClampsStateAtStandby() + " should be UNCLAMPED_ON_FILTER");
            }
            // Move AC to approach standby position
            autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();

            updateStateWithSensors();
            if (!carousel.isEmptyAtStandby()) {
                raiseAlarm(FcsAlert.HARDWARE_ERROR, "Carousel is still seeing the filter", name);
                throw new FcsHardwareException(name + " aborting autochanger movement because carousel is still seeing filter.");
            }

            // Release the clamps
            carousel.releaseClamps();
            updateStateWithSensors();
        });
    }

    // SET FILTER PRECONDITIONS

    public boolean autochangerNotInTravel() {
        return autochanger.isAtHandoff() || autochanger.isAtOnline() || autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return autochanger.getLatches().isClosed() || autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return autochanger.getOnlineClamps().isLocked() || autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !autochanger.isAtOnline() || autochanger.isEmpty() || autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !autochanger.isAtStandby() || autochanger.isEmpty() || autochanger.isHoldingFilter() || carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return carousel.isHoldingFilterAtStandby() || carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return autochanger.isAtStandby() || autochanger.isEmpty() || carousel.isReadyToGrabAFilterAtStandby();
    }


    /**
     * move filter to online. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on AC at
     * HANDOFF or ONLINE => carousel socket at STANDBY must be READY_TO_CLAMP &&
     * autochanger latches have to be CLOSED, if a filter is on AC at STANDBY =>
     * carousel or autochanger must hold the filter, if a filter is on AC at
     * ONLINE => online clamps must be LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilter(int filterID) {
        updateStateWithSensors();
        /* if a controller is in fault, FCS goes in ALARM state */
        checkControllers();
        FCSLOG.info(name + ": filter to set online:" + filterID);
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM), "can't execute commands in ALARM state.")
                .precondition(!isLoaderConnected(), "loader is connected - can't continue setFilter")
                .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name, filterID)
                .precondition(isFilterInCamera(filterID), "%s: filter %s in out of camera", name, filterID)
                .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                .precondition(autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY")
                .precondition(
                    latchesOpenOrClosed(),
                    "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                    autochanger.getLatches().getLockStatus()
                )
                .precondition(
                    onlineClampsOpenOrLocked(),
                    "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED",
                    autochanger.getOnlineClamps().getLockStatus()
                )
                .precondition(
                    filterAtOnlineMustBeLocked(),
                    "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED",
                    autochanger.getOnlineClamps().getLockStatus()
                )
                .precondition(filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by AC or by carousel.")
                .precondition(
                    carouselHoldingFilterOrReadyToGrab(),
                    "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                    carousel.getClampsStateAtStandby()
                )
                .precondition(
                    carouselReadyToClampAtStandby(),
                    "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP",
                    carousel.getClampsStateAtStandby()
                )
                .precondition(FilterReadinessState.READY)
                .action(() -> {
            final long beginTime = System.currentTimeMillis();
            if (autochanger.isHoldingFilter()) {
                FCSLOG.info(name + ": AC is holding a filter, filter name= " + autochanger.getFilterOnTrucksName());
            } else {
                FCSLOG.info(name + ": AC is not holding filter, filter name on AC" + autochanger.getFilterOnTrucksName());
                FCSLOG.info(name + ": filterID: " + filterID + " is on socket:" + carousel.getFilterSocket(filterID).getId());
            }
            /**
             /* If the filter we want to move ONLINE is NOT on AC :
             /* - if AC is at STANDBY it has to be moved empty to HANDOFF
             /* - if AC is
             * at HANDOFF or ONLINE, and another filter is on AC this filter has
             * to be store on carousel, /* then AC can be moved empty at
             * HANDOFF.
             */
            if (!carousel.isHomingDone() && (autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                /* we have to do homing as soon as possible because we use carousel position to know if a socket is at STANBY.*/
                carousel.homing();
            }
            if (!autochanger.isFilterOnAC(filterID)) {
                FCSLOG.info(name + ": filterID: " + filterID + " is NOT on AC");
                if (autochanger.isAtStandby()) {
                    FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                    autochanger.moveEmptyFromStandbyToHandoff();
                } else {
                    /* autochanger can be at HANDOFF or ONLINE with a filter or empty*/
                    /* if a filter is on AC*/
                    if (autochanger.isHoldingFilter()) {
                        FCSLOG.info(name + ": AC is holding filter: " + autochanger.getFilterOnTrucksName());
                        FCSLOG.info(name + ": is going to store a filter on carousel");
                        storeFilterOnCarousel();
                    }
                }

                /* Now autochanger is empty at Handoff */
                if (!(autochanger.isEmpty() && autochanger.isAtHandoff())) {
                    throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                }
                // Rotate desired filter to standby
                carousel.rotateSocketToStandby(
                        carousel.getFilterSocket(filterID).getName());
//                boolean isAtStandby = false;
//                int timeout = 1000;
//                while (!isAtStandby) {
//                    FcsUtils.sleep(50, name);
//                    carousel.updateStateWithSensors();
//                    isAtStandby = carousel.isAtStandby();
//                }
                FcsUtils.sleep(100, name);
                updateStateWithSensors();
                if (carousel.isAtStandby()) {
                    FCSLOG.info(name + ": carousel is at standby");
                } else {
                    throw new FcsHardwareException(name + ": carousel should be at standby after "
                            + "rotateSocketToStandby command.");
                }

                /* At this point filterID should be on carousel at STANDBY*/
                if (carousel.getFilterIDatStandby() == filterID) {
                    // Go lock filter on AC
                    autochanger.grabFilterAtStandby();

                } else {
                    throw new FcsHardwareException(name + " filterID at standby is "
                            + carousel.getFilterIDatStandby() + " should be: " + filterID);
                }
                updateStateWithSensors();
                if (!autochanger.isFilterOnAC(filterID)) {
                    throw new FcsHardwareException(name + " filter: " + filterID + " should be now on AC");
                }
            }
            /* now filterID is on AC. AC is at STANDBY or at HANDOFF or ONLINE.  */
            if (autochanger.isAtStandby()) {
                // Unclamp filter from carousel and move to approach position
                disengageFilterFromCarousel();
                updateStateWithSensors();
                // Move the filter to online position
                autochanger.moveAndClampFilterOnline();

            } else if (autochanger.isAtHandoff()) {
                autochanger.moveAndClampFilterOnline();

            } else if (autochanger.isAtOnline()) {
                autochanger.lockFilterAtOnline();
            }
            String filterName = filterManager.getFilterNameByID(filterID);
            String letter = filterName.substring(6, filterName.length());
            updateAgentState(
                    FilterState.valueOf("ONLINE_" + letter),
                    FilterReadinessState.READY
            );
            setFilterDuration = System.currentTimeMillis() - beginTime;
            FCSLOG.info("filter " + filterID + " is ONLINE. setFilterDuration = " + setFilterDuration);
            this.publishData();
                }); // END of action
    }


    /**
     * move filter to online. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on AC at
     * HANDOFF or ONLINE => carousel socket at STANDBY must be READY_TO_CLAMP &&
     * autochanger latches have to be CLOSED, if a filter is on AC at STANDBY =>
     * carousel or autochanger must hold the filter, if a filter is on AC at
     * ONLINE => online clamps must be LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Select filter and move to HANDOFF position without going to ONLINE.",
            timeout = 180000, autoAck = false)
    public void setFilterWithSafety(int filterID) {
        updateStateWithSensors();
        /* if a controller is in fault, FCS goes in ALARM state */
        checkControllers();
        FCSLOG.info(name + ": filter to set online:" + filterID);
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM), "can't execute commands in ALARM state.")
                .precondition(!isLoaderConnected(), "loader is connected - can't continue setFilter")
                .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name, filterID)
                .precondition(isFilterInCamera(filterID), "%s: filter %s in out of camera", name, filterID)
                .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                .precondition(autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY")
                .precondition(
                    latchesOpenOrClosed(),
                    "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                    autochanger.getLatches().getLockStatus()
                )
                .precondition(
                    onlineClampsOpenOrLocked(),
                    "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED",
                    autochanger.getOnlineClamps().getLockStatus()
                )
                .precondition(
                    filterAtOnlineMustBeLocked(),
                    "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED",
                    autochanger.getOnlineClamps().getLockStatus()
                )
                .precondition(filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by AC or by carousel.")
                .precondition(
                    carouselHoldingFilterOrReadyToGrab(),
                    "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                    carousel.getClampsStateAtStandby()
                )
                .precondition(
                    carouselReadyToClampAtStandby(),
                    "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP",
                    carousel.getClampsStateAtStandby()
                )
                .precondition(FilterReadinessState.READY)
                .action(() -> {
            final long beginTime = System.currentTimeMillis();
            if (autochanger.isHoldingFilter()) {
                FCSLOG.info(name + ": AC is holding a filter, filter name= " + autochanger.getFilterOnTrucksName());
            } else {
                FCSLOG.info(name + ": AC is not holding filter, filter name on AC" + autochanger.getFilterOnTrucksName());
                FCSLOG.info(name + ": filterID: " + filterID + " is on socket:" + carousel.getFilterSocket(filterID).getId());
            }
            /**
             /* If the filter we want to move ONLINE is NOT on AC :
             /* - if AC is at STANDBY it has to be moved empty to HANDOFF
             /* - if AC is
             * at HANDOFF or ONLINE, and another filter is on AC this filter has
             * to be store on carousel, /* then AC can be moved empty at
             * HANDOFF.
             */
            if (!carousel.isHomingDone() && (autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                /* we have to do homing as soon as possible because we use carousel position to know if a socket is at STANBY.*/
                carousel.homing();
            }
            if (!autochanger.isFilterOnAC(filterID)) {
                FCSLOG.info(name + ": filterID: " + filterID + " is NOT on AC");
                if (autochanger.isAtStandby()) {
                    FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                    autochanger.moveEmptyFromStandbyToHandoff();
                } else {
                    /* autochanger can be at HANDOFF or ONLINE with a filter or empty*/
                    /* if a filter is on AC*/
                    if (autochanger.isHoldingFilter()) {
                        FCSLOG.info(name + ": AC is holding filter: " + autochanger.getFilterOnTrucksName());
                        FCSLOG.info(name + ": is going to store a filter on carousel");
                        storeFilterOnCarousel();
                    }
                }

                /* Now autochanger is empty at Handoff */
                if (!(autochanger.isEmpty() && autochanger.isAtHandoff())) {
                    throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                }
                // Rotate desired filter to standby
                carousel.rotateSocketToStandby(
                        carousel.getFilterSocket(filterID).getName());
//                boolean isAtStandby = false;
//                int timeout = 1000;
//                while (!isAtStandby) {
//                    FcsUtils.sleep(50, name);
//                    carousel.updateStateWithSensors();
//                    isAtStandby = carousel.isAtStandby();
//                }
                FcsUtils.sleep(100, name);
                updateStateWithSensors();
                if (carousel.isAtStandby()) {
                    FCSLOG.info(name + ": carousel is at standby");
                } else {
                    throw new FcsHardwareException(name + ": carousel should be at standby after "
                            + "rotateSocketToStandby command.");
                }

                /* At this point filterID should be on carousel at STANDBY*/
                if (carousel.getFilterIDatStandby() == filterID) {
                    // Go lock filter on AC
                    autochanger.grabFilterAtStandby();

                } else {
                    throw new FcsHardwareException(name + " filterID at standby is "
                            + carousel.getFilterIDatStandby() + " should be: " + filterID);
                }
                updateStateWithSensors();
                if (!autochanger.isFilterOnAC(filterID)) {
                    throw new FcsHardwareException(name + " filter: " + filterID + " should be now on AC");
                }
            }
            /* now filterID is on AC. AC is at STANDBY or at HANDOFF or ONLINE.  */
            if (autochanger.isAtStandby()) {
                // Unclamp filter from carousel and move to approach position
                disengageFilterFromCarousel();
                updateStateWithSensors();
                // Move the filter to online position
                autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();

            } else if (autochanger.isAtOnline() && autochanger.getOnlineClamps().isOpened()) {
                autochanger.getAutochangerTrucks().moveToApproachOnlinePositionWithLowVelocity();
                autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
            }
            String filterName = filterManager.getFilterNameByID(filterID);
            String letter = filterName.substring(6, filterName.length());
            updateAgentState(
                    FilterState.valueOf("ONLINE_" + letter),
                    FilterReadinessState.READY
            );
            setFilterDuration = System.currentTimeMillis() - beginTime;
            FCSLOG.info("filter " + filterID + " is ONLINE. setFilterDuration = " + setFilterDuration);
            this.publishData();
                }); // END of action
    }

    /**
     * just for mcm testing
     *
     * @param state
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "change subsystem state.")
    public void changeState(String state) {
        updateAgentState(
            FilterState.valueOf(state),
            FilterReadinessState.READY
        );
        FCSLOG.fine("SUBSYSTEM STATE=" + isInState(FilterState.valueOf(state)));
    }


    /**
     * Load a filter from the loader to the camera. At the end of this operation the
     * filter is grabbed on the autochanger at Handoff position.
     *
     * @param filterID
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void loadFilter(int filterID) {
        // TODO check the conditions for this action
        // the camera is in the position "horizontal"
        // the auto changer trucks are empty at "Hand-Off" position
        // the carousel socket at "Stand-by" position is empty
        // The Loader carrier is retracted at "Storage" position and the filter is
        // clamped

        if (!carousel.isReadyToGrabAFilterAtStandby()) {
            throw new RejectedCommandException(
                    name + " can't load a filter when socket at STANDBY position is not empty and ready.");
        }

        if (!(autochanger.getAutochangerTrucks().isAtHandoff() && autochanger.isEmpty())) {
            throw new RejectedCommandException(
                    name + " autochanger is not empty at STANDOFF position; can't load a filter.");
        }

        if (!(loader.isHoldingFilter() && loader.getCarrier().isAtStorage())) {
            throw new RejectedCommandException(
                    name + " can't load filter because loader is not holding a filter at storage position.");
        }

        loader.getCarrier().goToHandOff();
        autochanger.getAutochangerTrucks().goToHandOff();

        autochanger.grabFilterAtHandoff(filterID);
        loader.openClampAndMoveEmptyToStorage();

    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Carousel at STANDBY. The trucks
     * of the autochanger have to be empty at STANDOFF position.
     *
     * @param filterID
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    public void unloadFilter(int filterID) {
        // TODO check the conditions for this action
        // Conditions in which this operation is authorized :
        // the camera is in the position "horizontal"
        // the auto changer trucks are empty at the "Hand-Off" position
        // the carousel socket at standby position has the filter to extract
        // The Loader carrier is empty, in storage position, the loader clamps are
        // unclamped
        if (carousel.isFilterOnCarousel(filterID)) {
            throw new RejectedCommandException(
                    name + " : filter " + filterID + " is not on the carousel, can't unload it.");
        }
        if (!carousel.isAtStandby(filterID)) {
            throw new RejectedCommandException(
                    name + " : filter " + filterID + " is not on the carousel at standby position, can't unload it.");
        }
        if (!(autochanger.getAutochangerTrucks().isAtHandoff() && autochanger.isEmpty())) {
            throw new RejectedCommandException(
                    name + " autochanger is not empty at STANDOFF position; can't unload a filter.");
        }

        FCSLOG.info("=======> Autochanger trucks about to move empty to standby position.");
        autochanger.getAutochangerTrucks().goToStandby();
        autochanger.getLatches().close();
            // carousel clamps are opened and go in the READYTOCLAMP state
        // when autochanger moves up the filter
        FCSLOG.info("=======> Carousel is ungrabbing filter at standby position.");
        carousel.unlockClamps();
        //autochanger trucks hold the filter and move it to Online Position
        FCSLOG.info("=======> Autochanger trucks about to move loaded with the filter to handoff position.");
        autochanger.moveFilterToHandoff(filterID);
        FCSLOG.info("=======> Carousel about to release Clamps.");
        carousel.releaseClamps();

        loader.getCarrier().goToHandOff();

        loader.getClamp().close();

        autochanger.ungrabFilterAtHandoff(filterID);

        loader.getClamp().clamp();

        loader.getCarrier().goToStorage();

        this.disconnectLoader();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "check if controllers are in fault")
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer :
     * carousel, autochanger and loader. Read also AC trucks and ONLINE clamps
     * position.
     *
     * @throws FcsHardwareException
     */
    @Override
    public void updateStateWithSensors() {
        // Carousel
        carousel.updateStateWithSensors();
        if (carousel.isAtStandby()) {
            carousel.getSocketAtStandby().updateFilterID();
        }
        // Autochanger
        autochanger.updateStateWithSensors();
        if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.isHoldingFilter() && autochanger.getOnlineClamps().isLocked()) {
            String filterName = filterManager.getFilterNameByID(autochanger.getFilterID());
            String shortFilterName = filterName.substring(6, filterName.length());
            updateAgentState(FilterState.valueOf("ONLINE_" + shortFilterName), getFilterReadinessState());
        } else {
            updateAgentState(FilterState.valueOf("ONLINE_NONE"), getFilterReadinessState());
        }
        // Loader
        if (loader.isConnectedOnCamera()) {
            loader.updateStateWithSensors();
        }
    }

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
        autochanger.initializeHardware();
        loader.initializeHardware();
        postStart();
    }

    @Override
    public void publishData() {
        super.publishData();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData("setFilterDuration", setFilterDuration));
    }

    @Override
    public void postStart() {
    }
}
