
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * This class is the Main Module when carousel is in standalone mode.
 *
 * CarouselModle implements HardwareController because we must initialize
 * acPLutoGateway during initialization Phase.
 *
 * @author virieux
 */
public class CarouselMain extends MainModule {

    @LookupField(strategy = Strategy.CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = Strategy.CHILDREN)
    private FilterHolder autochanger;

    /**
     * Build a CarouselMainModule.
     *
     * @param bridge
     */
    public CarouselMain(BridgeToHardware bridge) {
        super(bridge);
    }

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make CarouselGUI listen to my
         * subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.carousel", "carousel");
    }

    /**
     * Update state in reading sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update state in reading sensors.", timeout = 2000)
    @Override
    public void updateStateWithSensors() {
        this.carousel.updateStateWithSensors();
        this.autochanger.updateStateWithSensors();
    }

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
    }

    /**
     * Check that Carousel hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            subs.getAgentService(AgentStateService.class).updateAgentState(FcsEnumerations.FilterState.READY, FcsEnumerations.FilterReadinessState.READY);
        }
    }

}
