
package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.drivers.EmergencyMessageListener;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is an interface for a piece of hardware on a can open network.
 *
 * @author virieux
 */
public interface PieceOfHardware extends EmergencyMessageListener, AlertRaiser {

    @Override
    String getName();

    String getSerialNB();

    int getNodeID();

    /**
     * @return true If this piece of hardware is booted and the nodeID on the
     *         hardware is the same than the one in the Configuration DB.
     */
    boolean isBooted();

    /**
     * set boolean isBooted in INITIALIZATION phase.
     *
     * @param isBooted
     */
    void setBooted(boolean isBooted);

    /**
     * @return true if the hardware has been initialized with success, false
     *         otherwise.
     */
    boolean isInitialized();

    /**
     * This methods write some configuration data on the hardware CPU.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void initializeAndCheckHardware();

    default void checkBooted() {
        if (!isBooted()) {
            throw new FcsHardwareException(getName() + ": not booted");
        }
    }

    default void checkInitialized() {
        if (!isInitialized()) {
            throw new FcsHardwareException(getName() + ": not initialized");
        }
    }

    /**
     * raise an ALARM if this device is not booted.
     */
    default void raiseAlarmIfMissing() {
        if (!isBooted()) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_MISSING, getName() + " device not booted", getName());
        }
    }

    default boolean checkId(String id) {
        return id.equals(getSerialNB());
    }

    /**
     * Initialize canOpenInterface in order to be able to retrieve PDOs if this
     * PieceOfHardware sends PDOS.
     *
     * @throws DriverException
     */
    void doInitializePDOs() throws DriverException;

    void updateFromPDO(PDOData pdo);

    /**
     * Initialize PDOs if device is booted. Does nothing if device is not booted.
     *
     * @throws DriverException
     */
    default void initializePDOs() throws DriverException {
        if (isBooted()) {
            doInitializePDOs();
        }
    }

    /**
     * Creates an object to be published on the STATUS bus by a PieceOfHardware.
     *
     * @return
     */
    default StatusDataPublishedByHardware createStatusDataPublishedByHardware() {
        return new StatusDataPublishedByHardware(isBooted(), isInitialized());
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    default void publishData() {
        StatusDataPublishedByHardware status = createStatusDataPublishedByHardware();
        FCSLOG.info(getName() + " is publishing:" + status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), status));
    }

    default String printState() {
        StringBuilder sb = new StringBuilder("name=" + getName());
        sb.append(",nodeID=0x");
        sb.append(Integer.toHexString(getNodeID()));
        sb.append(",serial number=");
        sb.append(getSerialNB());
        if (isBooted()) {
            sb.append(" is BOOTED");
        } else {
            sb.append(" is NOT BOOTED");
        }
        if (isInitialized()) {
            sb.append(" and INITIALIZED");
        } else {
            sb.append(" and NOT INITIALIZED");
        }
        return sb.toString();
    }

}
