package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_TRUCKS_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;

import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.*;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for autochanger trucks. Autochanger trucks move a filter from
 * STANDBY position to ONLINE position or HANDOFF position. ONLINE position is
 * just in front of the focal front. STANDBY position is where an exchange can
 * be done with the carousel. HANDOFF position is a position an exchange can be
 * done with the loader. The trucks are moved by 2 controllers, a master and a
 * slave. Each controller has an absolute encoder.
 *
 * @author virieux
 */
public class AutochangerTwoTrucks extends MobileItem implements ClearAlertHandler {

    @LookupField(strategy = Strategy.ANCESTORS)
    private Subsystem subs;

    private AutochangerTruck truckXminus;
    private AutochangerTruck truckXplus;

    @ConfigurationParameter(description = "minimal position for AC trucks")
    private final int minActualPositionValue = 0;

    @ConfigurationParameter(description = "maximal position for AC trucks")
    private final int maxActualPositionValue = 1200000;

    @ConfigurationParameter(description = "STANDBY position in micron")
    private int standbyPosition = 997500;

    @ConfigurationParameter(description = "HANDOFF position in micron")
    private int handoffPosition = 70000;

    @ConfigurationParameter(description = "ONLINE position in micron")
    private int onlinePosition = 2600;

    @ConfigurationParameter(description = "time for trucks motion from ONLINE to STANDBY in milliseconds")
    private long timeoutForTrucksMotion = 20000;

    /**
     * ******************
     * parameters to move filter to online position and adjust position (docking
     * process)
     *******
     */
    @ConfigurationParameter(description = "prelimary ONLINE position in micron, used by process which"
            + "moves a filter at ONLINE position.")
    private int prelimaryTargetPosition = 3367;

    @ConfigurationParameter(description = "natural position at ONLINE position in micron  used by process which"
            + "moves a filter at ONLINE position.")
    private int naturalPosition = 3215;

    @ConfigurationParameter(description = "lowest position to be used by command moveAndClampFilterOnline.")
    private int minTargetPosition = 1800;

    @ConfigurationParameter(description = "minimum value of voltage read on proximitySensor.", units = "Volt")
    private double umin = 4.2;

    @ConfigurationParameter(description = "maximum value of voltage read on proximitySensor. Unit: Volts")
    private double umax = 4.6;

    @ConfigurationParameter(description = "voltage target on proximitySensor. Unit: Volts")
    private double utarget = 3.85;

    @ConfigurationParameter(description = "adjustment factor. Unit: micrometre/volt")
    private double adjustmentFactor = 835;

    /**
     * ***********
     * end of parameters to move filter to online position and adjust position
     * (docking process)
     *******
     */
    /**
     * ***********
     * configurable parameters for the approach to standby position
     ************************
     */
    @ConfigurationParameter(description = "low speed to approach STANDBY position")
    private int lowSpeed = 50;

    @ConfigurationParameter(description = "high speed to approach STANDBY position")
    private int highSpeed = 260;

    @ConfigurationParameter(description = "low acceleration to approach STANDBY position")
    private int lowAcceleration = 150;

    @ConfigurationParameter(description = "high acceleration to approach STANDBY position")
    private int highAcceleration = 310;

    @ConfigurationParameter(description = "low deceleration to approach STANDBY position")
    private int lowDeceleration = 50;

    @ConfigurationParameter(description = "high deceleration to approach STANDBY position")
    private int highDeceleration = 310;

    @ConfigurationParameter(description = "approach STANDBY position")
    private int approachStandbyPosition = 950000;

    @ConfigurationParameter(description = "approach ONLINE position")
    private int approachOnlinePosition = 10000;

    /**
     * ***********
     * end of configurable parameters for the approach to standby position
     ************************
     */
    private int current = 0;
    private int speed = 0;
    private long profileVelocity;
    private int masterPosition;
    private int slavePosition;
    private int deltaPosition;

    /**
     * masterPosition is reached if |targetPosition - masterPosition| <
     * positionRange
     */
    private int positionRange = 100;

    /**
     * position is reached at standby if |targetPosition - masterPosition| <
     * positionRangeAtStandby
     */
    @ConfigurationParameter(description = "position is reached at STANDBY if |targetPosition - masterPosition| < positionRangeAtStandby")
    private int positionRangeAtStandby = 100;

    /**
     * masterPosition is reached if |targetPosition - masterPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached at ONLINE if |targetPosition - masterPosition| < positionRangeAtOnline")
    private int positionRangeAtOnline = 50;

    /**
     * masterPosition is reached if |targetPosition - masterPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached at HANDOFF if |targetPosition - masterPosition| < positionRangeAtOnline")
    private int positionRangeAtHandoff = 100;

    /**
     * masterPosition is reached if |targetPosition - masterPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "position is reached in TRAVEL if |targetPosition - masterPosition| < positionRangeAtOnline")
    private int positionRangeInTravel = 150;

    /**
     * an alignment of slave must be done when |deltaPosition| >
     * deltaPositionMax
     */
    @ConfigurationParameter(description = "an alignment of slave must be done when |deltaPosition| > deltaPositionMax")
    private int deltaPositionMax = 100;

    private int absoluteTargetPosition = 0;
    private int slaveTargetPosition = 0;

    private EPOSControllerForLinearRail linearRailMasterController;
    private EPOSControllerForLinearRail linearRailSlaveController;

    @LookupField(strategy = Strategy.TREE)
    private MainModule main;

    @LookupField(strategy = Strategy.TREE)
    private Autochanger autochanger;

    @LookupField(strategy = Strategy.TREE, pathFilter = ".*proximitySensorsDevice")
    private ADCInterface proximitySensorsDevice;

    @ConfigurationParameter(description = "input number on the ADC where proximity sensor is plugged", range = "1..4")
    private int proximitySensorInput = 1;

    /**
     * true if Handoff position sensors for TruckXminus and TruckXplus return
     * value 1*
     */
    private boolean atHandoff = false;

    /**
     * true if Online position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atOnline = false;

    /**
     * true if Standby position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atStandby = false;

    /**
     * Build a new AutoChangerTrucksModule.
     *
     * @param truckXminus
     * @param truckXplus
     * @param linearRailMasterController
     * @param linearRailSlaveController
     */
    public AutochangerTwoTrucks(AutochangerTruck truckXminus, AutochangerTruck truckXplus,
            EPOSControllerForLinearRail linearRailMasterController,
            EPOSControllerForLinearRail linearRailSlaveController) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.linearRailMasterController = linearRailMasterController;
        this.linearRailSlaveController = linearRailSlaveController;
    }

    @ConfigurationParameterChanger
    public void setNaturalPosition(int pos) {
        int minLimit = onlinePosition - 1500;
        int maxLimit = onlinePosition + 1500;
        if (pos < minLimit || pos > maxLimit) {
            throw new IllegalArgumentException(
                    pos + " bad value for natural position" + " must be > " + minLimit + " and < " + maxLimit);
        }
        this.naturalPosition = pos;
    }

    /**
     * Return actual trucks masterPosition. Doesn't read again the
     * masterPosition on the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return actual trucks position (master position). Doesn't read again the position on the controller.")
    public int getPosition() {
        return masterPosition;
    }

    /**
     * Return HANDOFF masterPosition. Doesn't read again the masterPosition on
     * the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * Return STANDBY masterPosition. Doesn't read again the masterPosition on
     * the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return ONLINE position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return onlinePosition;
    }

    public int getApproachStandbyPosition() {
        return approachStandbyPosition;
    }

    /**
     * Return the timeout for trucks motion. If trucks motion duration is
     * greater than timeoutForTrucksMotion, an ALERT is raised.
     *
     * @return timeoutForTrucksMotion
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Return timeout for trucks motion.")
    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoff() {
        return atHandoff;
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atOnline is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return atOnline;
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it
     * returns its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandby() {
        return atStandby;
    }

    /**
     * Return true if the homing of the trucks has been done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return linearRailSlaveController.isHomingDone() && linearRailMasterController.isHomingDone();
    }

    /**
     * Return true if TruckXminus position sensors or TruckXminus position
     * sensors are in error. Or position sensors are non consistant.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if TruckXminus position sensors or TruckXminus position sensors are in error.")
    public boolean isPositionSensorsInError() {
        return truckXminus.isPositionSensorsInError() || truckXplus.isPositionSensorsInError()
                || positionSensorsNonConsistant();
    }

    private boolean positionSensorsNonConsistant() {
        return ((isAtStandby() && isAtHandoff())
                || ((isAtStandby() && isAtOnline()) || ((isAtHandoff() && isAtOnline()))));
    }

    /**
     * At the end of a motion during tests, a bad positionning of the trucks in
     * a way that both sensors send the same value can be detected as an error.
     * In this case we don't raise an ALARM but a WARNING.
     *
     * @return true if position sensors error is transient
     *
     */
    public boolean isPositionSensorErrorsTransient() {
        int delta = 5000;
        boolean errorHandoff = truckXminus.isHandoffSensorsInError() || truckXplus.isHandoffSensorsInError();
        boolean errorOnline = truckXminus.isOnlineSensorsInError() || truckXplus.isOnlineSensorsInError();
        boolean errorStandby = truckXminus.isStandbySensorsInError() || truckXplus.isStandbySensorsInError();
        return ((errorHandoff && Math.abs(masterPosition - handoffPosition) < delta)
                || (errorOnline && Math.abs(masterPosition - onlinePosition) < delta)
                || (errorStandby && Math.abs(masterPosition - standbyPosition) < delta));
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");

        if (linearRailSlaveController.isBooted() && linearRailMasterController.isBooted()) {
            initializeControllers();
        }
        if (proximitySensorsDevice.isBooted()) {
            proximitySensorsDevice.initializeAndCheckHardware();
            proximitySensorsDevice.publishData();
        } else {
            this.raiseAlarm(HARDWARE_ERROR, " can't clamp at ONLINE if proximitySensorsDevice not booted", name);
        }
        FCSLOG.fine(name + " END postStart.");
    }

    public void initializeControllers() {
        try {
            linearRailSlaveController.initializeAndCheckHardware();
            linearRailMasterController.initializeAndCheckHardware();
            // configureControllers();
            if (autochanger.isLinearRailMotionAllowed()) {
                homing();
                FCSLOG.info(name + " homing of AC trucks done.");
            }

        } catch (FailedCommandException | FcsHardwareException ex) {
            // TODO voir quel controleur n'a pas pu etre initialisé.
            this.raiseAlarm(HARDWARE_ERROR, "Couldn't initialize controllers or do homing", name, ex);
        }
    }

    /**
     * Configure Autochanger trucks master and slave controllers.
     *
     * @throws FcsHardwareException
     */
    @Command(
            type = Command.CommandType.ACTION,
            level = Command.ENGINEERING1,
            description = "Configure Autochanger trucks master and slave controllers.",
            alias = "initControllers")
    public void configureControllers() {
        FCSLOG.info(name + " Begin configuration of the controllers");
        linearRailMasterController.configureDigitalInputOfLinearRails();
        linearRailSlaveController.configureDigitalInputOfLinearRails();
        linearRailMasterController.configureDigitalOutputOfLinearRails();
        linearRailSlaveController.configureDigitalOutputOfLinearRails();
        FCSLOG.info(name + " End configuration of the controllers");
    }

    /**
     * Return true if both controllers are initialized and homing of trucks are
     * done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return linearRailMasterController.isInitialized() && linearRailSlaveController.isInitialized()
                && this.isHomingDone();
    }

    /**
     * This method sends Exceptions if the conditions for trucks motion are not
     * filled. Autochanger trucks motion is allowed if : - controllers are
     * initialized - homing is done - trucks are empty OR (loaded with a filter
     * but latches are closed AND carousel or loader doesn't hold filter)
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion() {
        autochanger.checkFilterSafetyBeforeMotion();
        checkPrerequisitesForTrucksMotion();
    }

    private void checkPrerequisitesForTrucksMotion() {
        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = name + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        updatePosition();
        checkDeltaPosition();
    }

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks
     * are already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Handoff position.", alias = "goToHandoff")
    public void goToHandOff() {
        moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        moveToAbsoluteTargetPosition(this.onlinePosition);
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks
     * are already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        moveToAbsoluteTargetPosition(this.standbyPosition);
    }

    /**
     * Move the trucks to an absolute position given as argument. Doesn't align
     * slave at the end of motion. At the end of
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move Autochanger trucks to the absolute position given as argument. "
            + "At the end of motion : doesn't align slave then activate brakes and disable controllers.", alias = "mobeABSPos", timeout = 20000)
    public void moveToAbsoluteTargetPosition(int targetPosition) {

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(
                    name + "argument has to be between " + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.updatePosition();
        if (masterPosition == targetPosition) {
            FCSLOG.info(name + " is already at target position=" + targetPosition);
        } else {

            checkConditionsForTrucksMotion();
            checkControllersMode();
            checkReadyForAction();
            absoluteTargetPosition = targetPosition;

            FCSLOG.info(name + " going to absolute position: " + absoluteTargetPosition);
            if (absoluteTargetPosition == handoffPosition) {
                main.updateAgentState(FilterState.MOVING_TRUCKS_TO_HANDOFF, FilterReadinessState.NOT_READY);
                positionRange = positionRangeAtHandoff;

            } else if (absoluteTargetPosition == standbyPosition) {
                agentStateService.updateAgentState(FilterState.MOVING_TRUCKS_TO_STANDBY, FilterReadinessState.NOT_READY);
                positionRange = positionRangeAtStandby;

            } else if (absoluteTargetPosition == onlinePosition) {
                agentStateService.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE, FilterReadinessState.NOT_READY);
                positionRange = positionRangeAtOnline;

            } else if (absoluteTargetPosition < this.prelimaryTargetPosition) {
                agentStateService.updateAgentState(FilterState.MOVING_TRUCKS, FilterReadinessState.NOT_READY);
                positionRange = positionRangeAtOnline;

            } else {
                if (masterPosition > absoluteTargetPosition) {
                    agentStateService.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE, FilterReadinessState.NOT_READY);
                } else {
                    agentStateService.updateAgentState(FilterState.MOVING_TRUCKS_TO_STANDBY, FilterReadinessState.NOT_READY);
                }
                positionRange = positionRangeInTravel;
            }
            goToPosition(absoluteTargetPosition);
            checkPositionSensors();
            agentStateService.updateAgentState(FilterState.TRUCKS_STOPPED, FilterReadinessState.READY);
            publishData();
        }
    }

    /**
     * Move the trucks to an absolute position given as argument. Doesn't align
     * slave at the end of motion. Just for tests on prototype. Doesn't check
     * condition for trucks motion.
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Move Autochanger trucks to the absolute position given as argument. "
            + "At the end of motion : doesn't align slave then activate brakes and disable controllers"
            + " VERY DANGEROUS COMMAND. Only for tests with prototype.", timeout = 20000)
    public void moveToAbsoluteTargetPositionDangerous(int targetPosition) {

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(
                    name + " argument has to be between " + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.updatePosition();
        if (masterPosition == targetPosition) {
            FCSLOG.info(name + " is already at target position=" + targetPosition);
        } else {

            checkPrerequisitesForTrucksMotion();
            checkControllersMode();
            checkReadyForAction();
            absoluteTargetPosition = targetPosition;

            FCSLOG.info(name + " going to absolute position: " + absoluteTargetPosition);
            if (absoluteTargetPosition == handoffPosition) {
                agentStateService.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_HANDOFF,
                        FcsEnumerations.FilterReadinessState.NOT_READY);
            } else if (absoluteTargetPosition == standbyPosition) {
                agentStateService.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY,
                        FcsEnumerations.FilterReadinessState.NOT_READY);
            } else if (absoluteTargetPosition == onlinePosition) {
                agentStateService.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_ONLINE,
                        FcsEnumerations.FilterReadinessState.NOT_READY);
            } else {
                agentStateService.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS,
                        FcsEnumerations.FilterReadinessState.NOT_READY);
            }
            goToPosition(absoluteTargetPosition);
            checkPositionSensors();
            agentStateService.updateAgentState(FcsEnumerations.FilterState.TRUCKS_STOPPED,
                    FcsEnumerations.FilterReadinessState.READY);
            publishData();
        }
    }

    @Command(type = Command.CommandType.ACTION,
                level = Command.ENGINEERING2,
                description = "goToOnline with filter then adjust position then close and finally lock clamps",
                autoAck = false,
                timeout = 20000)
    public void moveAndClampFilterOnline() {
        long beginTime = System.currentTimeMillis();

        moveToApproachOnlinePositionWithHighVelocity();
        alignSlave();
        /* move to prelimaryTargetPosition */
        slowProfile();
        goToPosition(prelimaryTargetPosition);
        /* adjust position */
        double umeasured = readProximitySensorDevice();
        int newTargetPosition = (int) (prelimaryTargetPosition + adjustmentFactor * (this.utarget - umeasured));
        FCSLOG.info(name + " new computed target position=" + newTargetPosition);
        newTargetPosition = Math.max(minTargetPosition, newTargetPosition);
        FCSLOG.info(name + " new target position=" + newTargetPosition);
        goToPosition(newTargetPosition);

        if (!autochanger.getOnlineClamps().isHomingDone()) {
            autochanger.getOnlineClamps().homing();
        }

        autochanger.updateStateWithSensors();

        /* close and lock clamps */
        FCSLOG.info(name + " closing ONLINE clamps");
        autochanger.getOnlineClamps().closeClamps();
        FCSLOG.info(name + " locking ONLINE clamps");
        autochanger.getOnlineClamps().lockClamps();
        umeasured = readProximitySensorDevice();
        if (umeasured > umax) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too high.");
        } else if (umeasured < umin) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too low.");
        }
        goToPosition(naturalPosition);
        FCSLOG.info(name + " END process docking ONLINE.");

        autochanger.updateFCSStateToReady();

        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.fine(name + " moveAndClampFilterOnline duration = " + duration);

        this.publishData();
    }


    /**
     * Move a filter to standby position step by step. A first step with a high
     * speed to the approachStandbyPosition. a second step to standby position
     * with low speed.
     */
    @Command(type = Command.CommandType.ACTION,
             level = Command.ENGINEERING2,
            description = "move filter to approachPosition with high speed and move to STANDBY with lowSpeed.",
            autoAck = false,
            timeout = 30000)
    public void moveFilterToStandby() {
        updatePosition();
        autochanger.updateStateWithSensors();
        subs.helper()
            .precondition(FilterReadinessState.READY)
            .precondition(atHandoff || atOnline, "AC should be at Online or Handoff position")
            .action(() -> {
                agentStateService.updateAgentState(
                    FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY,
                    FcsEnumerations.FilterReadinessState.NOT_READY
                );
                if (atOnline) {
                    moveToApproachOnlinePositionWithLowVelocity();
                }
                alignSlave();
                moveToApproachStandbyPositionWithHighVelocity();
                moveToStandbyWithLowVelocity();

                publishData();
            });
}

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToApproachStandbyPositionWithLowVelocity() {
        slowProfile();
        moveToAbsoluteTargetPosition(approachStandbyPosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToApproachOnlinePositionWithLowVelocity() {
        slowProfile();
        moveToAbsoluteTargetPosition(approachOnlinePosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void alignSlaveAndMoveEmptyFromApproachToHandoff() {
        alignSlave();
        fastProfile();
        moveToAbsoluteTargetPosition(handoffPosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToApproachStandbyPositionWithHighVelocity() {
        fastProfile();
        moveToAbsoluteTargetPosition(approachStandbyPosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToApproachOnlinePositionWithHighVelocity() {
        fastProfile();
        moveToAbsoluteTargetPosition(approachOnlinePosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToStandbyWithLowVelocity() {
        slowProfile();
        moveToAbsoluteTargetPosition(standbyPosition);
    }

    @Command(type = Command.CommandType.ACTION)
    // TODO remove after tests
    public void moveToHandoffWithHighVelocity() {
        fastProfile();
        moveToAbsoluteTargetPosition(handoffPosition);
    }




    /**
     * do go to position given as argument without any control. no control are
     * done user must check that this moving is possible.
     *
     * @param targetPosition
     */
    private void goToPosition(int targetPosition) {
        FCSLOG.info(name + " moving to position " + targetPosition);
        if (!isHomingDone()) {
            homing();
        }
        this.absoluteTargetPosition = targetPosition;
        /* see startAction */
        this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
        agentStateService.updateAgentState(FcsEnumerations.FilterState.TRUCKS_STOPPED,
                FcsEnumerations.FilterReadinessState.READY);
    }

    private void goToPositionNoBreak(int targetPosition) {
        FCSLOG.info(name + " moving to position " + targetPosition);
        this.absoluteTargetPosition = targetPosition;
        /* see startAction */
        this.executeAction(MOVE_TO_ABSOLUTE_POSITION_NO_BREAK, timeoutForTrucksMotion);
        agentStateService.updateAgentState(FcsEnumerations.FilterState.TRUCKS_STOPPED,
                FcsEnumerations.FilterReadinessState.READY);
    }

//    @Command(type = Command.CommandType.ACTION,
//            level = Command.ENGINEERING3,
//            description = "move a filter from standby to online with no break. Very Dangerous.",
//            timeout = 20000)
    /**
     * very dangerous : in case of an Exception : no break is activated.
     *
     */
    public void moveAndClampFilterOnlineNoBreak() {
        long beginTime = System.currentTimeMillis();
        FCSLOG.info(name + " BEGIN moving filter to ONLINE ");
        checkConditionsForTrucksMotion();
        checkControllersMode();
        checkReadyForAction();
        if (!isHomingDone()) {
            homing();
        }
        agentStateService.updateAgentState(FilterState.MOVING_TRUCKS_TO_ONLINE, FilterReadinessState.NOT_READY);
        positionRange = positionRangeInTravel;
        try {
            fastProfile();
            goToPosition(approachOnlinePosition);
            /* to let time to the controllers to be disabled */
            FcsUtils.sleep(100, name);
            alignSlave();
            /* move to prelimaryTargetPosition */
            slowProfile();
            goToPositionNoBreak(prelimaryTargetPosition);
            /* adjust position */
            double umeasured = readProximitySensorDevice();
            int newTargetPosition = (int) (prelimaryTargetPosition + adjustmentFactor * (this.utarget - umeasured));
            FCSLOG.info(name + " new computed target position=" + newTargetPosition);
            newTargetPosition = Math.max(minTargetPosition, newTargetPosition);
            FCSLOG.info(name + " new target position=" + newTargetPosition);
            goToPositionNoBreak(newTargetPosition);
            autochanger.updateStateWithSensors();

            /* close and lock clamps */
            FCSLOG.info(name + " closing ONLINE clamps");
            autochanger.getOnlineClamps().closeClamps();
            FCSLOG.info(name + " locking ONLINE clamps");
            autochanger.getOnlineClamps().lockClamps();
            umeasured = readProximitySensorDevice();
            if (umeasured > umax) {
                this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too high.");
            } else if (umeasured < umin) {
                this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too low.");
            }
            /* go to natural position */
            goToPosition(naturalPosition);

        } catch (Exception ex) {
            this.raiseAlarm(HARDWARE_ERROR, " ERROR during moveAndClampFilterOnlineNoBreak", name, ex);
            throw new FcsHardwareException(name + " ERROR during moveAndClampFilterOnlineNoBreak", ex);

        } finally {
            FCSLOG.info(name + " finally in moveAndClampFilterOnlineNoBreak");
            activateBrakesAndDisable();
        }

        FCSLOG.info(name + " END process docking ONLINE.");
        autochanger.updateFCSStateToReady();

        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.fine(name + " moveFilterFromStandbyToOnline duration = " + duration);

        this.publishData();


    }



    /**
     * activate both brakes then disableOperation on the 2 controllers. For end
     * user.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Activate brakes and disable the 2 controllers.")
    public void activateBrakesAndDisable() {
        this.linearRailMasterController.activateBrake();
        this.linearRailSlaveController.activateBrake();
        linearRailMasterController.disableOperation();
        linearRailSlaveController.disableOperation();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        autochanger.checkLinearRailMotionAllowed();
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
            case MOVE_TO_ABSOLUTE_POSITION_NO_BREAK:
                checkControllersMode();
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(absoluteTargetPosition);
                linearRailMasterController.writeControlWord(0x3F);
                autochanger.increaseLinearRailsCurrentMonitoring();
                break;

            case ALIGN_SLAVE:
                linearRailSlaveController.enableAndReleaseBrake();
                linearRailSlaveController.writeTargetPosition(slaveTargetPosition);
                linearRailSlaveController.writeControlWord(0x3F);
                break;

            case MOVE_SLAVE_TRUCK_ALONE:
                linearRailSlaveController.enableAndReleaseBrake();
                linearRailSlaveController.writeTargetPosition(absoluteTargetPosition);
                linearRailSlaveController.writeControlWord(0x3F);
                break;

            default:
                assert false : action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.info(name + " ABORTING action: " + action.toString() + " within delay " + delay);
        if (action == MOVE_TO_ABSOLUTE_POSITION || action == MOVE_TO_STANDBY_POSITION_WITH_FCS) {
            quickStopAction(action, 0);
            activateBrakesAndDisable();
            autochanger.decreaseLinearRailsCurrentMonitoring();

        } else if (action == ALIGN_SLAVE) {
            stopAlignSlave();

        } else {
            activateBrakesAndDisable();
        }

    }

    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.info(name + " ENDING action: " + action.toString());
        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            activateBrakesAndDisable();
            autochanger.decreaseLinearRailsCurrentMonitoring();

        } else if (action == MOVE_TO_ABSOLUTE_POSITION_NO_BREAK) {
            FCSLOG.info(name + " nothing to do for action MOVE_TO_ABSOLUTE_POSITION_NO_BREAK");

        } else if (action == ALIGN_SLAVE) {
            stopAlignSlave();

        } else {
            linearRailMasterController.disableOperation();
            linearRailSlaveController.disableOperation();
        }
        linearRailMasterController.checkFault();
        linearRailSlaveController.checkFault();
    }

    @Override
    // tested on CPPM testbench in september 2015
        public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.linearRailMasterController.quickStop();
        this.linearRailSlaveController.quickStop();
    }

    /**
     * This method is used in the startAction method. It is was has to be done
     * before moving the trucks.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Enable and release brake for the 2 controllers.")
    public void enableControllersAndReleaseBrake() {
        linearRailMasterController.enable();
        linearRailSlaveController.enable();
        linearRailMasterController.doReleaseBrake();
        if (linearRailMasterController.isEnabled()) {
            linearRailSlaveController.doReleaseBrake();
        } else {
            throw new FcsHardwareException(
                    name + " can't release Slave Controller brake " + " because Master Controller is disabled.");
        }
    }

    private void checkControllersMode() {
        if (!linearRailMasterController.isInMode(PROFILE_POSITION)) {
            throw new FcsHardwareException(
                    name + " : linearRailMasterController is not" + " in PROFILE_POSITION mode, can't move.");
        }

        if (!linearRailSlaveController.isInMode(MASTER_ENCODER)) {
            throw new FcsHardwareException(name + " : linearRailSlaveController is not"
                    + " in MASTER_ENCODER mode, can't move. Slave controller mode ="
                    + linearRailSlaveController.getMode());
        }
    }

    /**
     * move slave truk to masterPosition. When moving along the rails with slave
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignement. This consists in : setting slaveController mode to
     * PROFILE_POSITION, go to absolute masterPosition pos, and then set
     * slaveController mode back to MASTER_ENCODER mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Align slave controller position to master controller position.", timeout = 20000)
    public void alignSlave() {
        // master controller must be disabled and brake activated
        if (linearRailMasterController.isEnabled()) {
            throw new FcsHardwareException(name + " master controller must be disabled.");
        }
        updatePosition();
        if (!isHomingDone()) {
            homing();
        }
        if (Math.abs(deltaPosition) > deltaPositionMax) {
            linearRailSlaveController.changeMode(PROFILE_POSITION);
            doAlignSlave(masterPosition);
        }
    }

    private void stopAlignSlave() {
        FCSLOG.info(name + " STOP ALIGN SLAVE");
        linearRailSlaveController.activateBrake();
        /* mandatory : we must let time to the controller to activate the brake */
        /*
         * otherwise if we disable voltage before brake are holding, a filter could
         * fall.
         */
        FcsUtils.sleep(100, name);
        linearRailSlaveController.changeMode(MASTER_ENCODER);
        linearRailSlaveController.disableOperation();
        FCSLOG.info(name + " END STOP ALIGN SLAVE");
    }

    private void doAlignSlave(int pos) {
        FCSLOG.info(name + " slave's alignment to be done. deltaPosition= " + deltaPosition);
        if (!linearRailSlaveController.isInMode(PROFILE_POSITION)) {
            throw new FcsHardwareException(
                    name + " can't align slave truck because" + " couldn't change its mode to PROFILE_POSITION.");
        }
        slaveTargetPosition = pos;
        /* see startAction and postAction */
        this.executeAction(ALIGN_SLAVE, timeoutForTrucksMotion);
    }

    /**
     * move truckXminus (slave controller) independently of truckXplus (master
     * controller)
     *
     * @param newPos
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Move truckXminus (slave controller) "
            + "independently of truckXplus (master controller). To be used with final products AC1 and AC2."
            + "Don't use with prototype.")
    public void moveTruckXminus(int newPos) {
        int slavePos = linearRailSlaveController.readPosition();
        if (Math.abs(slavePos - newPos) >= 2000) {
            throw new IllegalArgumentException(name + " newPos must be such as |newPos - slavePosition| < 2000");
        }
        // just to be safe.
        this.activateBrakesAndDisable();
        linearRailSlaveController.changeMode(PROFILE_POSITION);
        absoluteTargetPosition = newPos;
        this.executeAction(MOVE_SLAVE_TRUCK_ALONE, timeoutForTrucksMotion);

    }

    /**
     * This method is called at the end of actions to check that after a motion
     * to well known position as STANDBY, HANDOFF and ONLINE, position sensors
     * confirm the trucks position. Otherwise it throws a
     * FailedCommandException.
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void checkPositionSensors() {
        /*
         * Read position sensors and check that values return are consistant with trucks
         * position otherwise throw an Exception.
         */
        autochanger.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(
                    name + ": check with sensors: STANDBY sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition=" + this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition=" + this.handoffPosition);
            FCSLOG.info("this.atHandoff=" + this.atHandoff);
            throw new FailedCommandException(
                    name + ": check with sensors: HANDOFF sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(
                    name + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    /**
     * Do homing of both controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Do homing for both  controllers.")
    public void homing() {
        linearRailMasterController.homing();
        linearRailSlaveController.homing();
        updatePosition();
    }

    /**
     * Updates the field position of the trucks in reading the CPU of the
     * controller.
     *
     * *********************************************************************
     * MASTER CONTROLLER FOR truckXminus SLAVE CONTROLLER FOR truckXplus
     * *********************************************************************
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update trucks position in reading controller.")
    public void updatePosition() {
        this.masterPosition = linearRailMasterController.readPosition();
        this.slavePosition = linearRailSlaveController.readPosition();
        this.deltaPosition = this.masterPosition - this.slavePosition;
        this.truckXminus.updatePosition();
        this.truckXplus.updatePosition();
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read position on controllers then compute difference and throw exception"
            + "if difference is > 1000.")
    public void checkDeltaPosition() {
        updatePosition();
        if (Math.abs(deltaPosition) > 1000) {
            String msg = "Too much difference between trucks position. " + "Alignment of Slave Truck to be done.";
            throw new FcsHardwareException(name + ":" + msg);
        }
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks masterPosition
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update trucks position in reading controller and return position.")
    public int readPosition() {
        updatePosition();
        return masterPosition;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return a voltage in mV which represents a distance between between filter"
            + "and filter beam.")
    public double readProximitySensorDevice() {
        return proximitySensorsDevice.readVoltage(this.proximitySensorInput);
    }

    /**
     * @return true if both controllers are initialized and homing of trucks are
     * done.
     */
    @Override
    public boolean myDevicesReady() {
        return this.isInitialized();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
            case MOVE_TO_ABSOLUTE_POSITION_NO_BREAK:
                FCSLOG.finest(name + " linearRailMasterController.isTargetReached()="
                        + linearRailMasterController.isTargetReached() + " /masterPosition=" + masterPosition
                        + " /absoluteTargetPosition=" + absoluteTargetPosition);
                return Math.abs(this.masterPosition - this.absoluteTargetPosition) < positionRange;

            case ALIGN_SLAVE:
                return linearRailSlaveController.isTargetReached() && Math.abs(this.deltaPosition) < deltaPositionMax;

            case MOVE_SLAVE_TRUCK_ALONE:
                return linearRailMasterController.isTargetReached()
                        && Math.abs(this.masterPosition - this.absoluteTargetPosition) < positionRange;

            default:
                assert false : action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            linearRailMasterController.checkFault();
            linearRailSlaveController.checkFault();
            updatePosition();
            FCSLOG.finest(name + " position=" + this.masterPosition);
            autochanger.updateStateWithSensors();
        } catch (SDORequestException ex) {
            raiseWarning(SDO_ERROR, "=> ERROR IN READING CONTROLLERS:", name, ex);
        }
    }

    synchronized void updateState() {
        this.atHandoff = truckXminus.isAtHandoff() && truckXplus.isAtHandoff();
        this.atOnline = truckXminus.isAtOnline() && truckXplus.isAtOnline();
        this.atStandby = truckXminus.isAtStandby() && truckXplus.isAtStandby();
        updatePosition();
    }

    /**
     * change speed & acceleration profile to slow
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileVelocity parameter to lowSpeed and slowProfileAcceleration and slowProfileDeceleration")
    public void slowProfile() {
        slowProfileVelocity();
        slowProfileAcceleration();
        slowProfileDeceleration();

        checkProfileVelocity();
        checkProfileAcceleration();
        checkProfileDeceleration();
    }

    /**
     * change speed and acceleration profile to fast
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileVelocity and ProfileAcceleration and ProfileDeceleration parameters to highSpeed")
    public void fastProfile() {
        raiseProfileVelocity();
        raiseProfileAcceleration();
        raiseProfileDeceleration();

        checkProfileVelocity();
        checkProfileAcceleration();
        checkProfileDeceleration();
    }

    /**
     * change ProfileVelocity parameter to lowSpeed
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileVelocity parameter to lowSpeed and slowProfileAcceleration and slowProfileDeceleration")
    public void slowProfileVelocity() {
        this.profileVelocity = lowSpeed;
        linearRailMasterController.changeProfileVelocity(lowSpeed);
        linearRailSlaveController.changeProfileVelocity(lowSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + lowSpeed);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileVelocity and ProfileAcceleration and ProfileDeceleration parameters to highSpeed")
    public void raiseProfileVelocity() {
        this.profileVelocity = highSpeed;
        linearRailMasterController.changeProfileVelocity(highSpeed);
        linearRailSlaveController.changeProfileVelocity(highSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + highSpeed);
    }

    /**
     * check that ProfileVelocity has same value in master and in slave.
     */
    private void checkProfileVelocity() {
        int slaveV = linearRailSlaveController.readVelocity();
        int masterV = linearRailMasterController.readVelocity();
        if (slaveV != masterV) {
            throw new FcsHardwareException(name + " master and slave ProfileVelocity are not the same.");
        }
    }

    /**
     * check that ProfileAcceleration has same value in master and in slave.
     */
    private void checkProfileAcceleration() {
        long slaveV = linearRailSlaveController.readParameter(ProfileAcceleration);
        long masterV = linearRailMasterController.readParameter(ProfileAcceleration);
        if (slaveV != masterV) {
            throw new FcsHardwareException(name + " master and slave ProfileAcceleration are not the same.");
        }
    }

    /**
     * check that ProfileDeceleration has same value in master and in slave.
     */
    private void checkProfileDeceleration() {
        long slaveV = linearRailSlaveController.readParameter(ProfileDeceleration);
        long masterV = linearRailMasterController.readParameter(ProfileDeceleration);
        if (slaveV != masterV) {
            throw new FcsHardwareException(name + " master and slave ProfileDeceleration are not the same.");
        }
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileAcceleration parameter to highAcceleration")
    public void raiseProfileAcceleration() {
        linearRailMasterController.writeParameter(ProfileAcceleration, highAcceleration);
        linearRailSlaveController.writeParameter(ProfileAcceleration, highAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + highAcceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileAcceleration parameter to lowAcceleration")
    public void slowProfileAcceleration() {
        linearRailMasterController.writeParameter(ProfileAcceleration, lowAcceleration);
        linearRailSlaveController.writeParameter(ProfileAcceleration, lowAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + lowAcceleration);
    }

    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileAcceleration parameter to highAcceleration")
    public void raiseProfileDeceleration() {
        linearRailMasterController.writeParameter(ProfileDeceleration, highDeceleration);
        linearRailSlaveController.writeParameter(ProfileDeceleration, highDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + highDeceleration);
    }

    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
             description = "change ProfileAcceleration parameter to lowAcceleration")
    public void slowProfileDeceleration() {
        linearRailMasterController.writeParameter(ProfileDeceleration, lowDeceleration);
        linearRailSlaveController.writeParameter(ProfileDeceleration, lowDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + lowDeceleration);
    }

    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setMasterPosition(masterPosition);
        s.setSlavePosition(slavePosition);
        s.setCurrent(current);
        s.setSpeed(speed);
        s.setProfileVelocity(profileVelocity);
        s.setHomingDone(isHomingDone());
        s.setAtHandoff(atHandoff);
        s.setAtOnline(atOnline);
        s.setAtStandby(atStandby);
        s.setInError(isPositionSensorsInError());
        return s;
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByAutoChangerTrucks()));
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (linearRailMasterController.isBooted() && linearRailSlaveController.isBooted()) {
            this.activateBrakesAndDisable();
        }
        FCSLOG.info(name + " is shutdown.");
    }

}
